package Data::Tuple::Space;

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

use Carp;
use Data::Dumper;

@EXPORT = qw(
	
);
$VERSION = '0.01';

sub DEBUG () { 0 }

my($PARENS, $QUOTES);

#############################################################################
sub new
{
    my($package)=@_;
    return bless {ID=>0, members=>{}, byID=>{}}, $package;
}

########################################
sub DESTROY
{
    my($self)=@_;
    delete $self->{byID};
    foreach my $k (keys %{$self->{members}}) {
#        warn $k;
        delete $self->{members}{$k};
    }
    delete $self->{members};
}

########################################
sub add
{
    my($self, $tuple)=@_;
    $tuple=$self->prepare($tuple) unless ref $tuple;

    if(UNIVERSAL::isa($tuple, 'Data::Tuple::Space::Result')) {
        $tuple=$tuple->as_arrayref;
    }

    my $id="TUPLE:".(++$self->{ID});
    my $p=0;
    my $l=@$tuple;
    my $prev;
    if($l==0) {                         # fencepost case
        $self->{byID}{$id}=[0];
        return $id;
    }

    foreach my $member (@$tuple) {
        croak "All tuple members must be defined" unless defined $member;
        croak "Tuple member can't be ?" if $member eq '?';
        if(ref $member) {
            $member=$self->add($member);
        }

        $self->{byID}{$id} ||= [$l, $member];       # aka first

        $self->{members}{$member}[$p][$l-1]||=[];

        push @{$self->{members}{$member}[$p][$l-1]}, [$prev, $id];

        if(defined($prev)) {
            $self->{members}{$prev}[$p-1][$l-1][-1][2]=$member;
        }

        $p++;
        $prev=$member;
    }
    return $id;
}

########################################
sub get
{
    my($self, $id)=@_;
    return unless $id and exists $self->{byID}{$id};
    my($l, $member)=@{$self->{byID}{$id}};
    return [] if $l==0;       # fence post

    my $ret=[];
    my $p=0;
    while(defined $member) {
        if($member =~ /^(TUPLE:\d+)$/) {
            push @$ret, $self->get($member);
        } else {
            push @$ret, $member;
        }

        # $id shouldn't appear twice in {$member}[$p][$l-1]
        my($q)=grep {$_->[1] eq $id} @{$self->{members}{$member}[$p][$l-1]};
        die "$member($p) isn't a member of $id (len=$l)???" unless $q;
        die Dumper $self, $q unless ref $q;
        $member=$q->[2];        # next
        $p++;
    }
    die "$p != $l.  $id isn't right length?" unless $p==$l;

    return $ret;
}


########################################
sub remove
{
    my($self, $id)=@_;
    $id=$self->search($id) unless exists $self->{byID}{$id};
    return unless $id;

    my($l, $member)=@{delete $self->{byID}{$id}};
    my $p=0;
    my($next, $keep);
    while(defined $member) {
        if($member =~ /^(TUPLE:\d+)$/) {
            # TODO : refcount tuples (or something)
            next;
        } 
        next unless $self->{members}{$member}[$p][$l-1];
        undef($next);
        $keep=[];
        foreach my $q (@{$self->{members}{$member}[$p][$l-1]}) {
            if($q->[1] eq $id) {
                $next=$q->[2];        # next
            } else {
                push @$keep, $q;
            }
        }
        if(@$keep) {
            $self->{members}{$member}[$p][$l-1]=$keep;
        } 
        else {
            undef($self->{members}{$member}[$p][$l-1]);
            $self->_compact($member, $p);
        }
        $member=$next;
        $p++;
    }
}

sub _compact
{
    my($self, $member, $p)=@_;
        
    
    if(defined $p) {                    # Compact the tuple-length level
        while(not defined $self->{members}{$member}[$p][-1]) {
            pop @{$self->{members}{$member}[$p]};
            next if @{$self->{members}{$member}[$p]};
            # remove if no longer needed
            undef($self->{members}{$member}[$p]);
            last;
        }
    }

    if($member) {               # compact the position level
        while(not defined $self->{members}{$member}[-1]) {
            pop @{$self->{members}{$member}};
            next if @{$self->{members}{$member}};
            # no longer needed, remove it
            delete($self->{members}{$member});
            last;
        }
    }
}

########################################
sub search
{
    my($self, $search)=@_;
    $search=$self->prepare($search)
            unless ref $search eq 'ARRAY';

    my($one, $two, $missing);
    my $p=0;
    my $l=@$search;
    foreach my $member (@$search) {
        croak "All tuple members must be defined" unless defined $member;
        if($member eq '?') {
            push @$missing, $p++;
            next;
        }
        $one={};
        last unless exists $self->{members}{$member};

        foreach my $q (@{$self->{members}{$member}[$p][$l-1]}) {
            die join ', ', @{$self->{members}{$member}[$p][$l-1]} 
                                unless ref $q;

            my $id=$q->[1];
            $one->{$id}=(!$two || $two->{$id});
                # $two  $two->{$id} | $one->{$id}
                # ------------------+------------
                #   0       0       |    1
                #   0       1       |    1
                #   1       0       |    0
                #   1       1       |    1
        }
        $p++;
        $two=$one;
    }
    
    my @res;
    foreach my $id (keys %$one) {
        push @res, Data::Tuple::Space::Result->new($self, $id, $missing);
    }
    return @res;
}

########################################
sub prepare
{
    my($self, $string)=@_;
    require Text::DelimMatch;
    $PARENS||=Text::DelimMatch->new('\(', '\)', "\\", '', '"');
    $QUOTES||=Text::DelimMatch->new('"', '"', "\\", '');
    return $self->_prepare($string, 0);
}

########################################
sub _prepare
{
    my($self, $string, $n)=@_;

    my ($pre, $match, $post)=$PARENS->match($string);
    my $ret=[];
    if($PARENS->error) {
        die $PARENS->error;
    }
    elsif(not $match) {
        push @$ret, $self->_commas($string, $n);
    } else {
        my @todo;
        DEBUG and print "  pre='$pre', match='$match', post='$post'\n";
        push @$ret, $self->_commas($pre, $n) if $pre =~ /\S/;
        substr($match, 0, 1)='';        # remove parens
        substr($match, -1)='';
        if($n==0) {
            return $self->_prepare($match, $n+1);
        }
        push @$ret, $self->_prepare($match, $n+1);
        push @$ret, @{$self->_prepare($post, $n)} if $post =~ /\S/;
    }
    return $ret;
}

sub _commas
{
    my($self, $string, $n)=@_;
    my @ret;

    DEBUG and print "Commas in '$string'\n";
    my($pre, $match, $post)=$QUOTES->match($string);
    if($QUOTES->error) {
        die $QUOTES->error;
    } elsif($match) {
        push @ret, $self->_commas($pre, $n) if $pre =~ /\S/;
        $match=~s/^"//;
        $match=~s/"$//;
        DEBUG and print "    value($n): $match\n";
        push @ret, $match;
        $post=~s/^\s*,\s*//;
        push @ret, $self->_commas($post, $n) if $post =~ /\S/;
    } else {
        foreach my $q (split /\s*,\s*/, $string) {
            DEBUG and print "    value($n): $q\n";
            push @ret, $q;
        }
    }
    return @ret;
}

1;

#############################################################################
package Data::Tuple::Space::Result;

########################################
sub new
{
    my($package, $space, $id, $missing)=@_;
    return bless {space=>$space, ID=>$id, missing=>$missing}, $package;
}

########################################
sub missing_member
{
    my($self, $n)=@_;
    $n||=0;
    return unless $self->{missing}[$n];
    $self->{value}||=$self->{space}->get($self->{ID});

    return $self->{value}[$self->{missing}[$n]];
}

########################################
sub as_string
{
    my($self)=@_;
    $self->{value}||=$self->{space}->get($self->{ID});

    return $self->_as_string($self->{value});
}

########################################
sub _as_string
{
    my($self, $ar)=@_;
    my @ret;
    foreach my $v (@$ar) {
        if(ref $v) {
            push @ret, $self->_as_string($v);
        } else {
            push @ret, $v;
        }
    }
    return '('.join(', ', @ret).')';
}

########################################
sub as_arrayref
{
    my($self)=@_;
    return $self->{value}||=$self->{space}->get($self->{ID});
}

########################################
sub as_tuple
{
    my($self)=@_;
    return $self->{ID};
}

__END__



=head1 NAME

Data::Tuple::Space - Implement a tuple-space

=head1 SYNOPSIS

    use Data::Tuple::Space;

    my $space=new Data::Tuple::Space;
    $space->add("(Hugh, father, Philip)");
    $space->add("Hugh, father, Daniel");
    $space->add(['Hugh', 'father', 'Alice']);
    $space->add([qw(Hugh client Philip)]);
    
    foreach my $res ($space->search("(Hugh, father, ?)")) {
        # all of Hugh's children
        print $res->as_string;
    }

    my $relationships=Data::Tuple::Space->prepare("(Hugh, ?, Philip)");
    foreach my $res ($space->search($search)) {
        print "Hugh is Philip's ", $res->missing_member, "\n";
    }

    # multi-layer tuples
    $space->add("(Service, subscribed, (Kernel, Session, Event))");
    # equiv to 
    my $t=$space->add("(Kernel, Session, Event)");
    $space->add("(Service, subscribed, $t");

    # NO YOU CAN'T HAVE RECURSIVE TUPLES!  Thank you.


=head1 DESCRIPTION

Tuples are ordered sets of members.  A tuple member could be another tuple. 
The set of all tuples is called tuple-space.  Two identical tuples can
co-exist within tuple-space.

Examples of tuples:
    
    (hello, world)                  # 2 members
    (world, hello)                  # tuples are ordered, not that same
    (hello, out, there)             # 3 members
    ((hello, world), this is, fine) # NOT the same tuple (hello, world) as
                                    # previously
    ()                              # empty tuple

The only operation you do on a tuple space is search it, using a a tuple.
The C<?> is a special member.  It means "any old member here".  This means
that regular TUPLES CAN'T HAVE C<?> AS A MEMBER.

    (hello, world)                  # returns 2 tuples
    (hello, ?)                      # also returns 2 tuples
    (?,?)                           # all 2-member tuples
    (?, this is, fine)              # returns a tuple!

Data::Tuple::Space is an attempt at encoding a fast search algorythm, though
it wastes space, but not to heavily as long as you don't have really really
long tuples (ie, >10 members).  Tuples are represented as arrayrefs when
possible.  

=head1 METHODS

=over 

=item new

    my $space=new Data::Tuple::Space;

Create a brand-spanking new, clean, unblemished tuple-space.

=item add
    
    my $id1=$space->add("(hello, world)");
    my $id2=$space->add(['hello', 'world']);
    # $id1 and $id2 differ!

Adds a new tuple to the tuple-space.  Returns a unique ID for that tuple
within the tuple-space.  This ID can then be used with C<get()> to get the
tuple's value.  Tuples can be defined as an arrayref, or as a scalar that
gets parsed C<prepare>.

=item get

    my $arrayref=$space->get($id1);

Fetches the tuple identified by C<$id1> from the tuple-space.  Returns the
tuple as an arrayref, or as undef() if the tuple doesn't exist.

=item remove

    $space->remove($id2);

Fetches the tuple identified by C<$id2> from the tuple-space.  The tuple-space 
is compacted after this operation.

=item prepare

    my $arraytuple=$space->prepare($string_tuple);

Converts a string-representation of a tuple into an arrayref representation
using C<Text::DelimMatch>.  Here are some example scalars that could be
converted:

    "(hello, world)"
    "hello, world"      # equiv to previous
    "(a 1 (b c d))"     # multi-level tuple
    "(\"commas, in, member\")"
    "(a\(b)"            # ( in member

=item search

    my @res=$space->search("(hello, ?)");
    my @res=$space->search([hello=>'?']);

Searches the tuple-space for all tuples that match the search critieria. 
Returns an array of results, as described in L<"Search results"> later on.

=back




=head2 Search results

Each element in the array returned by C<search()> is a temporary object. 
The following methods can be used:

=over

=item missing_member

    my $world=$res[0]->missing_member($n);

Returns the value that corresponds to Nth C<?> in the search tuple.  If
C<$n> is omited, defaults to 0 (first).  Returns undef() if C<$n> is greater
then the number of C<?>s in the search tuple.

=item as_string

Returns the tuple as a string.  Useful for debuging.

=item as_arrayref

Returns the tuple as an arrayref.  Sub-tuples are also arrayrefs.

=item as_tuple

Returns the tuple's unique ID in the tuple-space.  Could be used to embed a
tuple like so:

    my $t=$res[0]->as_tuple;
    $space->add("$t, is fine");   # won't create new (hello, world) tuple

=back




=head1 AUTHOR

Philip Gwyn, perl-tuples at pied.nu

=head1 SEE ALSO

perl(1).

=cut
