# $Id: Catcher.pm,v 1.6 2002/06/07 08:22:34 fil Exp $
package JAAS::Catcher;
use strict;

sub DEBUG {0}

#############################################################################
# Sly magic to subvert Carp
BEGIN { 
    unless($INC{'Carp.pm'}) {
        require Carp;         # needed for when carp and croak outside of eval
        $INC{'Carp.pm'} = __FILE__;
    } elsif($INC{'Carp.pm'} ne __FILE__) {
        CORE::warn "You didn't use ", __PACKAGE__, " before Carp ($INC{'Carp.pm'})!\n";
        $INC{'Carp.pm'} = __FILE__;
    }
}

require Exporter;

package JAAS::Catcher::Carp;            
use strict;
# make sure only Carp-like functions are exported
use vars qw(@EXPORT_OK @ISA @EXPORT);
BEGIN {
    @EXPORT=qw(confess croak carp);
    @EXPORT_OK=qw(carp croak confess cluck);
    @ISA=qw(Exporter);
}

sub DEBUG () {0}
sub Carp::import
{
    DEBUG and print STDERR "My import\n";
    __PACKAGE__->export_to_level(1, @_);
}



##########################################################
sub _mk_oneup
{
    my($type, $name, $trace)=@_;
    return sub {
#        DEBUG and print STDERR __PACKAGE__, " $type $name\n";
        if($JAAS::Catcher::BUGGER) {
            DEBUG and print STDERR __PACKAGE__, "::$name\n";
            my $q=$_[0];
            $q={text=>join '', @_} unless ref $q;
            $q->{__level}||=5;
            $q->{stack_trace}=$trace;
            DEBUG and print STDERR __PACKAGE__, " __level is $q->{__level}\n";
            $SIG{"__${type}__"}->($q);
            if($type eq 'DIE') {   # we need to jump out of the eval {}
                local $SIG{__DIE__}='DEFAULT';
                die "\n";
            }
        } else {
            DEBUG and print STDERR "Carp::$name\n";
            local $Carp::CarpLevel=1;
            Carp->can($name)->(@_);
        }
    };
}
BEGIN {
    *carp=_mk_oneup('WARN', 'carp');
    *croak=_mk_oneup('DIE', 'croak');
    *cluck=_mk_oneup('WARN', 'cluck', 1);
    *confess=_mk_oneup('DIE', 'confess', 1);
}


#############################################################################
package JAAS::Catcher;

use JAAS::Error;

use vars qw(@EXPORT_OK @ISA @EXPORT $BUGGER);

@EXPORT=qw(around);
@EXPORT_OK=qw(EXEC_CTX monitor around carp croak confess cluck);
@ISA=qw(Exporter);

my @CAUGHT;
my($DEFAULT_WARN, $DEFAULT_DIE);

$!=0;

##########################################################
sub init
{
    my($package, $error_p)=@_;

    DEBUG and print STDERR __PACKAGE__, " in init\n";
    $DEFAULT_WARN=$SIG{__WARN__};
    $SIG{__WARN__}=$package->mk_handler($error_p, 'warning');
    $DEFAULT_DIE=$SIG{__DIE__};
    $SIG{__DIE__}=$package->mk_handler($error_p, 'fatal');
    $BUGGER=1;
}

##########################################################
sub deinit
{
#    my($package)=@_;

    DEBUG and print STDERR __PACKAGE__, " in deinit\n";
    $SIG{__WARN__}=$DEFAULT_WARN;
    $SIG{__DIE__}=$DEFAULT_DIE;
    $BUGGER=0;
}


##########################################################
sub warn
{
    my($package, $error_p, $msg)=@_;
    if(@_==2) {
        push @CAUGHT, $error_p;
    } else {
        push @CAUGHT, $error_p->new($msg, 'warning', $package->current_context);
    }
}

##########################################################
sub error
{
    my($package, $error_p, $msg)=@_;
    if(@_==2) {
        push @CAUGHT, $error_p;
    } else {
        push @CAUGHT, $error_p->new($msg, 'fatal', $package->current_context);
    }
}

##########################################################
sub mk_handler
{
    my($package, $error_p, $type)=@_;
    return sub {
        DEBUG and print STDERR __PACKAGE__, " in $type handler\n";
        local $SIG{__WARN__}='DEFAULT';     # make sure we don't recurse
        local $SIG{__DIE__}='DEFAULT';
        my $err=$_[0];
        eval {
            push @CAUGHT, $error_p->new($err, 
                                    $type, $package->current_context);
        };
        CORE::warn $@ if $@;
    };
}


##########################################################
sub get_stack
{
    [@CAUGHT];
}
sub clear_stack
{
    undef @CAUGHT;
}
END 
{
    print STDERR __PACKAGE__, " Unanswered for :\n" if @CAUGHT;
    foreach my $s (@CAUGHT) {
        print STDERR __PACKAGE__, " ", 
                $s->{full_text} ? $s->{full_text} : "$s->{text}\n";
    }
}

##########################################################
sub monitor ($$&)
{
    my($package, $error_p, $sub)=@_;
    if(2==@_) {
        $sub=$error_p;
        $error_p='JAAS::Error';
    }
    DEBUG and print STDERR __PACKAGE__, " in monitor\n";
    DEBUG and print STDERR __PACKAGE__, " before eval\n";
    eval {
        local @SIG{qw(__WARN__ __DIE__)};

        $package->init($error_p);
        $sub->();
        $package->deinit();
    };
    DEBUG and print STDERR __PACKAGE__, " after eval\n";
}

##########################################################
# syntatic sugar
sub around (&) {$_[0]}

sub wrap ($&)
{
    shift->monitor(@_);
}

##########################################################
my %CONTEXT;
sub EXEC_CTX
{
    my $holder=$_[0]=JAAS::Catcher::_CtxHolder->new($_[1], \%CONTEXT);
    1;
}
sub current_context {\%CONTEXT}




#############################################################################
# Class for objects that hold bits of execution context and restore them
# later on.
package JAAS::Catcher::_CtxHolder;
use strict;

my $CTX_KEY="__xtc_ _ctx__";
my $DEL_KEY="__eteled_ _delete__";
sub new
{
    my($package, $what, $ctx)=@_;
    my $self=bless {$CTX_KEY=>$ctx, $DEL_KEY=>[]}, $package;
    foreach my $key (keys %$what) {
        if(exists $ctx->{$key}) {
            $self->{$key}=$ctx->{$key};
        } else {
            push @{$self->{$DEL_KEY}}, $key;
        }
        $ctx->{$key}=$what->{$key};
    }
    return $self;
}

sub DESTROY
{
    my($self)=@_;
    my $ctx=delete $self->{$CTX_KEY};
    my $del=delete $self->{$DEL_KEY};
    @{$ctx}{keys %$self}=values %$self;
    delete @{$ctx}{@$del};
}







1;


__END__

# Below is the stub of documentation for your module. You better edit it!

=head1 NAME

JAAS::Catcher - JAAS's glue for catching errors and shite.

=head1 SYNOPSIS

    ############################################
    use JAAS::Catcher;
    wrap JAAS::Catcher 
        with JAAS::Error 
        around {
        # your code here
        };

    my $stack=JAAS::Catcher->get_stack;
    if(@$stack) {               
        JAAS::Catcher->clear_stack;
        # handle errors here
    }


    ############################################
    package MyError;
    sub new
    {
        my($package, $text, $type, $ctx)=@_;
        # fun fun fun!
    }

    package main;
    use JAAS::Catcher qw(EXEC_CTX);


    JAAS::Catcher->init('MyError');

    eval {
        CORE::warn "something";
        if($file) {
            EXEC_CTX(my $t, {file=>$file});
            open FILE, $file or die "Can't open";
            # ....
            close FILE
        }
    };

    JAAS::Catcher->deinit();


    ############################################
    # sly magic
    use JAAS::Catcher;
    use Carp;

    # yow!  We can subvert croak!
    wrap JAAS::Catcher around { croak "foo bar"};
    my $stack=JAAS::Catcher->get_stack;



=head1 DESCRIPTION

JAAS::Catcher is a package that encapsulates exception handling and
reporting.  It's purpose is to allow you to have exact details of what went
wrong where encapsulated in one object (defaults to JAAS::Error). 
Exceptions are accumulated onto a stack that can be accessed and cleared by
the caller.

It works the obvious way, by manipulating C<$SIG{__DIE__}> and
C<$SIG{__WARN__}> and wrapping code calls in C<eval {}>.  

What's more, it includes some sly magic to subvert Carp so that you don't
have to refactor all of CPAN to get those modules to work with this one. 
This works by messing with C<%INC> and C<Carp::import> and requires that
C<JAAS::Catcher> be loaded before C<Carp>.  Also, it will fail if code
references the functions directly (as in 
C<Carp::carp("Some thing is going on")>) which is bad form anyway.


=head1 METHODS

=head2 init

    JAAS::Catcher->init('JAAS::Error');

Sets up C<$SIG{__DIE__}> and C<$SIG{__WARN__}> to do what we want them to
do.  

It's sole parameter is the name of a package that will encapsulate the
error.  Only method called in this package is C<new>, which is described in
C<JAAS::Error>.
=head2 deinit

    JAAS::Catcher->deinit;

Turns of exception catching by us.

=head2 monitor

    JAAS::Catcher->monitor('JAAS::Error', sub { .... });
    JAAS::Catcher->monitor( sub { .... });
    JAAS::Catcher->monitor( \&subroutine );

Causes C<JAAS::Catcher> to monitor the given coderef.  If 2 parameters are
given, first is assumed to be the package that will encapsulate the
exceptions (see C<init>) and second is the coderef.  If only 1 parameter is
given, it is assumed to be a coderef and C<JAAS::Error> is used for
exceptions.

=head2 wrap

    wrap JAAS::Catcher with JAAS::Error around { .... };

This is syntatic sugar for the following :

    JAAS::Catcher->monitor('JAAS::Error', sub { .... });

=head2 get_stack

    my $stack=JAAS::Catcher->get_stack();

Returns an arrayref of all accumulated exceptions.  

=head2 clear_stack

    JAAS::Catcher->clear_stack();

Deletes all accumulated exceptions.  If any exceptions remain on the stack
during the C<END> phase, a message is printed to STDERR about this.

=head2 EXEC_CTX

Sets the execution context for furthur exceptions/messages.  The first param
must be a lexically scoped variable that is used to restore the context when
it goes out of scope.  The second param is a hashref of values that will be
added any exception with the scope.

For example :

    {
        EXEC_CTX(my $c, {file=>$file});
        open FILE, $file or die "Unable to open file";
        # ....
        close FILE;
    }

Note that while the first param variable is lexically scoped,
the context is in fact a dynamic scope.  

    sub marine
    {
        my($file)=@_;
        open FILE, $file or die "Unable to open file";
        # ....
        close FILE;
    }

    {
        my $file=shift @ARGV;
        EXEC_CTX(my $c, {file=>$file});
        marine($file);
    }

Also, you could do some fun things by manipulating the scope of the first
param.

    my $yo;
    sub marine
    {
        my($file)=@_;
        open FILE, $file or die "Unable to open file";
        undef($yo);                 # context is reset
        # ....
        warn "something"            # doesn't have {file=>$file}
        close FILE;
    }
    
    {
        my $file=shift @ARGV;
        EXEC_CTX($yo, {file=>$file});
        marine($file);
    }

=head1 AUTHOR

Philip Gwyn <jaas at awale.qc.ca>, 
Learry Gagn <mou-jaas at awale.qc.ca>.

=head1 SEE ALSO

perl(1), POE, L<JAAS::Error>

=cut
