# $Id: Error.pm,v 1.4 2002/06/07 08:22:34 fil Exp $
package JAAS::Error;

use strict;
use vars qw($VERSION);
use POSIX qw(strftime);

$VERSION = '0.03';

sub DEBUG () { 0 }

##########################################################
sub new
{
    my($package, $text, $type, $ctx)=@_;
    my $self;

    DEBUG and print STDERR "$package->new\n";    
    if(ref $text) {                     # are we already an object?
        DEBUG and print STDERR "$package passed a reference\n";    
        
        $self=$text;                    # yeah, so we don't create a new one
        $@='' if $@ and $@ == $self;    # eval {die {}}; in this case
                            
                                        # don't rebless
        undef($package) if UNIVERSAL::can($self, "can");
    } 
    else {      
        DEBUG and print STDERR "$package passed some text ($text)\n";    
        $self={                         # build a reference
#            full_text=>$text, 
            text=>$text,
        };
        # strip off the stuff perl-added stuff
        $self->{text} =~ s/ at (\S+|\(.*\)) line (\d+).*$//s;
    }

    $self->{type}||=$type;
    $self->{text}||='Something happened';

    ## and now turn into an object... if it isn't already
    bless $self, $package if $package;

    ## add info from the execution context
    if($ctx) {
        foreach (keys %$ctx) {
            $self->{$_}||=$ctx->{$_};
        }
    }
    ## save some info
    if($!) {
        $self->{errno}=0+$!;
        $self->{errno_str}=''.$!;
    }
    if($^E and $^E != $!) {
        $self->{OS_errno}=0+$^E;
        $self->{OS_errno_str}=''.$^E;
    }
    $^E=0;
    $!=0;
    if($@) {
        $self->{error}=$@;
        $@='';
    }
    $self->{__level}||=3;              # used by trace_context
    DEBUG and print STDERR ref($self), "__level is $self->{__level}\n"; 

    $self->{pid}  ||= $$;
    $self->{time} ||= strftime("%Y/%m/%d-%H:%M:%S", localtime);

    $self->trace_context;
    $self->mk_full_text;

    DEBUG and print STDERR ref($self), "->new done\n";    
    return $self;
}

##########################################################
sub yes_full_text {substr($_[0]->{full_text}, -1) ne "\n" }

##########################################################
sub add_locus
{
    my($self)=@_;
    $self->{full_text}.=" at $self->{src_file} line $self->{line}\n";
}

##########################################################
sub add_info
{
    my($self)=@_;
    my $m='';
    $m .= " $self->{file}" if $self->{file};
    $m .= ":" if $self->{file} and $self->{errno_str};
    $m .= " $self->{errno_str}" if $self->{errno_str};
    $self->{full_text} .= $m;
}
##########################################################
sub mk_full_text
{
    my($self)=@_;
    $self->{full_text}||=$self->{text};

    if($self->yes_full_text) {
        $self->add_info;
        $self->add_locus;
    }
}

##########################################################
sub build_trace_back
{
    my($self, $level)=@_;
    my @trace;
    while(@trace=caller(++$level)) {
        push @{$self->{back_trace}}, {
                package=>$trace[0], filename=>$trace[1], line=>$trace[2],
                caller=>$trace[3], has_args=>($trace[4]||0),
                wantarray=>($trace[5]||0),
                eval_text=>$trace[7], is_require=>($trace[8]||0),
            };
    }

}


##########################################################
sub trace_context
{
    my($self)=@_;

    my $level=delete $self->{__level};
    # _dump_stack($level);

    my @trace=caller($level);
    $self->{package}    = $trace[0]||'';
    $self->{src_file} ||= $trace[1]||'';
    $self->{line}     ||= $trace[2]||0;

    if($trace[3]) {
        if($trace[3] eq '(eval)') {
            $self->{in_eval}=$self->{is_eval} = 1
        } elsif(@trace=caller($level+1)) {
            $self->{subroutine}=$trace[3];
        }
    }

    $self->{back_trace}=[];
    $self->build_trace_back($level) if $self->{stack_trace};

    while(not $self->{in_eval} and @trace =  caller(++$level)) {
        next unless $trace[3] eq '(eval)';
        $self->{in_eval} = 1;
    }
    
}

##########################################################
# debuging help
sub _dump_stack
{
    my($level)=@_;
    my $l=-1;
    my @stack;
    while(my @trace=caller(++$l)) {
        unshift @stack, "l=$l package=$trace[0] file=$trace[1] line=$trace[2] sub=$trace[3]";
        $stack[0].=" ****" if $l==$level+1;
    }
    print join "\n", @stack;
    print "\n\n";
}



###############################################################################
# syntatic sugar
sub with {@_}



1;
__END__
# Below is the stub of documentation for your module. You better edit it!

=head1 NAME

JAAS::Error - JAAS exception encapsulation

=head1 SYNOPSIS

    # most of the time, you will get these objects from JAAS::Catcher

    my $stack=JAAS::Catcher->get_stack;
    foreach my $exception (@$stack) {
        # fiddle with the exception
    }

    # however, you might want to sub-class the object

    package MyError;
    use JAAS::Error;
    @ISA=qw(JAAS::Error);

    sub new
    {
        my $package=shift;
        my $self=$package->SUPER::new(@_);
        # ... do your thing here
    }

    sub add_locus               # called during SUPER::new
    {
        my($self)=@_;
        $self->{full_text}="$self->{package}($self->{line}) $self->{full_text}";
    }

=head1 DESCRIPTION

This object encapsulates an exception.  Exceptions can be warnings, errors or
anything else you want.  This implementation doesn't use accessor methods
because that's slow and painful.  See C<MEMBERS> to find out how to get at
the information.


=head1 MEMBERS

=head2 text, full_text

C<text> is the message that was passed to die or warn.  
C<full_text> is C<text> with any extra information added to it to make it
more inteligble to humans.  See C<mk_full_text>.

=head2 type

One of 'warning' or 'error'.

=head2 errno, errno_str

C<$!> as a number or a string.

=head2 OS_errno, OS_errno_str

C<$^E> as a number or a string, if it differs from C<$!>.

=head2 error

Contents of C<$@> at the time of the exception.  Yes, this can differ from
C<text> and C<full_text>.

=head2 package, file, line

Package, file and line of that file where the exception occured.

=head2 subroutine

Name of the in which the exception occured, or undef() if it was an
anonymous eval.

=head2 is_eval, in_eval

C<is_eval> is true if the exception occured within an anonymous eval. 
C<in_eval> is true if at any point in the call stack, an eval was used.  You
could use this to see if you want to throw an exception furthur.

=head2 back_trace

Arrayref containing hashrefs that describe the trace stack.  See
C<build_trace_back> for more details.





=head1 OVERLOADING

Here are the methods you might be interested in overloading

=head2 new

    $exception=$package->new($package, $text, $type, $ctx);

=over 4

=item $package

Name of the package.

=item $text

Text or reference that was passed to C<warn> or C<die>.  Note that versions
of perl previous to 5.005 can't throw references.  L<JAAS::Catcher> tries to
work around this.

=item $type

Type of exception... can be one of 'warning' or 'error'.

=item $ctx

Current execution context.  Normally a hashref containing any information
passed to C<EXEC_CTX> in L<JAAS::Catcher>.

=back


=head2 trace_context

    $exception->trace_context;

Should set data members dealing with the context or locus of the error. 

Default implementation sets members C<package>, C<src_file>, C<line>,
C<subroutine>, C<is_eval>, C<in_eval>.  It also builds C<back_trace> by
calling C<build_trace_back> if C<stack_trace> is set.

=head2 build_trace_back

    $exception->build_trace_back($level);

Fills in the arrayref C<back_trace> of information regarding the call stack. 
It should start and $level+1 and proceed until either you don't care anymore
or caller() returns undef().  

Default implementation is to create a bunch of hashrefs from the values
returned by caller().  Keys are C<package>, C<filename>, C<line>,
C<caller>, C<has_args>, C<wantarray>, C<eval_text>, C<is_require>.  See
perlfunc for an explanation of caller().


=head2 mk_full_text

    $exception->mk_full_text;

Should create a full text message that would be interesting and inteligable
to hioomans.

Default implementation is to set C<full_text> equal to C<text>.  And, if
C<yes_full_text> returns true, it then adds on the line number and source
file to the end, like standard perl does.

=head2 yes_full_text

    if($exception->yes_full_text) {
        ....
    }

Called from mk_full_text, should return true if you want members
C<full_text> and C<text> to be different.

Default implementation does what Perl does: returns true unless C<text> is
terminated by a newline.




=head1 AUTHOR

Philip Gwyn <jaas at awale.qc.ca>, 
Learry Gagn <mou-jaas at awale.qc.ca>.

=head1 SEE ALSO

perl(1), POE, L<POE::Catcher>

=cut

$Log: Error.pm,v $
Revision 1.4  2002/06/07 08:22:34  fil
Bumped version

Revision 1.3  2001/11/14 05:30:43  fil
Added support for all the funky, junky, monkey logging.  Hope I docoed it too.

Revision 1.2  2001/07/27 21:42:23  fil
Added tests for EXEC_CTX, damnit

Revision 1.1.1.1  2001/07/13 06:46:32  fil
Initial import

