# $Id: Log.pm,v 1.2 2002/06/07 08:22:34 fil Exp $
package JAAS::Log;

use strict;
use vars qw($VERSION @ISA);
use JAAS::Object;
use JAAS::Object::Services;
use POE::Session;

@ISA=qw(JAAS::Object);
$VERSION = '0.03';

BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () { 0 }; }
}

##########################################################
sub spawn
{
    my($package, $args, $factory)=@_;
#    $more||=[JAAS::Factory->new()];
    my $delegate=JAAS::Log::Delegate->new($args, $factory);
    $package->SUPER::spawn(args=>[$delegate]);
    return $delegate;
}

##########################################################
sub _start
{
    my($kernel, $session)=@_[KERNEL, SESSION];
    shift(@_)->SUPER::_start(@_);
    DEBUG and print STDERR __PACKAGE__, "_start\n";

    $jaas_relationships->nail($session->ID, 'on_error', '*', 
                                    {to=>{object=>$session->ID,
                                          method=>'checks'
                                         }
                                    });
    $jaas_relationships->nail($session->ID, 'on_warning', '*', 
                                    {to=>{object=>$session->ID,
                                          method=>'checks'
                                         }
                                    });
    $kernel->sig(HUP=>'HUP');
}

##########################################################
sub _stop
{
    my($kernel, $session)=@_[KERNEL, SESSION];
    shift(@_)->SUPER::_stop(@_);
    DEBUG and print STDERR __PACKAGE__, " _stop\n";
    $jaas_relationships->remove($session->ID);

}

##############################################################################
package JAAS::Log::Delegate;

use strict;
use Data::Dumper;
use JAAS::Object::Delegate;
use JAAS::Monitor;
use JAAS::Object::Services;

use vars qw(@ISA);
@ISA=qw(JAAS::Object::Delegate);
BEGIN {*DEBUG=\&JAAS::Log::DEBUG;}

#####################################################
sub new
{
    my($package, $args, $factory)=@_;
    my $self=$package->SUPER::new($args);

    # Create all the destinations (media)
    my $media=delete $self->{media};
    while(my($name, $args)=each %$media) {
        my $class=delete($args->{class}) || delete($args->{package});
        unless($class) {
            warn "media $name requires a class or package\n";
            next;
        }

        # this next line was a startlingly bad idea
        # $args->{name}||=$name if UNIVERSAL::isa($args, 'HASH');

        # Get the factory to make the object
        my $q=$factory->make_object($class, $args);
        unless($q) {
            warn "Unable to create media $name: $class $@\n";
            next;
        }

        $self->{media}{$name}=$q;
    }
        
    # create the monitoring object
    $self->{monitor} = JAAS::Monitor->new();

    # Now create all the triggers in the monitor
    my $triggers=delete $self->{trigger};
    my $count=0;
    foreach my $m (sort keys %$triggers) {
        unless($m) {
            warn "A trigger destination must refer to at least one media\n";
            next;
        }

        my $ts=$triggers->{$m};
        my $ms=[$m];

        $ts=[$ts] unless 'ARRAY' eq ref $ts;
        foreach my $t (@$ts) {
            my $q=delete $t->{media};
            if($q) {
                push @$ms, $q unless ref $q;
                push @$ms, @$q if ref $q;
            }
            my @medias;
            foreach my $m (@$ms) {                  # check that they exist
                unless($self->{media}{$m}) {
                    warn "A log destination refers to undefined media '$m'\n"
                        # don't warn on <trigger name> so that we can have
                        # fake names that modify the order of checking
                        unless $m eq $ms->[0];
                    next;
                }
                push @medias, $m;
            }
            die "No log destinations for trigger $m\n" unless @medias;
            # create a closure for the work we do at triggered time
            $self->{monitor}->add({%$t}, $self->mk_dest(\@medias));
            $count++;
        }
    }
    warn "Note: you don't define any triggers\n" unless $count;
    return $self;
}

#####################################################
sub mk_dest
{
    my($self, $list)=@_;
    return sub {
        DEBUG and print STDERR ref($self), "->callback()\n";
        # this subref is called by JAAS::Monitor when the triggers are met
        foreach my $m (@$list) {
            next unless $self->{media}{$m};
            DEBUG and print STDERR ref($self), ": $m->message\n";
            $self->{media}{$m}->message(@_);
        }
    };
}

#####################################################
sub methods
{
    return (checks=>{args_n=>1}, 
            HUP=>{});
}

##########################################################
# This is a signal... we want the media to close/reopen all their
# files
sub HUP
{
    my($self)=@_;
    foreach my $media (values %{$self->{media}}) {
        next unless $media->can('HUP');
        $media->HUP();
    }
}

##########################################################
# This is posted by JAAS::Catcher.  We delegate the work to our
# JAAS::Monitor
sub checks
{
    my($self, $args)=@_;
    DEBUG and print STDERR ref($self)."->checks\n";
    $self->{monitor}->checks($args);
}


1;

__END__
# Below is the stub of documentation for your module. You better edit it!

=head1 NAME

JAAS::Log - session for destination of log messages

=head1 SYNOPSIS

    use JAAS::Log;
   
    JAAS::Log->spawn({....});

    # Better yet, use a factory :

    <Session monitor>
        class=JAAS::Log
        <args>
            name=Monitor
            <media file1>
                class=JAAS::Log::File
                file=/var/log/file
                syntax=[time] 1- "[text]"
            </media>
            <media STDERR>
                class=JAAS::Log::STDERR
                syntax=[text]
            </media>
            <trigger file1>
                type=1
            </trigger>
            <trigger STDERR>
                type=fatal
            </trigger>
            <trigger file1>
                type=3
                media=STDERR
            </trigger>
        </args>
    </Session>

    # then you send exceptions to it :

    $jaas_services->post('checks@Monitor', {type=>1, text=>"EEP!"});
        # gets sent to /var/log/file
    
    $jaas_services->post('checks@Monitor', {type=>'fatal', text=>"bad vibes"});
        # gets sent to STDERR
    

=head1 DESCRIPTION

JAAS::Log is the session that coordinates exception "routing" between
various sources and sinks.  The source of the exceptions could be
JAAS::Services (via the 'on_error' or 'on_warning' relationship) or
posted explicitly.  JAAS::Log then uses a JAAS::Monitor object to check 
various triggers.  If the triggers match, the exception is sent to one 
or more sinks (aka media) which dispose of the exception as they see fit.

=head1 METHODS

=head2 spawn

    my $thing=JAAS::Log->spawn($params, $factory);

Creates the session and delegate object.  Sets up a relation ship for
'on_error' and 'on_warning' for all objects.  Returns the delegate object.

C<$params> is a complicated hashref :

=over 4

=item media

Hashref that defines the various places exceptions can go to.  The name of
each "sink" is the key.  Value tells must have a C<class> or C<package> key. 
The factory is used to create an object of that class, with the other
key/value pairs as parameters.

See L<JAAS::Log::Sink>, L<JAAS::Log::File> and L<JAAS::Log::File> for
examples of what media do.

=item trigger

Hashref that defines all the triggers that guide an exception to the any of
the sinks.  Each key can be a valid media type.  Each value can either be 
an arrayref of hashrefs, or just an arrayref.  These hashrefs are the
triggers that must be matched by the exception.  This hashref may also
contain the key 'media', which can be a scalar or arrayref.  If it does,
those medias are ALSO called when an exception matches the triggers.

=back

This is all harder to explain then to do.  If you use the config and fatory,
it's pretty DWIMy (I think).

Note that triggers are matched in accending asciibetical order.  This means
that you can influence the order in which media.  Combined with
'monitor_matches', this can be pretty powerful :

    <trigger 01>
        locus=cart
        media=cart-log
    </trigger>
    <trigger 02>
        type=fatal
        locus=dbi
        media=dbi-errors
    </trigger>
    <trigger zz>
        type=fatal
        monitor_matches=0
        media=stderr
    </trigger>

A fatal exception will only get to stderr if no other trigger matched it.
See L<JAAS::Monitor> for details about C<monitor_matches>.

=head2 HUP

    $monitor->HUP();
    $jaas_services->post('HUP@Monitor');

This is used for the HUP signal, so that the media can close/reopen their
files.  Normally you don't call it explicitly.

=head2 checks

    $monitor->checks(\@exceptions);
    $jaas_services->post('HUP@Monitor'=>\@exceptions);

Pass a bunch of exceptions to the monitor so that it matches them all
against the trigger.

=head1 AUTHOR

Philip Gwyn <jaas at awale.qc.ca>, 
Learry Gagn <mou-jaas at awale.qc.ca>.

=head1 SEE ALSO

perl(1), POE, L<JAAS::Catcher>, L<JAAS::Monitor>, L<JAAS::Log::Sink>,
L<JAAS::Log::File> and L<JAAS::Log::File>

=cut

$Log: Log.pm,v $
Revision 1.2  2002/06/07 08:22:34  fil
Bumped version

Revision 1.1  2001/11/14 05:30:43  fil
Added support for all the funky, junky, monkey logging.  Hope I docoed it too.

