# $Id: File.pm,v 1.1 2001/11/14 05:30:43 fil Exp $
package JAAS::Log::File;

use strict;
use vars qw($VERSION @ISA);
use JAAS::Log::Sink;
use POSIX qw(SEEK_END);

$VERSION = '0.01';
@ISA=qw(JAAS::Log::Sink);

BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () { 0 }; }
}

##########################################################
sub new
{
    my $self=shift(@_)->SUPER::new(@_);
    my $file=delete($self->{file}) || 
             delete($self->{filename}) || 
             delete($self->{dest});
    die "Must have a filename for ", ref $self unless $file;
    $self->{file}=$file;
    # open the file right now, so that we die at startup if it's not
    # writable
    $self->open();
    return $self;
}

##########################################################
sub HUP
{
    my($self)=@_;
    $self->close();
    $self->open();
}

##########################################################
sub output
{
    my($self, $msg)=@_;
    $self->open();

    # TODO: grab advisory lock
    $self->{io}->seek(0, SEEK_END);
    $self->{io}->print($msg);
    $self->{io}->print("\n") unless "\n" eq substr($msg, -length("\n"));;
    # TODO: release advisory lock
}

##########################################################
sub open
{
    my($self)=@_;
    return if $self->{io} and $self->{io}->opened;

    $self->{io}=new IO::File ">> $self->{file}";
    die "Unable to open $self->{file}: $!" unless $self->{io};
}

##########################################################
sub close
{
    my($self)=@_;
    delete $self->{io};
}

1;

__END__
# Below is the stub of documentation for your module. You better edit it!

=head1 NAME

JAAS::Log::File - Log a message to a file

=head1 SYNOPSIS

    <media file1>
        class=JAAS::Log::File
        file=/var/log/something
    </media>
   

=head1 DESCRIPTION

This is an "exception sink" that sends the output to a file.

=head1 METHODS

=head2 new

    my $sink=JAAS::Log::File->new($args);

Creates the object and opens the file in append mode to see if it's writable.
Will DIE DIE DIE if it's not.  $args is a hashref, of which we care about
C<filename>, C<file> or C<dest> which tells us where the file is.  See
L<JAAS::Log::Sink> for description of C<syntax> key.

=head2 open

    $sink->open();

Causes the sink to open it's file if it isn't already.

=head2 close

    $sink->close();

Causes the sink to close it's file.

=head2 HUP

    $sink->HUP();

Cases the sink to close it's file and flush buffers, then reopen the file. 
This allows you to move files around then HUP your daemon process.  You
shouldn't call HUP directly, but let C<JAAS::Log> do it instead.

=head2 output

    $sink->output($msg);

Write a formated message to the end of the file.



=head1 AUTHOR

Philip Gwyn <jaas at awale.qc.ca>, 
Learry Gagn <mou-jaas at awale.qc.ca>.

=head1 SEE ALSO

perl(1), POE, L<JAAS::Log::Sink>

=cut

$Log: File.pm,v $
Revision 1.1  2001/11/14 05:30:43  fil
Added support for all the funky, junky, monkey logging.  Hope I docoed it too.

