# $Id: Mail.pm,v 1.1 2001/11/14 05:30:43 fil Exp $
package JAAS::Log::Mail;

use strict;
use vars qw($VERSION @ISA $id);

$VERSION = '0.01';
@ISA=qw(JAAS::Object);

BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () { 0 }; }
}
$id=0;

##########################################################
sub spawn
{
    my($package, $args)=@_;
    $args->{name}=$package."-".$id++;
    my $delegate=JAAS::Log::Mail::Delegate->new($args);
    $package->SUPER::spawn(object=>$delegate);
    return $delegate;
}







#############################################################################
package JAAS::Log::Mail::Delegate;
use strict;
use JAAS::Log::Sink;
use JAAS::Object::Services;
use vars qw(@ISA);

@ISA=qw(JAAS::Log::Sink JAAS::Object::Delegate);


BEGIN {*DEBUG=\&JAAS::Log::Mail::DEBUG;}

##########################################################
sub new
{
    my $self=shift(@_)->SUPER::new(@_);
    die "Must specify a mailer session" unless $self->{mailer};
    die "Must specify who to mail to" unless $self->{to};
    foreach my $field (qw(to)) {
        $self->{$field}=[split ';', $self->{$field}] 
                        unless ref $self->{$field};
    }
    $self->{text}=[];

    # TODO: parse things like 5kb, 15m
    $self->{'max-time'}||=$self->{'min-time'} * 5 if $self->{'min-time'};

    return $self;
}

##########################################################
sub methods
{
    return (
        flush=>{n_args=>0},
    );
}

##########################################################
sub HUP
{
    my($self)=@_;
    $self->flush() if @{$self->{text}};
}

##########################################################
sub output
{
    my($self, $msg)=@_;
    
    chomp $msg;
    push @{$self->{text}}, $msg;

    unless($self->{'min-time'}) {
        $self->send();
        return;
    }
    if($self->{'max-size'} and
            length(join "\n", @{$self->{text}}) >= $self->{'max-size'}) {
        DEBUG and print STDERR "$self: It's big enough, flushing\n";
        $self->flush();
        return;
    }

    # clear any previous waiting
    $jaas_services->alarm_remove($self->{min})     if $self->{min};
    # wait for min-time seconds
    $self->{min}=$jaas_services->delay_set({to=>{method=>'flush', 
                                                 object=>$self->name}}, 
                                            $self->{'min-time'});
    # wait for a MAX of max-time seconds
    $self->{max} ||= $jaas_services->delay_set({to=>{method=>'flush', 
                                                     object=>$self->name}},
                                                $self->{'max-time'});
    return;
}

##########################################################
sub flush
{
    my($self)=@_;
    # send() is called on HUP, when max-size is reached or when min-time or 
    # max-time timeouts.  This means we MUST send it NOW.  No, not soon.
    $jaas_services->alarm_remove(delete $self->{min})      if $self->{min};
    $jaas_services->alarm_remove(delete $self->{max})      if $self->{max};

    unless(@{$self->{text}}) {
        DEBUG and print STDERR "$self: Nothing to be sent!\n";
        return;
    }

    DEBUG and print STDERR "$self: Flush\n";
    $self->send();
}

##########################################################
sub send
{
    my($self)=@_;

    my $text=join "\n", @{$self->{text}};
    my $args={  to=>$self->{to}, 
                subject=>$self->{subject},
                text=>$text, 
            };
    $args->{'reply-to'}=$self->{'reply-to'}         if $self->{'reply-to'};

    
    $jaas_services->post({to=>{object=>$self->{mailer},
                               method=>($self->{pgp} ? 'pgp5mail' : 'mail')}},
                         $args);
    $self->{text}=[];
    return;
}

1;

__END__
# Below is the stub of documentation for your module. You better edit it!

=head1 NAME

JAAS::Log::Mail - Send a log message to an e-mail address

=head1 SYNOPSIS

    <media mailit>
        syntax=[type]\n[text]
        class=JAAS::Log::Mail
        to=my@something.com
        to=other@something.com
        subject=Something fatal happened

        # Session or object that should do the mailing
        mailer=Mail
        # wait for 10 seconds for more messages before sending.  
        min-time=10
        # wait for max of 5 minutes before sending
        max-time=300
        # send mail right away if size > 5k
        max-size=5120
   </media>

    <trigger mailit>
        type=fatal
    </trigger>
   

=head1 DESCRIPTION

This is an "exception sink" that sends the output in an e-mail.

=head1 METHODS

=head2 new

    my $sink=JAAS::Log::Mail->new($args);

Creates the object.  $args is a hashref with a few values :

=over 4

=item to

List of addresses that the mail is sent to.  Can be an arrayref, or a semicolon
';' seperated list.  First one is the To: fields, others are Cc: fields

=item subject

Text of the Subject: header.  Used as is.  No substitution is done (for
now).

=item mailer

Name of the object/session that will do the actual mailing.  It will have the
method 'mail' called.

=item pgp

If set, then the pgp5mail@mailer is called to send the e-mail.  

=item syntax

See L<JAAS::Log::Sink> for more details.

=item min-time

When a new message is to be sent, it is kept for this number of seconds
before being sent.  This allows multiple messages to be batched together. 
Defaults to 0 seconds.

=item max-time

This is the maximum amount of time to hold on to a message.  This means that
even if a stream of messages are being batched, max-time seconds after the
first one was received, the batch will be sent as it is, and a new batch
sent.  Defaults to 5 times min-time.

=item max-size

Forces the message to be sent if the batch grows to be bigger then this in
bytes.  If set to 0, a batch can grow indefinately (or until max-time is
reached).  Defaults to 0.

=back

=head2 output

    $sink->output($msg);

Takes a prepared message and batches it.  If no min-time is specified,
message is sent right away.

=head2 HUP

    $sink->HUP();

Causes the sink to send any pending messages "RIGHT FUCKING NOW" as we say
in Qubec.

=head2 send

    $sink->send();

When this method is called, the batch of messages is sent.  It might be fun
and original to opverload this method.


=head1 AUTHOR

Philip Gwyn <jaas at awale.qc.ca>, 
Learry Gagn <mou-jaas at awale.qc.ca>.

=head1 SEE ALSO

perl(1), POE, L<JAAS::Log::Sink>

=cut

$Log: Mail.pm,v $
Revision 1.1  2001/11/14 05:30:43  fil
Added support for all the funky, junky, monkey logging.  Hope I docoed it too.

