# $Id: Sink.pm,v 1.1 2001/11/14 05:30:43 fil Exp $
package JAAS::Log::Sink;

use strict;
use vars qw($VERSION);
use POSIX qw(strftime);
use Data::Dumper;

$VERSION = '0.01';

BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () { 0 }; }
}

##########################################################
sub new
{
    my($package, $args)=@_;

    my $self=bless {%$args}, $package;
    $self->{syntax} ||= "[time] - [text]";
    $self->{syntax} =~ s{\\n}{\n}g;
    $self->{syntax} =~ s{\\t}{\t}g;
    return $self;
}

##########################################################
sub HUP
{
    return;
}

##########################################################
sub message
{
    my($self, $excp)=@_;
    DEBUG and print STDERR "$self: message\n";

    my $msg=$self->flatten($excp, '');
    $msg->{time} ||= strftime("%Y/%m/%d-%H:%M:%S", localtime);
    $msg->{dump} ||= Data::Dumper->Dump([$excp], [qw(exception)]);

    my $syntax=$self->{syntax};
    my $match=join '|', map quotemeta, keys %$msg;
    # use Data::Dumper;
    # print STDERR "match=$match\n", Dumper $msg;
    $syntax =~ s{\[($match)\]} 
                {$msg->{$1}}g;
    
    $self->output($syntax);
}

##########################################################
sub output
{
    my($self, $msg)=@_;
    print STDERR "Please overload ", ref($self), "->output!\n$msg\n";
}

##########################################################
sub flatten
{
    my($self, $msg, $prefix)=@_;
    my $out={};
    return {text=>$msg} unless ref $msg;
    if(UNIVERSAL::isa($msg, 'HASH')) {
        while(my($k, $v)=each %$msg) {
            unless(ref $v) {
                $out->{$prefix.$k}=$v;
                next;
            }
            my $t=$self->flatten($v, "$prefix$k.");
            @{$out}{keys %$t}=values %$t;
        }
    } 
    elsif(UNIVERSAL::isa($msg, 'ARRAY')) {
        my $k=0;
        foreach my $v (@$msg) {
            unless(ref $v) {
                $out->{$prefix}=$v if $k==0;
                $out->{$prefix.$k}=$v;
            } else {
                my $t=$self->flatten($v, "$prefix$k.");
                @{$out}{keys %$t}=values %$t;
            }
            $k++;
        }
    }
    else {
        $out->{$prefix||'text'}="$msg";
    }
    return $out;
}

1;

__END__
# Below is the stub of documentation for your module. You better edit it!

=head1 NAME

JAAS::Log::Sink - base class for a destination of log messages

=head1 SYNOPSIS

    package My::Sink;

    use JAAS::Log::Sink;
    @ISA=qw(JAAS::Log::Sink);   

    sub output
    {
        my($self, $msg)=@_;
        chomp($msg);
        # Do something with $msg here
    }

=head1 DESCRIPTION

This is a base class for all exception sinks (media).  It provides support
for formating the output.  You should not use it directly, but one of it's
subclasses, like L<JAAS::Log::File>, L<JAAS::Log::STDERR> or
L<JAAS::Log::Mail>.

=head1 METHODS

=head2 new

    my $sink=JAAS::Log::Sink->new($params);

Instantiates the object.  Basically blesses the params :)  $params should
contain a 'syntax' key.  It's value is what gets output.

=head2 output

    $sink->output($msg);

This is the method that sends the formated message to whereever it should
go.  You really must overload this function.

=head2 HUP

Does nothing.  Overload it please.

=head2 message

    $sink->message($exception);

This is where L<JAAS::Monitor> will send the exception.  We format the
exception into text using C<syntax>.  C<syntax> may contain words between
square brakets.  These are words are interpolated with values from the
exception.  If the value isn't in the exception, it's left alone.  It may 
also contain \n which is converted to a newline and \t which is converted to
a tab.

=head2 flatten

    my $hashref=$sink->flatten($exception, $prefix);

C<flatten>'s job is to recursively turn the exception into a hashref that
will be used to interpolate the [words] in the output lines (C<syntax>).  It
shouldn't be called by you.  It is documented so that you can overload it if
you so feel.

=head1 AUTHOR

Philip Gwyn <jaas at awale.qc.ca>, 
Learry Gagn <mou-jaas at awale.qc.ca>.

=head1 SEE ALSO

perl(1), POE, L<POE::Catcher>

=cut

$Log: Sink.pm,v $
Revision 1.1  2001/11/14 05:30:43  fil
Added support for all the funky, junky, monkey logging.  Hope I docoed it too.

