# $Id: Monitor.pm,v 1.5 2002/06/07 08:22:34 fil Exp $
package JAAS::Monitor;

use strict;
use vars qw($VERSION);
use Carp;
use Data::Dumper;

$VERSION = '0.03';

BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () { 0 }; }
}

##########################################################
sub new
{
    my($package)=@_;
    return bless {triggers=>{}, ID=>1}, $package;
}

##########################################################
sub add
{
    my($self, $triggers, $what)=@_;

    my $r=ref $what;
    unless($r and ($r eq 'ARRAY' or $r eq 'CODE')) {
        carp "Must specify an arrayref or coderef";
        return;
    }

    unless(ref($triggers)||'' eq 'ARRAY') {
        carp "Must specify an arrayref";
        return;
    }

    my $id=$self->{ID}++;
    $self->{triggers}{$id}=[$what, $triggers];
    return $id;
}

##########################################################
sub remove
{
    my($self, $id)=@_;
    delete $self->{triggers}{$id};
}

##########################################################
sub checks
{
    my($self, $exceptions)=@_;
    return unless $exceptions;
    DEBUG and print STDERR ref($self), "->checks()\n";
    $exceptions=[$exceptions] unless 'ARRAY' eq ref $exceptions;
    foreach my $ex (@$exceptions) {
        my $is_hash=1 if UNIVERSAL::isa($ex, 'HASH');
        $ex->{monitor_matches}=0 if $is_hash;
        my @todo;
        foreach my $trigger (values %{$self->{triggers}}) {
            next unless $self->matches($trigger->[1], $ex);
            push @todo, $trigger->[0];
            $ex->{monitor_matches}++ if $is_hash;
        }
        foreach my $q (sort @todo) {
            if(ref $q eq 'CODE') {
                DEBUG and print STDERR ref($self), " calling coderef\n";
                $q->($ex);
            } 
            else {
                # assume user isn't dumb enough to want to post
                # but never created a session!
                $POE::Kernel::poe_kernel->post(@$q, $ex);
            }
        }
    }
}

##########################################################
sub matches
{
    my($self, $trigger, $excp)=@_;

    # DEBUG and print STDERR Dumper $excp, $trigger;
    foreach my $k (keys %$trigger) {
        # if any one match fails, then they don't match
        return unless exists $excp->{$k} 
                        and $trigger->{$k} eq $excp->{$k};
    }
    DEBUG and print STDERR "Matched!\n";
    # if we get this far, no match failed, so all of $trigger is happy
    return 1;
}

1;

__END__
# Below is the stub of documentation for your module. You better edit it!

=head1 NAME

JAAS::Monitor - JAAS monitoring encapsulation

=head1 SYNOPSIS

    use JAAS::Monitor;
    use JAAS::Catcher;

    my $monitors=JAAS::Monitor->new();

    # define a coderef monitoring point
    my $id1=$monitors->add({type=>'warning', package=>'MyPackage'}, 
                            \&my_handler);

    # define a POE session monitoring point
    my $id2=$monitors->add({type=>'error', package=>'MyPackage'}, 
                           ['session', 'event']);

    # check a bunch of exceptions
    my $stack=JAAS::Catcher->get_stack;
    if(@$stack) {
        JAAS::Catcher->clear_stack;
        $monitors->checks($stack);
    }

    # remove the monitoring point
    $monitors->remove($id1);

=head1 DESCRIPTION


=head1 METHODS

=head2 new

    my $monitor=JAAS::Monitor->new();

Creates the object.  As there are no parameters, there is little doco either.

=head2 add

    my $id=$monitor->add($triggers, $coderef);
    $id=$monitor->add($triggers, ['session', 'event']);

Adds a new trigger to the monitor.  C<$triggers> should be a hashref.  It is
compared to each exception.  If all key/value pairs match, the exception is
sent to the second parameter (the action), which can be a coderef, or an
arrayref.  If it's a coderef, the it's called with the exception as it's
sole parameter.  If the action is an arrayref, then it's assumed to be a POE
session and event and it gets posted with the exception as it's sole
parameter.

=head2 remove

   $monitor->remove($id);

Removes a previously defined trigger from the collection.  C<$id> should be a
value that was previously retruned by C<add>.

=head2 checks

    $monitor->checks($array_of_exceptions);

Scans the C<$array_of_exceptions> to see if they match any of the triggers. 
If it does, then the trigger's action is called.  If the exception is a
hashref the key monitor_match is incremented.

=head2 matches

    $monitor->matches($trigger, $exception);

Used to check if an exception matches a trigger.  Returns true if all the
keys in trigger are present in exception and all the values match.  May be
called recursively.  Normaly you don't care about this, but it's documented
here if you wanted to overload this class.

Default implementation assumes that trigger and exception are hashrefs.

=head1 AUTHOR

Philip Gwyn <jaas at awale.qc.ca>, 
Learry Gagn <mou-jaas at awale.qc.ca>.

=head1 SEE ALSO

perl(1), POE, L<JAAS::Catcher>

=cut

$Log: Monitor.pm,v $
Revision 1.5  2002/06/07 08:22:34  fil
Bumped version

Revision 1.4  2001/11/14 05:30:43  fil
Added support for all the funky, junky, monkey logging.  Hope I docoed it too.

Revision 1.3  2001/10/19 05:24:42  fil
Added 1; true to file

Revision 1.2  2001/07/27 21:42:23  fil
Added tests for EXEC_CTX, damnit

Revision 1.1.1.1  2001/07/13 06:46:41  fil
Initial import

