package JAAS::Object;
use strict;
use vars qw($VERSION);

$VERSION = '0.03';

use Carp;
use Data::Denter;
use JAAS::Object::Relationships;
use JAAS::Object::Services;
use POE::Kernel;
use POE::Session;
use vars qw($CATCHER $ERROR $MONITOR $CTX);

BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () { 0 }; }

    if($INC{'JAAS/Catcher.pm'}) {
        $CATCHER='JAAS::Object::_Monitored';
        $ERROR='JAAS::Object::_Error';
        JAAS::Catcher->import(qw(EXEC_CTX));
    } 
    else {
        $CATCHER='JAAS::Object::_Catcher';   
        $ERROR='JAAS::Object::_TextError';
        $CTX={};
        *EXEC_CTX=sub {
            my $n=$_[1];
            @{$CTX}{keys %$n}=values %$n;
        };

        if($INC{'JAAS/Error.pm'}) {
            $ERROR='JAAS::Object::_Error';
        }
    }
}

################################################################
sub spawn
{
    my $package=shift @_;
    my %args=@_;

    my %create;
    foreach my $k (qw(options args)) {
        next unless exists $args{$k};
        $create{$k}=delete $args{$k};
    }
    my $self=$package->new();
    my $me=POE::Session->create(
            args=>[$args{object}],
            object_states=>[
                $self=>[qw(_start _shutdown  _default _stop _attach _detach
                           _postback _monitor 
                           _alarum_do _alarum_set _alarum_remove 
                           _alarum_adjust)],
            ],
            %create,
        );
    return $me->ID;
}

################################################################
sub new
{
    return bless {alarms=>{}}, $_[0];
}

################################################################
sub _start
{
    my($kernel, $self, $session, $delegate)=@_[KERNEL, OBJECT, SESSION, ARG0];


    # +GC
    # $kernel->refcount_increment($session->ID, 'waiting for delegate');
    $kernel->alias_set(0+$session);

    if($delegate) {
        shift(@_)->_attach(@_);         # pain
    }
    return;
}


################################################################
sub _attach
{
    my($kernel, $session, $self, $delegate)=@_[KERNEL, SESSION, OBJECT, ARG0];
    shift(@_)->_detach(@_);     # look ma, no call()
    DEBUG and warn "#", $session->ID, " attach\n";

    croak "No delegate" unless $delegate;

    # get the new methods
    my @m=$delegate->methods;
    my %m;
    if(scalar(@m) & 1) {
        print STDERR "Bad return from $delegate\->methods\n";
    } else {
        %m=@m;
    }
    my %methods;
    foreach my $method (keys %m) {
        croak "Please don't call your method $method" if $method=~ /^_/;
        $methods{$method}=$m{$method};
    }
    $self->{methods}=\%methods;
    # save object
    $self->{delegate}=$delegate;

    # addressing
    $self->{alias}=$delegate->name;
    DEBUG and
        warn "#", $session->ID, " is called $self->{alias}\n";
    $kernel->alias_set($self->{alias});

    # -GC
    # $kernel->refcount_decrement($session->ID, 'waiting for delegate');
    $kernel->alias_remove(0+$session);
}

################################################################
sub _detach
{
    my($kernel, $session, $self)=@_[KERNEL, SESSION, OBJECT];
    return unless $self->{delegate};
    DEBUG and warn "#", $session->ID, " detach\n";
    # +GC
    # $kernel->refcount_increment($session->ID, 'waiting for delegate');
    $kernel->alias_set(0+$session);
    # -addressing
    $kernel->alias_remove($self->{alias});
    # clear relationships
    $jaas_relationships->remove($self->{alias}) if $self->{alias};
    # clean-up
    delete $self->{alias};
    delete $self->{methods};
    delete $self->{delegate};
}

################################################################
sub _shutdown
{
    my($kernel, $session)=@_[KERNEL, SESSION];
    DEBUG and warn $session->ID, "/shutdown";
    shift(@_)->_detach(@_);     # look ma, no call()
    # -GC
    # $kernel->refcount_decrement($session->ID, 'waiting for delegate');
    $kernel->alias_remove(0+$session);
}










################################################################
sub _alarum_do
{
    my($kernel, $self, $id, $to, $args) =
                    @_[KERNEL, OBJECT, ARG0, ARG1, ARG2];
    DEBUG and print STDERR "Doing alarm $id ($self->{alarms}{$id})\n";
    delete $self->{alarms}{$id};
    $jaas_services->post($to=>@$args);
}

sub _alarum_set
{
    my($kernel, $self, $id, $to, $time, $args) =
                    @_[KERNEL, OBJECT, ARG0, ARG1, ARG2, ARG3];
    $self->{alarms}{$id} =
                $kernel->alarm_set(_alarum_do=>$time, $id, $to, $args);
    DEBUG and print STDERR "Setting alarm $id ($self->{alarms}{$id})\n";
}

sub _alarum_remove
{
    my($kernel, $self, $id) = @_[KERNEL, OBJECT, ARG0];
    DEBUG and print STDERR "Removing alarm $id ($self->{alarms}{$id})\n";
    $kernel->alarm_remove(delete $self->{alarms}{$id});
}

sub _alarum_adjust
{
    my($kernel, $self, $id, $delta) = @_[KERNEL, OBJECT, ARG0, ARG1];
    return unless $self->{alarms}{$id};

    $kernel->alarm_adjust($self->{alarms}{$id}, $delta);
}


################################################################
sub _default
{
    # gratuitous use of splice() detected at line 123 of file objects.perl
    my @POE=splice @_, 0, ARG0; 

    my($method, $args)=@_;
    return if $method =~ /^_/;

    my($self)=$POE[OBJECT];
    $POE[STATE]=$method;                # fix this bug please



    local $CTX;
    EXEC_CTX(my $c0, $jaas_services->exec_ctx());
    EXEC_CTX(my $c2, {method=>$method, alias=>$POE[SESSION]->ID});

    ##############################################
    # set up call
    my $delegate=$self->{delegate};
    unless($delegate) {
        $self->usage_error("Post without delegate object");
        return;
    }
    EXEC_CTX(my $c1, {delegate=>ref($delegate), alias=>$self->{alias}});

    DEBUG and
        print STDERR "\nAttempt: session ", $POE[SESSION]->ID, " $method\@$self->{alias} w/ ", 0+@$args, " parms\n";

    ##############################################
    # check for OOB
    my $OOB;
    my $c12;
    if($args and @$args and UNIVERSAL::isa($args->[-1], 'JAAS::OOB')) {
        $OOB=pop @$args;
        DEBUG and warn "We have OOB data, ", Denter $OOB;
        EXEC_CTX($c12, {OOB=>$OOB});
    }

    ##############################################
    # does method exist?
    unless($self->{methods}{$method}){
        warn "YEARCH! $method not exposed";
        $self->usage_error("unexposed method");
        return;
    }

    ##############################################
    # for real?
    unless($delegate->can($method)) {
        $self->usage_error("undefined method");
        return;
    }

    ##############################################
    my $about_method=$self->{methods}{$method};

    ##############################################
    # set up postback 
    # NB: must come before argument check, because we modify args
    my $postback;
    if($about_method->{respond} and not ($OOB and $OOB->{'reply-to'})) {
        unless(@$args) {
            EXEC_CTX(my $c3, {sender=>$POE[SENDER]->ID});
            $self->usage_error("no postback method to respond to");
            return;
        }
        # print STDERR Dumper $OOB;

        $postback=pop @$args;
        die "$postback is wrong", Dumper $args 
                        if UNIVERSAL::isa($postback, 'JAAS::OOB');
        $postback={to=>$postback} unless ref $postback;
        # TODO : This isn't 100% right.  Maybe SENDER has many names...
        # how can we check that $session is in fact equiv to SENDER?
        $postback->{default_object}=$POE[SENDER]->ID;
        DEBUG and warn "We are going to respond";
    }


    ##############################################
    # check arguments
    if(defined($about_method->{args_n}) and 
                $about_method->{args_n}!=@$args) {
        EXEC_CTX(my $c3, {sender=>$POE[SENDER]->ID});
        DEBUG and warn "FOOOIE!";
        $self->usage_error("requires $about_method->{args_n} argument(s), we have ".(0+@$args));
        return;
    }

    my $code=$delegate->can($method);
    unless($code) {
        $self->usage_error("no method $method in class ".ref($delegate));
        return;
    }

    ###############################################################
    # do the call
    $@='';
    my($OK, $ret);

    $CATCHER->monitor($ERROR, 
        sub {
            # set up services
            $jaas_services->_POE(\@POE);
            $jaas_services->_OOB($OOB);

            DEBUG and print STDERR "Invoking ", ref($delegate), "->$method\n";
            # always give scalar context    
            $ret=$code->($delegate, @$args);

            # don't leave our values hanging
            $jaas_services->_OOB();
            $jaas_services->_POE();

            $OK=1;
        }
    );

    #################################
    # The next few things are done in a "continuation" states, so that we 
    # pop up the stack at some point

    # check the monitor
    # TODO: error-to should be a temporary monitor or something
    $POE[KERNEL]->yield('_monitor') if $CATCHER->if_caught;

    # bad things ... maybe error-to would be handled here?
    return unless $OK;                  

    # try to not do anything
    if($OOB) {
        if($OOB->{continuing}) {
            $OOB->{continuing}=0;
            DEBUG and warn "OOB but continuing ", Denter $OOB;
            return;
        }
    }


    if($OOB and $OOB->{'reply-to'}) { 
        DEBUG and 
            print STDERR "*********** OOB reply-to from $method\@$self->{alias}";
        # doing both a "respond" style postback AND a 'reply-to' would
        # be illogical, so reply-to has priority
        $POE[KERNEL]->yield('_postback', $OOB->{'reply-to'}, $ret, $OOB);
    } 
    elsif($postback) {                     # method has "respond" fingerprint
        DEBUG and print STDERR "normal respond style postback\n";
        $POE[KERNEL]->yield('_postback', $postback, $ret, $OOB);
    } 
    else {
        return $ret; 
    }
    return;
}

################################################################
# Handy wrapping paper
sub usage_error
{
    my($self, $msg)=@_;
    EXEC_CTX(my $c0, {usage=>1});
    $CATCHER->error($ERROR, $msg);
    $jaas_services->continuation('_monitor');
    $jaas_services->_POE();
    return;
}

################################################################
sub _monitor
{
    my($self, $kernel)=@_[OBJECT, KERNEL];
    DEBUG and warn __PACKAGE__, "->_monitor\n";
    return $CATCHER->check_stack;
}

################################################################
sub _postback
{
    my($postback, $ret, $OOB)=@_[ARG0, ARG1, ARG2];

    if(ref $postback) {
        $postback={to=>$postback} unless $postback->{to};
    }
    DEBUG and 
        do {
            my $a=$postback;
            $a=$postback->{to} if ref $a;
            $a=$a->{method}.'@'.$a->{object} if ref $a;
            my $p=defined $ret ? $ret : 'undef()';
            warn __PACKAGE__, "->_postback ->- $a ($p)\n";
        };

    my @ret=($ret);
    if($OOB->{'wantarray'} and ref $ret eq 'ARRAY') {
        @ret=@$ret;                 # ARRAY + wantarray == hapiness
    }
    if($OOB->{__todo}) {
        $postback->{__todo}=$OOB->{__todo};
    }
    $jaas_services->_OOB($OOB);
    $jaas_services->post($postback => @ret);
    $jaas_services->_OOB();
}

################################################################
sub _stop
{
    my($kernel, $self)=@_[KERNEL, OBJECT];
    # we don't have to do anything now, because if we are GCed, it's because
    # we have no delegate anyway
    DEBUG and warn "_stop";
}






##############################################################################
## Pretend we are a JAAS::Catcher class

package JAAS::Object::_Catcher;
use strict;
use Carp;

sub DEBUG ();
BEGIN {*DEBUG=\&JAAS::Object::DEBUG;}
my @CAUGHT;

sub dump_stack
{
    my $q=1;
    my @stack;
    print STDERR "stack: \n";
    while(@stack=caller($q)) {
        print STDERR "$q: $stack[3] $stack[2] $stack[1]\n";
        $q++;
    }
}

################################################################
sub monitor ($$&)
{
    my($package, $error_p, $sub)=@_;
    $@='';
    die "NO SUB" unless defined $sub;
    # dump_stack;
    eval { 
        local $SIG{__WARN__}=
                sub { $error_p->new($_[0], 'warning');};
        $sub->();
    };

    $package->error($error_p, $@) if $@;
    $@='';
}

################################################################
sub if_caught
{
    my($package)=@_;
    return 0 != @CAUGHT;
}

################################################################
sub check_stack
{
    my($package)=@_;
    my @e=@CAUGHT;
    return unless @e;
    @CAUGHT=();
    
    foreach my $ex (@e) {
        $JAAS::Object::Services::jaas_services->monitor_inform(@$ex);
    }
}

################################################################
sub warn 
{
    my($package, $error_p, $msg)=@_;
    my $w=$error_p->new($msg, 'warning', $JAAS::Object::CTX);
    DEBUG and CORE::warn "warn -- ".(ref($w) ? $msg : $w )."\n";
    push @CAUGHT, ['on_warning', $JAAS::Object::CTX->{alias}, $w];
}

################################################################
sub error
{
    my($package, $error_p, $msg)=@_;
    my $e=$error_p->new($msg, 'fatal', $JAAS::Object::CTX);
    DEBUG and CORE::warn "error -- ".(ref($e) ? $msg : $e )."\n";
    push @CAUGHT, ['on_error', $JAAS::Object::CTX->{alias}, $e];
}

################################################################
sub mk_handler
{
    my($package)=@_;
    croak "Please don't use $package this way";
}

##############################################################################
## Straight text as our exception "object"
package JAAS::Object::_TextError;
use strict;

################################################################
sub new
{
    my($package, $error, $type, $ctx)=@_;
    my $ret='';
    $ret.=$ctx->{method}                if $ctx->{method};
    $ret.="\@$ctx->{alias}: "           if $ctx->{alias};
    $ret ="$ctx->{delegate}: $ret"      if $ctx->{delegate};
    $ret.=$error;
    return $ret;
}















##############################################################################
## Uses JAAS::Catcher as a catcher

package JAAS::Object::_Monitored;
use strict;
use vars qw(@ISA);

@ISA=qw(JAAS::Catcher);

################################################################
sub if_caught
{
    my($package)=@_;
    return 0 != @{$package->get_stack()};
}


################################################################
sub check_stack
{
    my($package)=@_;
    local $SIG{__DIE__}='DEFAULT';
    local $SIG{__WARN__}='DEFAULT';
    my $errors=$package->get_stack();
#    warn "Gots me ", 0+@$errors, " errors";
    return unless @$errors;
    $package->clear_stack();

    foreach my $ex (@$errors) {
        my $t=$ex->{type} eq 'fatal' ? 'on_error' : 'on_warning';
        $JAAS::Object::Services::jaas_services->monitor_inform(
                        $t, $ex->{alias}, $ex);
    }
}

##############################################################################
## Uses JAAS::Error for exception encapsulation

package JAAS::Object::_Error;
use strict;

use vars qw(@ISA);
@ISA=qw(JAAS::Error);

################################################################
sub add_locus
{
    my($self)=@_;
    my $ret="$$: ";
    if($self->{delegate}) {
        my $q=$self->{delegate};
        $q.="(".$self->{line}.")"       if $self->{line};
        $ret.=$q." ";
    }
    $ret.=$self->{method}               if $self->{method};
    $ret.='@'.$self->{alias}            if $self->{alias};
    $self->{full_text}=$ret.' '.$self->{full_text}."\n";
    return $ret;
}








1;
__END__

=head1 NAME

JAAS::Object - JAAS object layer for POE

=head1 SYNOPSIS

    use JAAS::Object;

    package Household::Thermometer;
    #  .... implementation details

    package main;
    my $thermo=new Household::Thermometer->new(port=>'/dev/ttyC10',
                                               name=>'TAttic');

    JAAS::Object->create(object=>$thermo);

    # clear the min/max in the thermometer, from anywhere in POE-land
    $poe_kernel->post(TAttic=>'clear');

    # or inside of JAAS
    $jaas_services->post('clear@TAttic');



    # This package controls a hypothetical temperature probe connected
    # to a cyclades card
    package Household::Thermometer;

    #  ... new() and IOcommand() are left as an exercise for the reader

    sub name { $_->{name}}

    sub methods
    {
        return (read=>{args_n=>0, respond=>1},
                interval=>{args_n=>1},
                clear=>{});
    }

    sub read {
        my($self)=@_;
        return $self->Icommand('READ PROBE');
    }

    sub interval {
        my($self, $t)=@_;

        die "$t is MUCH to big" if $t > 3600;       # Exceptions get caught

        $self->Ocommand("SET INTERVAL=$t");
    }

    sub clear {
        my($self)=@_;
        $self->Ocommand("CLEAR");
    }


=head1 DESCRIPTION

The JAAS object layer is an object model that interfaces with POE.  It sits
on top of POE and provides services to all objects that I knows about.
Services include posting messages to other objects, monitoring objects,
postbacks and responses.  It also provides a tuple-space for creating and
maintaining relationships between objects.

JAAS::Object actually only handles receiving messages from POE-space for a
particular object.  Posting a message to POE-space is handled by
JAAS::Object::Services.  Object relationships are, naturally enough,
encapsulated in JAAS::Object::Relationships.

JAAS::Object will use JAAS::Error and JAAS::Catcher to handle execptions if 
they are loaded before JAAS::Object is.

=head2 ADDRESSING

Message and object addressing issues are going to be sticky.  Currently, I
use method@object, but this probably won't last.  Note that object is any
POE alias, including sessions that have nothing to do with JAAS and that
'method' could be a POE event.

    If you subscribed to a session via IKC
    event@poe://Kernel/Session

    Mythical Stem interoperability
    target@stem:Hub/Cell

THIS IS GOING TO CHANGE.


=head1 METHODS

=head2 spawn

    JAAS::Object->spawn(%parameters);

This creates a POE::Session that deals with receiving messages (aka events)
from POE.  C<%parameters> is a hash with the following keys:

=over 4

=item C<object>

Value is your blessed object.  See THE DELEGATE OBJECT below.

=item C<options>

Options to be passed to C<POE::Session>->C<create()>.  Useful mainly for
debugging purposes.

=back

=head1 EVENTS

=head1 _attach

    $kernel->post($id=>'_attach', $object);

Replace the object that the session is attached to.

=head1 _detach

    $kernel->post($id=>'_detach');
    $object=$kernel->call($id=>'_detach');

Break the link between the session and the object.  Note that after you
detach the object, the session will be GCed by the kernel.  To prevent
this from happening, you should use external refcounts or something.

=head2 _shutdown

    $kernel->post($id=>'_shutdown');

Tells the session to go away.  You must call this after a C<_detach>
otherwise the session will never be GCed.



=head1 POE GARBAGE COLLECTION

JAAS::Object uses 2 tactics to prevent the kernel from collecting it.
While it is attached to an object, it registers a session alias.  When
it doesn't have an object (after a C<_detach> or between spawn() and
new()) it keeps an external reference.

This has 2 concequences :

When the kernel detects that the only thing keeping it alive is aliases,
it attempts to declare those sessions zombies, and will throw them away
until it can GC everything and shutdown.  This will work if you have
attached objects.

If you have detached sessions, they will never every go away.  Hence the
kernel will never shut down.  Make sure that each C<_detach> has a
corresponding C<_attach> or C<_shutdown>.



=head1 THE DELGATE OBJECT

The delegate object must have 2 methods:

=head2 name

    $name=$object->name;

The name is how the object will be addressed, via POE's session aliases.

=head2 methods

    %methods=$object->methods;

Defines the objects public interface.  Returns a hash, with public
method names as keys, and method "prototypes" (see below) as values.  A
method name corresponds to a POE event with the same name.  Method names
must not start with '_' because POE uses that for internal use.  And it
would be bad form anyway.  There are also 2 specials methods:

Method prototypes are a hashref that may contain the following:

=over 4

=item C<args_n>

Number of arguments the method expects.  Method will not be called if you do
not supply this exactly this amount.  Currently, a warning is displayed and
the method is not called.  This will change when I get exception handling
sorted out.

COMING SOON: args_n could also be an arrayref of [min, max] so that one
could specify a range.

=item C<respond>

Flag that indicates that at the last parameter at invocation is in fact the
address of a message that the method will send it's output to.  This is one
way of avoiding the dreaded C<call()>.  The other is 'reply-to' (see
L<JAAS::Object::Services>).

To use the C<read> method in the example given in SYNOPSIS, you do the
following:

    $jaas_services->post('read@TAttic', 'readed_values');
    # method 'readed_values' in current object will receive 
    # the current temperature reading


=back

=head1 AUTHOR

Philip Gwyn <jaas at awale.qc.ca>, 
Learry Gagn <mou-jaas at awale.qc.ca>.

=head1 SEE ALSO

perl(1).

=cut
