# $Id: Delegate.pm,v 1.7 2001/10/19 05:25:02 fil Exp $
package JAAS::Object::Delegate;
use strict;
use vars qw($VERSION);

$VERSION = '0.01';

use Carp;

BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () {0}; }
}

################################################################
sub new
{
    my($package, $args, $children)=@_;
    $children||={};
    return bless {%$args, %$children}, $package;
}

################################################################
sub name
{
    my($self)=@_;
    return $self->{name};
}

################################################################
sub methods
{
    my($self)=@_;
    die "Must implement ", ref($self), "::methods\n";
}


################################################################
sub continuation
{
    my($self, $to, @params)=@_;     
    DEBUG and warn "continuation with ", 0+@params, " arguments";
    JAAS::Object::Services->continuation($to, @params);
    return;         
}



################################################################
# For JAAS::Object::Context
sub lifetime_save
{
    my($self, $when)=@_;
    return;                         # refuse persistance
}

################################################################
# For JAAS::Object::Context
sub lifetime_restore
{
    my($self, $when, $data)=@_;     
    # blindly accept everything
    @{$self}{keys %$data}=values %$data;
    return;         
}

################################################################
# For JAAS::Widget
sub get_field
{
    my($self, $field)=@_;
    return unless UNIVERSAL::isa($self, 'HASH');
    return $self->{$field};
}

################################################################
# For JAAS::Widget
sub set_field
{
    my($self, $field, $value)=@_;
    return unless UNIVERSAL::isa($self, 'HASH');
    DEBUG and 
        warn "Setting field $field to $value\n";
    $self->{$field}=$value;
}


1;
__END__

=head1 NAME

JAAS::Object::Delegate - Base class for JAAS delegate object 

=head1 SYNOPSIS

    package My::Object
    use strict;
    use JAAS::Object::Delegate;
    use vars qw(@ISA);
    @ISA=qw(JAAS::Object::Delegate);

    sub new
    {
        my($package, $args)=@_;
        # here we check $args

        my $self=$package->SUPER::new($args);

        # here we would init things in $self

        return $self;
    }

    sub methods
    {
        # .....
    }

=head1 DESCRIPTION

The JAAS delegate objects are the bits that do the real work in the JAAS
object layer.  This objects provides a virtual base class for the delegates
so that you don't have to think of every single thing when building a new
class.

=head1 METHODS

=head2 new

    sub new
    {
        my($package, $parameters);
        # ...
        return $self
    }

Instanciates a new delegate object.  The object will a shallow copy of
$parameters blessed into $package.  $parameters is assumed to be hashref with the
following keys:

=over 4

=item C<name>

Name of the objects, used by C<name> below.

=back

=head2 name

    sub name
    {
        my($self)=@_;     
        # ....
        return $name
    }

Returns a name for this object.  See C<JAAS::Object> and
C<JAAS::Entrypoint>.

Default behaviour is to return the {name} value in $self.

=head2 methods

    sub methods
    {
        my($self)=@_;     
        # ....
        $return $methods;
    }

Returns a hashref of exposed methods for this objects.  See C<JAAS::Object>.

Default behaviour is to throw an exception, so it must be overloaded.


=head2 lifetime_save

    sub lifetime_save
    {
        my($self, $when)=@_;     
        # ....
    }

Returns a hashref of what we want to be saved at a given point.  Return
undef() if you want to refuse persistance.  See C<JAAS::Object::Lifetime>.

Default behaviour is to refuse persistance.

=head2 lifetime_restore

    sub lifetime_restore
    {
        my($self, $when, $data)=@_;     
        # ....
    }

Captures the "persisted" data into the current object.  See
C<JAAS::Object::Lifetime>.

Default behaviour is to blindly copy all the loaded data into the object.


=head2 get_field

    sub get_field
    {
        my($self, $field_name)=@_;
        # ....
    }

=head2 set_field

    sub set_field
    {
        my($self, $field_name, $new_value)=@_;
        # ....
    }




=head1 AUTHOR

Philip Gwyn <jaas at awale.qc.ca>, 
Learry Gagn <mou-jaas at awale.qc.ca>.

=head1 SEE ALSO

perl(1).

=cut


