# $Id: Persistant.pm,v 1.10 2002/04/17 19:23:21 fil Exp $
package JAAS::Object::Cache::Persistant;

use strict;
use JAAS::Object::Cache::Simple;
use Carp;
BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () {0}; }
}

###########################################################
sub new
{   
    my($package, $lifetimer)=@_;
    return bless {life=>$lifetimer,
                  interest=>{},
                  obj=>{}}, $package;
}

###########################################################
sub setup
{
    my($self, $id, $who)=@_;
    confess "No ID" unless $id;
    
#    use Data::Dumper;
#    warn "We already have $id: ", Dumper $self->{obj}{$id};
    my $ret=0;
    DEBUG and warn "$who is interested in $id";
    unless($self->{obj}{$id}) {
        DEBUG and warn "restoring $id for $who using $self->{life}";
        $self->{obj}{$id}={};
        DEBUG and warn "$id to be restored";
        my $lo=$self->{life}->lock($id, 'context', 1);
        unless($lo) {
            die "No lock (context will be EMPTY)";
        }
        if(ref $lo) {
            $self->{locks}{$id}=$lo;
            $self->{life}->restore($self->{obj}{$id}, 'context', $id);
            # warn "We got ", Dumper $self->{obj}{$id};
            $ret=1;
        } 
        elsif($lo==0) {
            warn "Unable to open lock file: $! (context will be EMPTY)";
        }
        else {              # flock() failed
            die "Failed to lock context: $! (context will be EMPTY)";   
        }
    } 
    elsif(0) {
        use Data::Dumper;
        warn "We already have $id: ", Dumper $self->{obj}{$id};
    }
    $self->{interest}{$id}{$who}=1;
    return $ret;
}

###########################################################
sub clear
{
    my($self, $id, $who)=@_;
    return unless exists $self->{interest}{$id}{$who};

    delete $self->{interest}{$id}{$who};
    DEBUG and warn "$who isn't interested in $id";

    return if keys %{$self->{interest}{$id}};

    DEBUG and warn "$id is GONE, saving";
    # no more interest
    $self->{life}->save($self->{obj}{$id}, 'context', $id);
    $self->{locks}->{$id}->();

    delete $self->{obj}{$id};
    delete $self->{interest}{$id};
    delete $self->{locks}->{$id};
    return 1;
}

###########################################################
sub shutdown
{
    my($self, $ids)=@_;
    foreach my $id (@$ids) {
        foreach my $who (keys %{$self->{_interested}}) {
            $self->clear($id, $who);
        }
    }
}

###########################################################
sub fetch
{
    my($self, $id, $name)=@_;
    confess "No ID" unless $id;
    DEBUG and warn "Fetching $id $name";
    return unless exists $self->{obj}{$id};
    return unless exists $self->{obj}{$id}{$name};
    return $self->{obj}{$id}{$name}->[0]; # unless wantarray;
#    return @{$self->{obj}{$id}{$name}};
}

###########################################################
sub store
{
    my($self, $id, $name, $data)=@_;
    confess "No ID" unless $id;
    $self->{obj}{$id}{$name}||=[];

    DEBUG and warn "Saving $id $name";
    # TODO: This is broken!  Widgets are a bunch of arrays
    # TODO: so we need to get rid of all this bullshit, or make saving
    # TODO: widgets a bit smarter... :-/
    # if(ref $data eq 'ARRAY') {      
    #   DEBUG and warn "Adding ARRAY to $id/$name";
    #   return push @{$self->{obj}{$id}{$name}}, $data;
    # }

    # use Data::Dumper;
    # warn Dumper $self->{obj}{$id};

    # If we turn the arrayref into a "scalar", we might mess up
    # lifetime.... :/
    $self->{obj}{$id}{$name}->[0]=$data;
}



###########################################################
sub delete
{
    my($self, $id, $name)=@_;
    confess "No ID" unless $id;
    die "NO NAME" unless $name;
    my $ret= delete $self->{obj}{$id}{$name};
    delete $self->{obj}{$id} unless keys %{$self->{obj}{$id}};
    return $ret;
}

###########################################################
sub list
{
    my($self, $id)=@_;
    return [keys %{$id ? ($self->{obj}{$id}||{}) : $self->{obj}}] unless $id;
}

sub _status
{
    my($self, $id)=@_;
    if($id) {
        return join "\n", "Contains of $id:",
                        map {"    $_ ".@{$self->{obj}{$id}{$_}}}
                        keys %{$self->{obj}{$id}};
    }
    return join "\n    ", "Contains contexts:", keys %{$self->{obj}};
}

###########################################################

1;

__DATA__

=head1 NAME

JAAS::Object::Cache::Persistant - Persistant object repository cache

=head1 SYNOPSIS

    my $cache=JAAS::Object::Cache::Persistant->new($lifetime);
    $cache->setup($id);
    $cache->store($id, $name, $obj);
    $cache->clear($id);

    $cache->setup($id);                     
    $obj$cache->fetch($id, $name);      # $id/$name still present
    $cache->clear($id);


=head1 DESCRIPTION


JAAS::Object::Cache::Persistant is an implementation of the repository cache
interface that adds persistance.  It does so via an object called a
"lifetimer" (JAAS::Object::Lifetime) that saves and restores the context
when the need arrises.

=head1 METHODS

=head2 new

    my $cache=JAAS::Object::Cache::Persistant->new($lifetime);

Object constructor.  C<$lifetime> is an object that is used for persistance.

=head2 setup

    $cache->setup($id);

If the context C<$id> doesn't exist in memory, the lifetimer is called as
follows:

    $lifetimer->restore($object, 'context', $id);

Where C<$object> is, as described in JAAS::Object::Lifetime.

=head2 clear

    $cache->clear($id);

If no one else is interested in context C<$id>, the lifetimer is used to
save the context as follows:

    $lifetimer->save($object, 'context', $id);

Where C<$object> is described in JAAS::Object::Lifetime.

=head1 BUGS

This interface is still under construction.  I will probably redo it very
soon.

=head1 AUTHOR

Philip Gwyn <jaas at awale.qc.ca>, 
Learry Gagn <mou-jaas at awale.qc.ca>.

=cut

$Log: Persistant.pm,v $
Revision 1.10  2002/04/17 19:23:21  fil
*** empty log message ***

Revision 1.9  2002/04/17 19:04:05  fil
Added lifetime locking to Persistant

Revision 1.8  2001/10/11 02:16:56  fil
Added comments
Added refcounts to setup/clear so that a context can be setup after a clear
but before being deleted from memory.

Revision 1.7  2001/09/21 18:52:15  fil
Debugging mess

Revision 1.6  2001/09/13 00:37:12  fil
Fixed Persistant so that it doesn't look like a context exists
even if it was only quieried t/20_cache.t

Revision 1.5  2001/08/14 06:21:11  fil
Tests are broken :(

Revision 1.4  2001/08/14 06:07:57  fil
Ooops, it's not persist__!

Revision 1.3  2001/08/14 05:57:08  fil
Removed some annoying debug output

Revision 1.2  2001/08/09 16:35:12  fil
Small tweaks

Revision 1.1  2001/07/24 21:12:31  fil
Fibble

Revision 1.2  2001/07/05 23:56:37  fil
ID in Context->list could be a hash ref
Doco fix in Cache::Persistant
Moved Factory over to JAAS::Config

Revision 1.1  2001/06/22 20:44:07  fil
Fixed some CVS problems :)

