package JAAS::Application;

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

use Data::Dumper;
use POE::Session;
use JAAS::Entrypoint;

$VERSION = '0.03';
my $ID="RID0000";

BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () {1}; }
}

###############################################################################
sub spawn
{
    my($package, %heap)=@_;
#    use Data::Dumper;
#    die Dumper \%heap;
    POE::Session->create(
            args=>[delete $heap{config}],
            package_states=>[
                $package=>[qw(_start _default _child _shutdown 
                                response __timeout)],
            ],
            heap=>{%heap},
        );
}

##################################################
sub _start
{
    my($package, $kernel, $heap, $session)=@_[OBJECT, KERNEL, HEAP, SESSION];

    # how long can a request take?
    $heap->{timeout}||=600;     # 10 minutes

    # Create entry-point sessions now, rather then at request time
    $package->create_entries($heap, $heap->{entry_points}, $heap->{objects},
                                                $session);
    delete $heap->{entry_points};
    delete $heap->{objects};


    # set up aliases
    foreach my $a ($heap->{name}, @{$heap->{aliases}||[]}) {
        DEBUG and 
            warn "$package: name or alias: $a\n";
        $kernel->alias_set($a);
    }

    DEBUG and warn "$package: entry points: ", 
                    join(', ', keys %{$heap->{entry_point}}), "\n";

    # publish our entry-points
    $kernel->post(IKC=>'publish', 0, [keys %{$heap->{entry_point}}]);

    # monitor for shutdown events
    $kernel->post(IKC=>'monitor', '*'=>{'shutdown'=>'_shutdown'});


}


##################################################
# Entry points are in fact child sessions of ours
# _default will "route" the relevant requests to those sessions
sub create_entries
{
    my($package, $heap, $entries, $uses, $session)=@_;

    DEBUG and warn "$package: Creating entry points\n";

    # use Data::Dumper;
    # die Dumper $entries;
    while(my($name, $ep)=each %$entries) {
        DEBUG and warn "$package: entry point: $name\n";
        # $ep was prepared by JAAS::Server::create_applications from the
        # config file
        my $pack=delete $ep->{package};

        # fill in a few values from us    
        $ep->{name}||=$name;
        $ep->{objects}||=$uses;
        $ep->{application}=$heap->{name};

        # create
        my $sesID=$pack->spawn(%$ep);

        DEBUG and warn "$package: $name is $sesID ($pack)\n";
        $heap->{entry_point}{$name}=$sesID;
        $heap->{ses_to_entry}{$sesID}=$name;
    }
}


##################################################
sub _shutdown
{
    my($package, $kernel, $session, $heap)=@_[OBJECT, KERNEL, SESSION, HEAP];

    DEBUG and
        warn "$package: $heap->{name} _shutdown\n";

    foreach my $req_id (keys %{$heap->{request}}) {
        # response deals with cleaning up $heap and timeouts
        # we use call() because we want to send the response before
        # IKC goes away
        $kernel->call($session=>'response', 
                      $req_id=>{error=>{IKC=>'shutdown'}});
    }

    foreach my $entry (keys %{$heap->{entry_point}}) {
        # tell child it session can disappear now
        $kernel->post($heap->{entry_point}{$entry}=>'shutdown');

        # _child will deal with cleaning up 
        # $heap->{entry_point and ses_to_entry}
    }

}




##################################################
sub _default
{
    my($package, $kernel, $heap, $entry, $args)=
                                        @_[OBJECT, KERNEL, HEAP, ARG0, ARG1];
    $args=$args->[0];
    DEBUG and warn "_default ", Dumper $args;

    unless(exists $heap->{entry_point}{$entry}) {
            # not an entry point, handle it elsewhere
        my $package=shift @_;
        return $package->__default(@_);
    }
    # Normally, we should create a new entry point for each request
    # which would then create jaas::object sessions... which i find
    # wasteful.  This is only useful for many requests per-process anyway
    # so I'm creating all entries in create_entries
    my $ses=$heap->{entry_point}{$entry};

    # set up house keeping for the request
    my $req_id=$ID++;
    $heap->{request}{$req_id}={rsvp=>pop(@$args), session=>$ses};

    die "NO RSVP" unless $heap->{request}{$req_id}{rsvp};
    DEBUG and warn "rsvp: $heap->{request}{$req_id}{rsvp}";

    # set up a timeout
    my $to="__timeout_$req_id";
    $kernel->state($to, $_[OBJECT], '__timeout');
    $kernel->delay_add($to=>$heap->{timeout}, $req_id);

    # mess with GC
    $kernel->refcount_increment($ses, $req_id);

    DEBUG and warn "$package: poe:$heap->{name}/$entry ($ses / #$req_id)\n";

    # post it to the child session
    $kernel->post($ses=>'entry', $req_id, @$args);
}

##################################################
# this is what a "regular" _default would be
sub __default
{
    my($kernel, $heap, $state, $args)=@_[KERNEL, HEAP, ARG0, ARG1];
    return if $state =~ /^_/;
    DEBUG and warn "Someone posted state $state!\n";
    return;
}

##################################################
sub _child
{
    my($kernel, $heap, $how, $who)=@_[KERNEL, HEAP, ARG0, ARG1];
    if($how eq 'lose') {            # child went away
        my $id=$who->ID;
        my $entry=delete $heap->{ses_to_entry}{$id};

        DEBUG and warn "$entry ($id) is going away";

        $kernel->call(IKC=>'retract', 0, [$entry]);
        delete $heap->{entry_point}{$entry};

        # TODO: Maybe we also want to delete pending {request} for this $id?
        # TODO: probably not.  shutdown doesn't want us too
        # TODO: inform monitors?
    }
}

##################################################
sub __timeout
{
    my($kernel, $heap, $req_id)=@_[KERNEL, HEAP, ARG0];

    my $req=$heap->{request}{$req_id};
    if($req) {
        DEBUG and warn "$req_id: Request timed out\n";
        $kernel->yield('response', $req_id, "request timed out");
        $kernel->post($req->{session}, '_timeout', $req_id);
    }
    else {
        DEBUG and warn "$req_id: Time out for unknown request";
    }
}

##################################################
sub response
{
    my($kernel, $heap, $req_id, @resp)=@_[KERNEL, HEAP, ARG0..$#_];

    # delete the timeout
    $kernel->delay("__timeout_$req_id");
    $kernel->state("__timeout_$req_id");

    my $req=delete $heap->{request}{$req_id};
    if($req) {
        DEBUG and warn "$req_id: Sending response to $req->{rsvp}\n";
        # post a response to RSVP specifier
        # use call() so that shutdown() messages get out as soon as possible
        $kernel->call(IKC=>'post', $req->{rsvp}, @resp);
        # mess with GC
        $kernel->refcount_decrement($req->{session}, $req_id);
    }
    else {
        DEBUG and warn "$req_id: Response already sent!\n";
    }
}

1;

__END__

=head1 NAME

JAAS::Application - Application 

=head1 SYNOPSIS

  use JAAS::Application;

=head1 DESCRIPTION

A JAAS application is one or many entry points.  This module deals with
interfacing IKC with those entry points and publishing them.  It also
handles timeouts, so that if a request takes too long, at least *something*
is posted back to the requester.


=head1 CONFIGURATION

    <Application name>
        Alias other-name
        Uses object-name
        Uses other-object
        # EntryPoint definitions
    </Application>

=head2 Application

The name is a POE session alias used for IKC.

    poe://KernelName/name/

=head2 Alias

Another kernel name for IKC.

    poe://KernelName/other-name/

=head2 Uses

Name of objects that must be will be requested from the context for each
request.

=head2 EntryPoint

Note that due to a bug in Config::General, you can't have list and named
section EntryPoints as described in JAAS::Entrypoint.  I'm going to have to
reimplement Config::General.





=head1 METHODS

=head2 spawn

    JAAS::Application->create(name=>$name, aliases=>$aliases, 
                              objects=>$uses, entry_points=>\%things);

Creates the application session, any entry point sessions.  Publishes the
entry points via IKC.

=over 4

=item name

Name of this application.  This is the main name/alias as published via IKC.

=item aliases

Arrayref of aliases for this applicaiton.  These are also published via IKC.

=item objects

Arrayref of object names that each request will need by default.

=item entry_points

Hashref of data to be used for creating entry points.  This is how
the config file (via JAAS::Server) "talks" to JAAS::Entrypoint.

=back



=head1 STATES

=head2 response

This state sends the response back to the requester, and clears the timeout. 
Obviously, this means you can't have an entry point named "response". 
Sorry.


=head1 AUTHOR

Philip Gwyn, <jaas at pied.nu>

=head1 SEE ALSO

perl(1).

=cut




$Log: Application.pm,v $
Revision 1.9  2002/06/07 08:30:25  fil
Save ID between _start and _stop, because of something strange in POE.
Action can now have an additional number at the end of it.  SMELLS TODO
Rearranged get/set field a bit, so that more warnings happen

Revision 1.8  2001/10/27 03:11:59  fil
Fiddled warnings a bit.
Added support for more params to IKC::Server... still more todo, though

Revision 1.7  2001/10/11 02:30:15  fil
Fixed "prev" problem
Fixed many other problems
I hope I've got it right.

Revision 1.6  2001/09/18 22:16:07  fil
Added Multipart, which almost works.  Needs doco
Moved tests around a bit.

Revision 1.5  2001/08/07 00:00:19  mou
Typo in __timeout

Revision 1.4  2001/08/03 07:14:14  fil
Server/name now propagates to Entrypoint

Revision 1.3  2001/08/01 05:36:30  fil
Added documentation!
First pass at keepalive.
Muddled with how entry/respond promotes things to a hashref.

Revision 1.2  2001/07/27 21:37:37  fil
Beat into better shape.  Tests pass!

Revision 1.1  2001/07/26 06:40:52  fil
Wow!  It works... sorta :/
