package JAAS::Entrypoint;

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);
use Carp;
use Data::Dumper;
use Data::Denter;
use Digest::MD5 qw(md5_hex);
use JAAS::Object;
use JAAS::Object::Services;
use JAAS::Widget::Hidden;
use POE::Session;
use Socket;

$VERSION = '0.03';
BEGIN {
    *DEBUG=sub () {$jaas_services->debug()};
}

###############################################################################
sub spawn
{
    my $package=shift @_;
    my %heap=@_;

    croak "NO NAME" unless $heap{name};
    croak "NO OBJECTS" unless $heap{objects};
    DEBUG and warn "$package: spawn\n";
    # DEBUG and warn join ', ', sort keys %heap;
    my $session=POE::Session->create(
        package_states=>[
            $package=>[qw(_start entry _stop shutdown
                        get_PID find_action 
                        build_ctx objects_back get_widgets
                        validate_input capture_input
                        do_action noitca_od
                        respond persist build_widgets built_widgets cleanup
                        app_error populate_these
                      )],
        ],
        heap=>{%heap},
    );
    return $session->ID;
}

##################################################
sub _start
{
    my($package, $kernel, $session, $heap)=@_[OBJECT, KERNEL, SESSION, HEAP];

#    $heap->{ctx}||="poe://".$kernel->ID."/Context";
    $heap->{ctx}||='Context';
#    $session->option('trace', 1);
    DEBUG and 
        warn "$package: $heap->{name} (", $session->ID, ") is coming\n";
    
    # Because we are being lazy, we create empty JAAS::Objects before-hand
    foreach (@{$heap->{objects}}) {
        push @{$heap->{sessions}}, JAAS::Object->spawn();
    }
    $heap->{ID}=$session->ID;
}


##################################################
sub _stop
{
    my($package, $session, $heap)=@_[OBJECT, SESSION, HEAP];
    my $n=$heap->{ID};
    DEBUG and 
        warn "$package: $n is going away\n";
    $jaas_relationships->remove($n);
}

##################################################
sub shutdown
{
    my($package, $kernel, $heap)=@_[OBJECT, KERNEL, HEAP];
    DEBUG and
        warn "$package: $heap->{name} shutdown\n";

    # tell all the objects to go away
    foreach my $ses (@{$heap->{sessions}}) {
        $kernel->post($ses=>'_shutdown');
    }
}

##################################################
sub entry
{
    my($kernel, $sender, $heap, $req_id, @args)=
                        @_[KERNEL, SENDER, HEAP, ARG0..$#_];


    $heap->{RID}=$req_id;
    DEBUG and 
        warn "\n****************************************\n";
    DEBUG and warn "$heap->{RID}: entry\n";
    DEBUG and warn "With args ", Dumper \@args;

    $heap->{$req_id}=bless {  
                        input=>\@args, RID=>$req_id,
                        uses=>[@{$heap->{objects}}],
                        name=>$heap->{name},
                        keepalive=>0,           # TODO
                        sender=>$sender->ID,
                        ctx=>$heap->{ctx},
                        CTX=>{entrypoint=>$heap->{name}, 
                              application=>$heap->{application},
                              server=>$heap->{server}
                             },
                        main=>$heap->{main}
                     }, 'JAAS::Entrypoint::Duffus';

    $kernel->yield('get_PID');
    return;
}



##################################################
sub get_PID
{
    my($heap, $kernel)=@_[HEAP, KERNEL];


    DEBUG and warn "$heap->{RID}: get_PID\n";

    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='get_PID';
    my $input=$req->{input}[0];
    my $OOB=$input->{OOB};

    my $ip=0;
    foreach my $q ($OOB->{IP}, $OOB->{Remote}{IP}) {
        next unless $q;
        $ip=sprintf '%08X', unpack "N", inet_aton $q;
        last;
    }

    my $id=$input->{PID} || $OOB->{PID};
    my($pid, $tip, $time, $idled);
    if($id and $id =~ /([a-fA-F0-9]+)-([a-fA-F0-9]+)-([a-fA-F0-9]+)/) {
        ($pid,$tip,$time)=($1,$2,$3);
        $time=hex "0x$time";
        # TODO: change idle timeout
        if($time + 12*60*60 < time) {           # make sure PID didn't expire
            # DEBUG and 
                warn "PID idled";
            $idled=$pid;
            undef $pid;
        }
        if($tip ne $ip) {                       # make sure IP is valid
            # DEBUG and 
                warn "EMPTY IP changed $tip vs $ip";
            # 2002-04: Clearing PID on new IP is safer privacy wise
            # However! it fails the "least suprise principal"
            # undef $pid;

        }
    }

    unless($pid) {
        DEBUG and 
            warn "new PID  ($$, ", time, ", $ip, $req->{RID}, $req)";
        $pid=md5_hex($$, time, $ip, $req->{RID}, $req);
        warn "$$: $idled idled, is now $pid" if $idled;
        $req->{newPID}=1;
    }
    DEBUG and
        warn "$$: 1 PID=$pid\n";
    $req->{CTXID}=$pid;
    $req->{PID}=sprintf "$pid-$ip-%X", time;

    $req->{CTX}{PID}=$pid;

    $kernel->yield('find_action');
}

##################################################
sub find_action
{
    my($heap, $kernel)=@_[HEAP, KERNEL];

    DEBUG and warn "$heap->{RID}: find_action\n";

    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='find_action';

    my $input=$req->{input}[0];
    $req->{default_response}=$req->{action}=$heap->{name};
    foreach my $k (keys %$input) {
        # warn "Is $k an action?";
        next unless $k=~ /^(?:SUBMIT|ACTION)[._](\w+?)(?:[._](\d+))?$/;
        $req->{action}=$1;
        if(defined $2) {
            $req->{args}=[$2];
            # DEBUG and 
                warn "Extra argument: $2";
        }
        
        $req->{default_response}='response';
        last;
    }

    $req->{CTX}{action}=$req->{action};
    DEBUG and 
        warn "$heap->{RID}: action is $req->{action}";

    if($req->{keepalive}==2) {
        $kernel->yield('get_widgets') 
    } 
    else {
        $kernel->yield('build_ctx');
    }
}

##################################################
sub build_ctx
{
    my($package, $heap, $kernel)=@_[OBJECT, HEAP, KERNEL];
    DEBUG and warn "$heap->{RID}: build_ctx\n";

    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='build_ctx';

    if($req->{uses}) {
        $req->post_ctx('setup');
        $kernel->yield('objects_back');
    } else {
        $kernel->yield('do_action');
    }
}



##################################################
sub objects_back
{
    my($kernel, $session, $heap, $obj)=@_[KERNEL, SESSION, HEAP, ARG0];
    DEBUG and warn "$heap->{RID}: objects_back\n";
    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='objects_back';

    unless(defined $obj) {      # first call
        # go get the first object on the list
        DEBUG and warn "$req->{RID}: fetching object $req->{uses}[0]\n";
        $req->post_ctx('get', {obj=>$req->{uses}[0]}, "objects_back");
    }
    elsif($obj==0) {               # object locked
        $kernel->delay('objects_back', 1);
        # TODO: put this object at end of {uses} array, so we can
        # TODO: ask for other objects while waiting
    }
    elsif($obj==-1) {
        # Error -1 means no context or can't create or something bad like that
        # Hmm... maybe a better way to handle this?
        
        $kernel->yield('respond', 
                        {error=>[text=>"Can't get object $req->{uses}[0] from context."]});
    }
    elsif($obj<0) {
        $kernel->yield('respond', 
                        {error=>[text=>"Error fetching saved objects: $obj"]});
    }
    else {
        my $name=shift @{$req->{uses}};
        # here is where we have to associate the object and it's
        # object-layer session
        # Once again, we use pre-created sessions
        my $jaas=$heap->{sessions}[0+@{$req->{uses}}];
        $kernel->post($jaas=>'_attach', $obj);

        # monitor errors in this object
        # this gets turned off in cleanup
        $jaas_relationships->nail($session->ID, 'on_error', $obj->name,
                                    {to=>{object=>$session->ID,
                                          method=>'app_error'
                                         }   
                                    });

        # and save for persistance and putback
        $req->{objects}{$name}=$obj;

        if(@{$req->{uses}}) {       # get next object in list
            splice @_, ARG0;        # knock object off @_
            shift(@_)->objects_back(@_);
        }
        else {                      # otherwise get widgets
            $kernel->yield('get_widgets');
        }
    }
}

##################################################
sub get_widgets
{
    my($kernel, $heap, $widgets)=@_[KERNEL, HEAP, ARG0];
    DEBUG and warn "$heap->{RID}: get_widgets\n";
    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='get_widgets';

    unless(defined $widgets) { # first time in get_widgets() this invocation

        # We in FACT want to name each new set of widgets uniquely ...
        my $w=$req->{widget_name};
        my $input=$req->{input}[0];
        $w||=$input->{__widgets__};
        if(ref $w) {            # gah!  assume it's an ARRAY
            warn "__widgets__ was $w!  yerche!";
            $w=$w->[-1];
        }

        if($w) {
            DEBUG and 
                warn "$heap->{RID}: Getting widgets.$w\n";
            $req->post_ctx('get', {obj=>'widgets.'.$w, no_create=>1}, 
                                    "get_widgets");
            return;
        } else {
            $widgets=-1;
        }
    }
    if($widgets==0) {            # locked
        $kernel->delay('get_widgets', 1);
    }
    elsif($widgets == -1 ) {   # at this point, means no widgets
        # no widgets means we aren't in a forced march... i think
        DEBUG and 
            warn "$heap->{RID}: no widgets\n";
        $req->{args}=delete $req->{input};
        $kernel->yield('do_action');

    }
    elsif($widgets < 0 ) {    # some sort of error :(
        $kernel->yield(respond=> {
                            error=>[text=>"Fetching widgets: $widgets"]});

    }
    else {                    # got them... validate it all
        $kernel->yield('validate_input', $widgets);
    }
}

##################################################
# now check all the input stuff against the widgets
sub validate_input
{
    my($self, $kernel, $heap, $widgets)=@_[OBJECT, KERNEL, HEAP, ARG0];

    DEBUG and 
        warn "$heap->{RID}: validate_input\n";
    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='validate_input';

    my @error=eval{$self->widgets_validate($req, $widgets)};
    push @error, 'ERROR'=>$@ if $@;

    if(@error) {
        $kernel->yield('respond', {error=>\@error, widgets=>$widgets});
    } else {
        $kernel->yield('capture_input', $widgets);
    }
}

##################################################
sub capture_input
{
    my($self, $kernel, $heap, $widgets)=@_[OBJECT, KERNEL, HEAP, ARG0];
    DEBUG and warn "$heap->{RID}: capture_input\n";

    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='capture_input';
    eval {$self->widgets_capture($req, $widgets)};
    if($@) {
        $kernel->yield('app_error', $@);
        return;
    }

    $kernel->yield('do_action');
}

##################################################
sub do_action
{
    my($package, $kernel, $session, $heap, $action)=
                        @_[OBJECT, KERNEL, SESSION, HEAP, ARG0];
    DEBUG and warn "$heap->{RID}: do_action\n";

    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='do_action';

    $action||=delete $req->{action};
    unless($action) {
        $kernel->yield('respond', 
                        {error=>[text=>"NO ACTION"]});
        return;
    }

    DEBUG and 
        warn "$heap->{RID}: doing action $action\@$req->{main}\n";
    DEBUG and 
        warn "With arguments $heap->{RID}: ", Dumper $req->{args};
    $req->{CTX}{entry_step}='do_action';

    ######################
    # fake out a JAAS call :/
    $jaas_services->exec_ctx($req->{CTX});
    $jaas_services->_POE([@_[0..(ARG0-1)]]);
    $jaas_services->post(
                {
                    to=>{object=>$req->{main}, method=>$action},
                    "reply-to"=>{object=>$session->ID, method=>'noitca_od'},
                }, 
                @{$req->{args}}, $req->{RID});
    $jaas_services->_POE();
    $jaas_services->exec_ctx({});
    return;
}


##################################################
sub noitca_od
{
    my($kernel, $heap, $next)=@_[KERNEL, HEAP, ARG0];
    DEBUG and warn "$heap->{RID}: noitca_od\n";

    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='noitca_od';

    if($req->{error}) {
        # an error occured
        return;
    }

    DEBUG and 
        warn "$heap->{RID}: Response is ".
                            (defined($next) ? $next : 'undef()')."\n";

    unless($next) {                             # no response
        $kernel->yield('build_widgets') if $req->{PID};   # get widgets
        # If there was an error, we don't want to build_widgets, eh
    }
    elsif(ref $next) {                          # object or other
                                                # is the answer
        $kernel->yield('respond', $next);
    }
    elsif($next eq '_wait') {
        DEBUG and warn "Waiting for response";
    } 
    else {                                    # another step
        DEBUG and warn "Chaining to $next";
#        $req->{default_response}='response';
        $kernel->yield('do_action', $next);
    }
    return;
}

##################################################
sub build_widgets
{
    my($self, $kernel, $session, $heap, $p)=
                @_[OBJECT, KERNEL, SESSION, HEAP, ARG0];
    DEBUG and 
            warn "$heap->{RID}: build_widgets\n";
    die "NO RID" unless $heap->{RID};
    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='build_widgets';

    if($req->{responded}) {
        DEBUG and warn "Someone wanted us to build widgets, but we've already responded\n";
        return;
    }
    
    my @args=($req->{RID});
    push @args, $p if defined $p;   # allow overloaders to pass more params

    # ask main object what widgets they are interested in
    $jaas_services->post(
                {  to=>{object=>$req->{main},
                        method=>'widgets', }, 
                    OOB=>1,
                    "reply-to"=>{object=>$session->ID, 
                                 method=>'built_widgets'}
                }, @args); 
    return;
}

##################################################
sub built_widgets
{
    my($self, $kernel, $heap, $widgets)=@_[OBJECT, KERNEL, HEAP, ARG0];
    DEBUG and 
        warn "$heap->{RID}: built_widgets\n";
    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='built_widgets';

    unless($widgets) {                              # no widgets
        if(defined $widgets) {                      # but this is desired...
            $kernel->yield(respond=>{});
        } else {                                    # but maybe not?
            $kernel->yield(respond=>{error=>[text=>"NO WIDGETS"]});
        }
        return;
    }
    eval {$self->widgets_populate($req, $widgets)};
    if($@) {
        $kernel->yield(app_error => $@);
        return;
    }

    # Uniquely name each form
    $req->{widget_name} ||= $req->{PID};

    # but only once... (ie, don't when 'prev' is pressed, say)
    my $already=0;
    foreach my $w (@$widgets) {
        next if 'ARRAY' eq ref $w;
        next unless $w->isa('JAAS::Widget::Hidden') and
                    $w->{name} eq '__widgets__';
        $already=1;
        last;
    }
    push @$widgets, JAAS::Widget::Hidden->new(name=>'__widgets__', 
                                              content=>$req->{widget_name})
                unless $already;

    # HTML::Mason will handle turning widgets into HTML
    $kernel->yield('respond', {widgets=>$widgets});
}

##################################################
sub respond
{
    my($kernel, $heap, $what)=@_[KERNEL, HEAP, ARG0];
    DEBUG and 
        warn "$heap->{RID}: respond\n";
    unless($heap->{RID}) {
        warn "Response came way to early\n", Dumper $what;
        return;
    }
    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='respond';

    # munge the response back into shape
    unless($what) {
        $what={error=>[error=>'noResponse']};
    } 
    elsif('HASH' ne ref $what) {
        my $n=$req->{default_response}||'response';
        $what={$n=>$what};
    }

    # for cookies and what-not
    $what->{PID}=$req->{PID};

    # main bit of code.  Sends info back to requester
    $kernel->call($req->{sender}=>'response', $heap->{RID}, $what);

    $req->{responded}=1;

    # no persist on error, unless it's forced
    if($what->{force_persist} or not $what->{error}) {
        $kernel->yield('persist', $what->{widgets});
    } 
    elsif(not $req->{keepalive}) {
        $kernel->yield('cleanup');
    }

    if(DEBUG and $what->{error} and $what->{error}[0] eq 'text') {
        warn "$req->{RID}: $what->{error}[1]\n";
    }
}


##################################################
sub persist
{
    my($self, $kernel, $heap, $widgets)=@_[OBJECT, KERNEL, HEAP, ARG0];
    DEBUG and warn "$heap->{RID}: persist\n";
    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='persist';

    # we in want to name each new set of widgets uniquely
    my $w=$req->{widget_name};
    if($w and $widgets) {
        DEBUG and 
            warn "$heap->{RID}: Saving widgets.$w\n";
        my $dups=$self->widgets_persist($widgets);

        $req->post_ctx('putback', {obj=>'widgets.'.$w, in=>$dups, new=>1})
    }
    elsif(not $w and $widgets) {
        DEBUG and warn "$heap->{RID}: We have widgets, but no name for them\n";
    }

    while(my($name,$obj)=each %{$req->{objects}}) {
        DEBUG and 
            warn "$heap->{RID}: Saving $name\n";
        $req->post_ctx('putback', {obj=>$name, in=>$obj});
    }

    $kernel->yield('cleanup') unless $req->{keepalive};
}

##################################################
sub cleanup
{
    my($kernel, $heap)=@_[KERNEL, HEAP];
    DEBUG and 
        warn "$heap->{RID}: cleanup\n";
    return unless $heap->{RID};
    my $req=$heap->{$heap->{RID}};
    return unless $req;
    $req->{CTX}{entry_step}='cleanup';

    # no longer interested in CTX... this causes persistance to happen I hope
    $req->post_ctx('clear');

    foreach my $obj (values %{$req->{objects}}) {
        $jaas_relationships->remove($obj->name);
        $kernel->post($obj->name =>'_detach');
    }

    # house keeping chores
    delete $req->{objects};
    DEBUG and warn "$heap->{RID}: DONE\n\n";
    delete $heap->{delete $heap->{RID}};
}



###########################################################################
sub app_error
{
    my($kernel, $session, $heap, $error)=
                    @_[KERNEL, SESSION, HEAP, ARG0];
    DEBUG and warn "$heap->{RID}: app_error\n";
    DEBUG and warn Dumper $error;
    unless($heap->{RID}) {
        warn "Error is too late ", Dumper $error;
        return;
    }
    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='app_error';

    $req->{error}=1;
    $kernel->yield('respond', {error=>[jaas=>$error]});
}








###########################################################################
# Code for dealing with widgets

##################################################
# Recursively call "populate" on a bunch of widgets
sub widgets_populate
{
    my($self, $req, $widgets)=@_;

    my $something=new JAAS::Entrypoint::_Objects $req->{objects};

    foreach my $g (@$widgets) {
        die "Widgets aren't a reference: $g" unless ref $g;
        if(ref $g eq 'ARRAY') {
            foreach my $w (@$g) {
                $w->populate($something);
            }
        }
        else {
            $g->populate($something);
        }
    }
}

##################################################
# Save the widgets to the current context
sub widgets_persist
{
    my($self, $widgets)=@_;
    
    my $out=[];
    foreach my $g (@$widgets) {
        die "Widgets aren't a reference: $g" unless ref $g;
        if(ref $g eq 'ARRAY') {
            push @$out, [];
            foreach my $w (@$g) {
                if($w->can('persist_save')) {
                    $w=$w->persist_save();
                }
                push @{$out->[-1]}, $w;
            }
        }
        else {
            $g=$g->persist_save() if $g->can('persist_save');
            push @$out, $g;
        }
    }
    return $out;
}

##################################################
# Ask all widgets to validate input
sub widgets_validate
{
    my($self, $req, $widgets)=@_;
    my @error;

    my $input=new JAAS::Entrypoint::_Input $req->{input};

    # Validation returns things like
    # 'invalid'=>'field', 'missing=>'otherfield'.  But that's what
    # HTML::Mason has to decode
    foreach my $g (@$widgets) {
        die "Widgets aren't a reference: $g" unless ref $g;
        if(ref $g eq 'ARRAY') {
            foreach my $w (@$g) {
                DEBUG and warn ref($w), "->validate\n";
                push @error, $w->validate($input);
            }
        } 
        else {
            DEBUG and warn ref($g), "->validate\n";
            push @error, $g->validate($input);
        }
    }
    return @error;
}

##################################################
# Ask all widgets to send input data to it's source
sub widgets_capture
{
    my($self, $req, $widgets)=@_;

    my $something=new JAAS::Entrypoint::_Objects $req->{objects};
    foreach my $g (@$widgets) {
        die "Widgets aren't a reference: $g" unless ref $g;
        if(ref $g eq 'ARRAY') {
            foreach my $w (@$g) {
                DEBUG and warn ref($w), "->capture\n";
                $w->capture($something);
            }
        }
        else {
            DEBUG and warn ref($g), "->caputre\n";
            $g->capture($something);
        }
    }
}

##################################################
# Grah!  Wish this didn't have to exist
# Dastardly event that allows external objects or sessions to get their
# widgets populated.  It needs to be call()ed, or noOOB=>1
# so that JAAS 'reply-to' works...
sub populate_these
{
    my($self, $heap, $widgets)=@_[OBJECT, HEAP, ARG0];
    my $req=$heap->{$heap->{RID}};

    $self->widgets_populate($req, $widgets);
    return $widgets;
}








###########################################################################
package JAAS::Entrypoint::Duffus;
use strict;
use POE::Kernel;
use Carp;

BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () {0}; }
}


##################################################
sub post_ctx
{
    my($self, $state, $arg, $postback)=@_;
    $arg||={};
    if($arg and not ref $arg) {
        $arg={obj=>$arg};
    }
    if('HASH' eq ref $arg) {
        $arg->{id}=$self->{CTXID};
        $arg->{name}=$self->{name};
    }

#    warn "$state $arg->{id}\n" if $state eq 'setup' or $state eq 'clear';

    unless($postback) {
        $poe_kernel->post($self->{ctx}, $state, $arg);
        return;
    }
    carp "IKC not supported for post_ctx" if $postback =~ /^poe:/;

    my $ret=$poe_kernel->call($self->{ctx}, $state, $arg);
    $poe_kernel->yield($postback, $ret);

    return;

    # TODO: full IKC support.
    # IKC was failing because not everthing was being delivered before
    # the shutdown
    my @params=(IKC=>'post', "$self->{ctx}/$state"=>$arg);
    if($postback) {
        $params[1]='call';
        $postback="poe:$postback" unless $postback =~ /^poe:/;
        push @params, $postback;
    }
    DEBUG and carp ref($self), ": $params[1]ing $params[2]\n";
    $poe_kernel->post(@params);
}












###############################################################################
package JAAS::Entrypoint::_Input;
use strict;
use Carp;
use Data::Dumper;

BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () {0}; }
}


##################################################
sub new
{
    return bless $_[1], $_[0];
}

##################################################
sub param
{
    my($self, $field)=@_;
    my $OOB=$self->[0]{OOB};
#    warn "$field ", (defined($self->[0]{$field}) ? $self->[0]{$field} : 'undef()');
    return $self->[0]{$field};
}














###############################################################################
package JAAS::Entrypoint::_Objects;
use strict;
use Carp;

BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () {0}; }
}


##################################################
sub new
{
    return bless $_[1], $_[0];
}

##################################################
sub get_field
{
    my($self, $source)=@_;
    confess "No source" unless $source;
    $source=~m(^(.+?)/(.+)$);
    my($obj, $field)=($1, $2);

    unless($self->{$obj}) {
        DEBUG and warn "Unknown object $obj wanted for $source";
        return;
    }
    unless($self->{$obj}->can('get_field')) {
        DEBUG and warn "get_field: $obj can't do get_field\n";
        return;
    }
    DEBUG and warn "get_field: $obj/$field\n";
    return $self->{$obj}->get_field($field);
}

##################################################
sub set_field
{
    my($self, $source, $value)=@_;
    unless($source=~m(^(.+?)/(.+)$)) {
        warn "$source isn't object/field\n";
        return;
    }
    my($obj, $field)=($1, $2);
    unless($self->{$obj}) {
        warn "Don't have object $obj\n";
        return
    }
    unless($self->{$obj}->can('set_field')) {
        warn "Object $obj can't set_field\n";
    }
    DEBUG and warn "set_field: $obj/$field => $value\n";
    return $self->{$obj}->set_field($field, $value);
}





1;
__END__
# Below is the stub of documentation for your module. You better edit it!

=head1 NAME

JAAS::Entrypoint - Application action entry point

=head1 SYNOPSIS

    use JAAS::Entrypoint;

    # most of the time, the entry point will be set up in 
    # JAAS::Server->spawn();  If not, then read the source.


=head1 DESCRIPTION

An entry point is how message/action/method gets to an application object. 
JAAS::Entrypoint deals with setting up the context (creating objects if
needed), widgets and so on for the request.  It also allocates an request ID
(RID) that is used for all communication w/in the server about a given
request.

=head1 CONFIGURATION


  <Application ...>
    EntryPoint name
    <EntryPoint name>
        package JAAS::Entrypoint
        <args>
            main object-name
            objects other-object
        </args>
    </EntryPoint>
  </Application>


=head2 EntryPoint name

Name of this entry point.  See C<spawn>.

    poe://KernelName/app/name

=head2 package

What package implements this entry point.  In the future, there will be
several flavours of entry points.  For instance,
JAAS::Multipart will implement a series of dialog boxes.

=head2 main

Name of the main business object.  See C<spawn>.





=head1 METHODS

=head2 spawn

    JAAS::Entrypoint->spawn(main=>$main_object, name=>$name, 
                   objects=>[qw(object names used for this application)],
                    application=>$session_ID);

Creates a new POE session for this entry point.  This session will deal with
requests for the application.  Arguments are :

=over 4

=item name

Name of this entry point.  This is used as the default action (see
C<find_action>) and for the name reported to the context server.

=item main

Name of the object that methods are called on.  This object must be listed
in C<objects>.

=item application

ID or alias of the JAAS::Application session.  NOT USED.

=item objects

Names of all the objects this entry point will need for each request. 
Currently, JAAS::Entrypoing precreates the JAAS::Object sessions, but only
attaches objects when they are fetched from the context.

=back    

Returns the POE session ID.

















=head1 STATES

=head2 shutdown

Tells the entry point to go away.  This currently consists of closing down
all the JAAS::Object sessions.

=head2 entry

This is what JAAS::Application posts to when it hands the request to the
entry point.  The request is cut up into smaller steps, as follows:

=head2 get_PID

Looks in input for a PID.  If one is found, it verifies that it comes
from the same IP and hasn't expired yet.  If a new PID is required, it will
create the new PID.

The PID is 3 hex numbers joined by an '-'.  The first one is an MD5 hash
created using various bits of information, in an attempt to be unguesssable. 
The second number is the IP of requester, as transmited via
$input->{OOB}{ip}.  The third is the time, in seconds since the epoch, when
the PID was last used.  

The MD5 hash is used to fetch/store objects in the context.

Jumps to C<find_action>.

=head2 find_action

Searches through the input looking for keys of the form "SUBMIT.something",
"SUBMIT_something", "ACTION.something" or "ACTION_something".  If found, the
"something" becomes the current action/method.  If none is found, the action
defaults to the name of the entry point.

Jumps to C<build_ctx>.


=head2 build_ctx

Asks the context to build all the objects this entrypoint uses.  See
C<JAAS::Context> for details about this.  When the objects are returned from
the context server, it attaches them to JAAS::Object sessions.

Jumps to C<get_widgets>.


=head2 get_widgets

Fetches any saved widgets from the context.  Currently, this is just an
objected named "saved.widgets".  Ideally, we need to keep a backlog of
widgets, so that if the user backs up and resubmits, we can revalidate the
input.  Maybe this should go in JAAS::Forcedmarch.

If no widgets where saved, jumps straight to C<do_action>, other wise jumps
to C<validate_input>.  


=head2 validate_input

Loops through the widgets and gets them to validate the input.  They
should return an array, that looks like a key/value pair:

    'missing', 'something', 'invalid', 'otherthing', 'invalid, 'yetmore'.

See C<JAAS::Widgets> for more details.

If the input was OK, goes on to C<capture_input>.  If not, responds with the
errors using C<respond>.

=head2 capture_input

Tells the widgets to put the values back to the given objects.  See
C<JAAS::Widgets>.

Jumps to C<do_action>.


=head2 do_action

This is where the action is!  It posts to the main object of the entry point
(either the first Uses, or the main arg to spawn).  The method is the action
found previously (C<find_action>).  The arguments are whatever
JAAS::Application gave us, with the RID tacked on the end.  The response
from this method decides how execution will proceed:

=over 4

=item Nothing

Jumps to C<build_widgets>.

=item '_wait'

Hangs around, letting the object deal with calling respond.  Note that
JAAS::Application will time out eventually.

=item scalar

Loops and posts to the method so named.

=item Reference  

Assumes that this is the answer and posts it back, via C<respond>.

=back


=head2 build_widgets

Asks all the objects to create any necessary widget objects (see
C<JAAS::Widgets>) that it will save for input validation and then send to
the requester.

Jumps to C<respond>, with an array of widgets in {widgets}.



=head2 respond

    $kernel->post($entry=>'respond', $object);

Send a response back to requester.  If C<$object> is not an a hash
reference, it is prompted to one. If the response isn't an error, or if you
set "force_persist" in the response object, starts the persistance process
(C<persist>).

Note that the business objects should post to this method if they need to.

=head2 persist

Puts all the object back into the context server.

  Then it jumps to C<cleanup>.

=head2 cleanup

Clears our interest in the context (which provokes persistance), then
detaches all the objects from the application layer.



=head1 TODO

A foreign kernel is going to want to post more then one request per
connection, so we should try to optimise the cleaning up and so on for each
request.


=head1 AUTHOR

Philip Gwyn, jaas at pied.nu

=head1 SEE ALSO

perl(1).

=cut


$Log: Entrypoint.pm,v $
Revision 1.37  2002/06/07 08:30:25  fil
Save ID between _start and _stop, because of something strange in POE.
Action can now have an additional number at the end of it.  SMELLS TODO
Rearranged get/set field a bit, so that more warnings happen

Revision 1.36  2002/04/17 19:58:04  fil
Fibble

Revision 1.35  2002/04/17 19:53:23  fil
Grrr...

Revision 1.34  2002/04/17 19:03:50  fil
Small debug

Revision 1.33  2002/04/15 15:14:32  fil
More debuging.... looking for emtpy cart bug

Revision 1.32  2002/04/15 15:10:28  fil
Added debuging stuff

Revision 1.31  2001/10/27 03:11:59  fil
Fiddled warnings a bit.
Added support for more params to IKC::Server... still more todo, though

Revision 1.30  2001/10/19 05:55:32  fil
Better checks on source when doing widget capture.

Revision 1.29  2001/10/12 21:46:29  fil
Reactivated IP changes detection.

Revision 1.28  2001/10/12 19:20:51  fil
No longer changes PID on IP change... for debuging

Revision 1.27  2001/10/12 19:19:24  fil
Fibble.

Revision 1.26  2001/10/12 04:18:52  fil
Removed leetle debug message

Revision 1.25  2001/10/12 03:01:07  fil
Deactivated IP checking.

Revision 1.24  2001/10/12 01:10:16  fil
Uses Denter

Revision 1.23  2001/10/11 02:30:15  fil
Fixed "prev" problem
Fixed many other problems
I hope I've got it right.

Revision 1.22  2001/09/27 04:30:23  fil
Moved widget stuff into methods and wrapped it up in eval {}
When saving widgets, we ask them if they really want to be saved, so that
    CC {content} could be ignored.

Revision 1.21  2001/09/27 02:00:04  fil
Minor fiddling

Revision 1.20  2001/09/26 04:40:33  fil
Allow widgets to return NO widgets (defined but false)

Revision 1.19  2001/09/26 02:20:52  fil
Script can now tell Multipart that it's DONE.  This causes an extra
kludged on _post to happen.

Revision 1.18  2001/09/25 20:56:26  mou
Typo squashed

Revision 1.17  2001/09/25 20:12:14  fil
Fixed widget handeling.  Widgets now have a unique ID.

Revision 1.16  2001/09/21 18:49:40  fil
Fixed Multipart / Entrypoint interaction.
Fixed script fetching for Multipart.
Interaction w/ CTX no longer goes through IKC.  This needs to be corrected
at some point.

Revision 1.15  2001/09/20 01:11:10  fil
Widget fetching now uses no_crete.

Revision 1.14  2001/09/20 01:05:07  fil
Cleanup debug messages, fixed the cancel operation.

Revision 1.13  2001/09/19 08:24:25  fil
Blow me

Revision 1.12  2001/09/19 07:52:33  fil
Fixed Entrypoint so that it works with the new 'reply-to' stuff

Revision 1.11  2001/09/18 22:16:07  fil
Added Multipart, which almost works.  Needs doco
Moved tests around a bit.

Revision 1.10  2001/08/22 02:25:04  mou
Fixed an endless loop in noitca_od

Revision 1.9  2001/08/14 05:53:18  fil
Hack so that widget population works for the cart

Revision 1.8  2001/08/11 19:36:02  fil
Tweeked IP handling

Revision 1.7  2001/08/09 16:24:48  fil
Improved handling of more then one response.  IE, an error during cleanup, say.

Revision 1.6  2001/08/09 03:42:25  fil
Widget handling actually works now :)

Revision 1.5  2001/08/03 07:14:14  fil
Server/name now propagates to Entrypoint

Revision 1.4  2001/08/02 04:40:29  fil
Hadn't really tested the context stuff.  Fixed

Revision 1.3  2001/08/01 05:36:30  fil
Added documentation!
First pass at keepalive.
Muddled with how entry/respond promotes things to a hashref.

c