# $Id: Multipart.pm,v 1.10 2002/04/15 15:10:28 fil Exp $
package JAAS::Multipart;
use strict;

use POE::Session;
use JAAS::Entrypoint;
use JAAS::Object::Services;
use JAAS::Widget::Group::Action;

use Data::Dumper;

use vars qw(@ISA $VERSION);
@ISA=qw(JAAS::Entrypoint);

$VERSION = '0.03';
BEGIN {
    *DEBUG=sub () {$jaas_services->debug()};
}

##############################################################################
sub spawn
{
    my $package=shift @_;
    my %heap=@_;

    $heap{default_script} ||= ['start'];
    $heap{default_script} = [$heap{default_script}] 
                unless ref $heap{default_script};

    $package->SUPER::spawn(%heap);
}




##################################################
sub _start
{
    my($object, $kernel, $heap, $script)=@_[OBJECT, KERNEL, HEAP];

    shift(@_)->SUPER::_start(@_);

    $kernel->state("script_add", $object);
    $kernel->state("script_get", $object);
    $kernel->state("position_get", $object);
    $kernel->state("get_script", $object);
    $kernel->state("script_done", $object);
}

##################################################
sub find_action
{
    my($heap)=@_[HEAP, KERNEL];
    shift(@_)->SUPER::find_action(@_);
    my $req=$heap->{$heap->{RID}};

    DEBUG and warn "$req->{RID}: Action is $req->{action}\n";
    $req->{action}='' if $req->{action} eq $heap->{name};
    return;
}

##################################################
sub build_ctx
{
    my($kernel, $heap, $script)=@_[KERNEL, HEAP, ARG0];

    DEBUG and warn "$heap->{RID}: build_ctx (multipart)\n";
    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='build_ctx';
    $req->post_ctx('setup');
    $kernel->yield('get_script');
}

##################################################
# load the script and current step from context server
sub get_script
{
    my($self, $kernel, $heap, $script)=@_[OBJECT, KERNEL, HEAP, ARG0];

    DEBUG and warn "$heap->{RID}: get_script (multipart)\n";
    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='get_script';
    
    unless($req->{action}) {            # first step
        $script=-1;                     # pretend no script in context
        DEBUG and warn "$req->{RID}: No action, first step\n";
    }

    unless(defined $script) {           # first time through
        DEBUG and 
            warn "$req->{RID}: Fetching the script";
        $req->post_ctx('get', {obj=>'script', no_create=>1}, "get_script");
    } 
    elsif($script==0) {                 # locked
        $kernel->delay('build_ctx', 1);
    }
    elsif($script < -1 ) {              # some sort of error :(
        $kernel->yield(resond=>{error=>[text=>"Fetching script: $script"]});
    }
    else {                     
        if($script != -1) {             # Either we got what we want...
            $self->script_restore($req, $script);
        }
        else {                          # or this is the first step
            $self->script_start($req, $heap->{default_script});
        }

        # We NEVER allow Entrypoint to do build_ctx()!  
        # Because we don't want to call 'setup/Context' twice.
        # This is going to be bad if we refactor Entripoint::build_ctx to do
        # more work.  Another work around would be to call 'clear' twice also
        $kernel->yield('objects_back');
    }
}



##################################################
sub get_widgets
{
    my($self, $kernel, $heap, $widgets)=@_[OBJECT, KERNEL, HEAP, ARG0];


    DEBUG and warn "$heap->{RID}: get_widgets (multipart)\n";
    my $req=$heap->{$heap->{RID}};

    if(($req->{action}||'') eq 'cancel') {      # skip-out early
        $req->{canceled}=1;
        $kernel->yield('do_action', 'cancel');
        return;
    }


    # When a user presses 'prev', we don't want to load the widgets that
    # are defined in __widgets__.  We also don't want to validate them.
    # Note that no step_pre is called.  However, step_post will be called
    # after the user presses 'next'
    if($req->{action} eq 'prev') {
        unless(defined($widgets)) {                 # first call
            $self->script_prev($req);               # back up a step

            # load widgets for current (ie previous) step
            $req->{widget_name}=$req->{script}[$req->{script_position}];
        }
        elsif(ref $widgets) {                       # respond w/ those widgets
            $kernel->yield(built_widgets => $widgets);
            return;
        }
    }

    shift(@_)->SUPER::get_widgets(@_);
}


##################################################
sub do_action
{
    my($self, $kernel, $heap, $action)=@_[OBJECT, KERNEL, HEAP, ARG0];

    # if we are called with an action, we are already inside our "loop"
    return shift(@_)->SUPER::do_action(@_) if $action;

    DEBUG and warn "$heap->{RID}: do_action (multipart)\n";
    my $req=$heap->{$heap->{RID}};

    $req->{todo}=[];
    $req->{done}={};
    
    if(($req->{action}||'') eq 'next') {        # 'next' button
        $self->script_todo($req, '_post');      # do this step's step_post
        $req->{do_next}=1;                      # get noitca_od to do it's stuff
    }

    $kernel->yield('noitca_od');        # let noitca_od loop on TODO
}

##################################################
sub noitca_od
{
    my($self, $kernel, $heap, $next)=@_[OBJECT, KERNEL, HEAP, ARG0];

    DEBUG and warn "$heap->{RID}: noitca_od (multipart)\n";
    my $req=$heap->{$heap->{RID}};

    if(delete $req->{todo_post2}) {  # A step_post that comes after 'respond'
        $kernel->yield('persist');
        return;
    }

    if($req->{action} eq 'cancel') {        # 'cancel' action is sorta
                                            # like a regular action
        return shift(@_)->SUPER::noitca_od(@_);
    }

    # sanity check
    if($next) {
        $kernel->yield(app_error=>"NO!  Don't give me a next='$next'");
        return;
    }

    if(@{$req->{todo}}) {                   # loop on our TODO list
        my $name=shift @{$req->{todo}};     # pull one off
        $kernel->yield(do_action => $name); # do it
        return;
    } 

    # Move on to the next step.  This goes here, not in do_action(), because
    # if 'step_post' calls script_add(), {script_position} might be messed up
    # script_next() makes sure that it only happens once per invocation
    if(($req->{action}||'') eq 'next') {        # 'next' button
        $self->script_next($req);
    }

    if((($req->{action}||'next') eq 'next')     # first time, or 'next'
                                # allow _post methods to append to the script
                or delete $req->{check_script}) { 

        if($self->script_todo($req, '_pre')) { # check for another _pre...
            $kernel->yield('noitca_od');   # and make sure it's executed
            return;
        }
    }

    $kernel->yield('build_widgets'); # then we build the widgets for this step
}



##################################################
sub build_widgets
{
    my($self, $heap)=@_[OBJECT, HEAP];

    DEBUG and 
        warn "$heap->{RID}: build_widgets (multipart)\n";
    my $req=$heap->{$heap->{RID}};

    # save widgets as #
    unless($req->{script_done}) {
        # TODO: should this include PID?
        $req->{widget_name}=$req->{script}[$req->{script_position}];
    } else {
        # TODO: script_done, why use default widget_name?
    }

    $_[ARG0] ||= $req->{script}[$req->{script_position}];
    shift(@_);
    $self->SUPER::build_widgets(@_);
}

##################################################
sub built_widgets
{
    my($self, $kernel, $heap, $widgets)=@_[OBJECT, KERNEL, HEAP, ARG0];
    DEBUG and 
        warn "$heap->{RID}: built_widgets (multipart)\n";
    my $req=$heap->{$heap->{RID}};
    $req->{CTX}{entry_step}='built_widgets';

    my $has_actions=$self->widgets_has_actions($widgets);

    # add the << prev, cancel and next >> buttons
    if($widgets and not $req->{canceled} and not $req->{script_done}
                and not $has_actions) {
        my @actions=();
        my $n=0+@{$req->{script}};

        if($n==1 or $req->{script_position} < $n) {
            #  { local $"=", "; warn "Script=@{$req->{script}}"; }
            push @actions, "cancel";
            if($req->{script_position} > 0) { # position already incremented
                DEBUG and warn "$req->{script_position} > 0 (prev)";
                push @actions, "prev";
            }
            DEBUG and 
                warn "$req->{script_position} < $n (next)";
            push @actions, "next";
        }
        elsif(DEBUG) {
            warn "$req->{RID}: DONE! $n < script_position=$req->{script_position}\n";
        }
        push @$widgets, JAAS::Widget::Group::Action->new(\@actions)
                        if @actions;
    } elsif(DEBUG) {
        $widgets||='';
        $req->{canceled}||=0;
        $req->{script_done}||=0;
        $has_actions||=0;
        warn "$req->{RID}: widgets=$widgets canceled=$req->{canceled} script_done=$req->{script_done} has_actions=$has_actions\n"
    }
    shift(@_)->SUPER::built_widgets(@_);
}

##################################################
sub persist
{
    my($self, $kernel, $heap)=@_[OBJECT, KERNEL, HEAP];

    DEBUG and warn "$heap->{RID}: persist (multipart)\n";
    my $req=$heap->{$heap->{RID}};

    # Here we insert a _post after respond, but before persist
    if($req->{todo_post}) {
        $kernel->yield(do_action => delete $req->{todo_post});
        # this tells noitca_od to jump to persist
        $req->{todo_post2}=1;
        return;
    }

    # save script and current step and stuff
    # we save no matter what, even if the script is finished or canceled
    $req->post_ctx('putback', {obj=>'script', new=>1, 
                                in=>$self->script_save($req)});
    shift(@_)->SUPER::persist(@_);
}

##################################################
sub widgets_has_actions
{
    my($self, $widgets)=@_;
    use Carp;
    use Data::Dumper;
    croak Dumper $widgets unless 'ARRAY' eq ref $widgets;
    foreach my $g (@$widgets) {
        unless(ref $g) {
            $POE::Kernel::poe_kernel->yield(app_error=>
                                            "Widgets aren't a reference: $g");
            return 0;
        }
        if(ref $g eq 'ARRAY') {
            return 1 if $self->widgets_has_actions($g);
        }
        else {
            return 1 if $g->isa('JAAS::Widget::Group::Action');
        }
    }
    return 0;
}











##############################################################################
sub script_add
{
    my($kernel, $heap, $new)=@_[KERNEL, HEAP, ARG0];

    my $req=$heap->{$heap->{RID}};

    $new=[$new] unless ref $new;

    my $script=$req->{script};
    # truncate at current position
    foreach my $step (splice @$script, 1+$req->{script_position}) {
        DEBUG and 
            warn "$req->{RID}: Removing widgets.$step\n";
        $req->post_ctx('remove', {obj=>'widgets.'.$step});
    }
    push @$script, @$new;               # add new script

    if($req->{script_done}) {           # we thought we were done
        $req->{script_done}=(0==@$new); # but maybe maybe we aren't
        # note : only script_next() should set 'script_done'
    }

    local $"=", ";
    DEBUG and 
        warn "$req->{RID}: Script is now: @$script\n";

}

##################################################
# For information purposes, I suppose
sub script_get
{
    my($kernel, $heap)=@_[KERNEL, HEAP];

    my $req=$heap->{$heap->{RID}};
    return [@{$req->{script}}];
}

##################################################
sub position_get
{
    my($kernel, $heap)=@_[KERNEL, HEAP];

    my $req=$heap->{$heap->{RID}};
    return $req->{script}[$req->{script_position}];
}

##################################################
sub script_done
{
    my($self, $kernel, $heap, $done)=@_[OBJECT, KERNEL, HEAP, ARG0];

    $done=1 if not defined $done;
    my $req=$heap->{$heap->{RID}};
    $req->{script_done}=$done;
    if($done) {
        # if the step_pre called done(), then we might do a step_post as well
        my $name=$self->script_todo($req, '_post', 0);
        DEBUG and warn "Script is done, one last $name\n" if $name;
        $req->{todo_post}=$name if $name;
    }
    return;
}






##################################################
sub script_start
{
    my($self, $req, $default)=@_;
    
    $req->{script}=[@{$default}];
    $req->{script_position}=0;
    $req->{script_done}=0;
    $req->{script_step}={name=>'', action=>''};

    local $"=", ";
    DEBUG and 
        warn "$req->{RID}: Starting: script=@{$req->{script}}, position=$req->{script_position}\n";
}

##################################################
sub script_save
{
    my($self, $req)=@_;
    local $"=", ";
    DEBUG and 
        warn "$req->{RID}: Saving: script=@{$req->{script}}, position=$req->{script_position}\n";
    return { script=>[@{$req->{script}}], 
             script_done=>$req->{script_done},
             script_step=>$req->{script_step},
             script_position=>$req->{script_position}}
}

##################################################
sub script_restore
{
    my($self, $req, $script)=@_;

    $req->{script}=$script->{script};
    $req->{script_done}=$script->{script_done};
    $req->{script_step}=$script->{script_step};
    $req->{script_position}=$script->{script_position};
    local $"=", ";
    DEBUG and 
        warn "$req->{RID}: Restore: script=@{$req->{script}}, position=$req->{script_position}\n";
}


##################################################
sub script_next
{
    my($self, $req)=@_;
    return if $req->{script_done};
    return if $req->{script_already_next}++;

    # warn "UNCOMMENT NEXT LINE!";
    $req->{script_position}++;          


    if($req->{script_position} > $#{$req->{script}}) {
        DEBUG and 
            warn "$req->{RID}: Script done!\n";
        $req->{script_done}=1;
    } else {
        $req->{script_done}=0;

        DEBUG and 
            warn "$req->{RID}: Moving on: position=$req->{script_position}\n";
        DEBUG and 
            warn "$req->{RID}: Step is called '$req->{script}[$req->{script_position}]'\n";
    }
}

##################################################
sub script_todo
{
    my($self, $req, $suffix, $add)=@_;
    $add=1 unless @_==4;
    my $name=$req->{script}[$req->{script_position}];
    my $todo=($name||'').$suffix;

    if($name and not $req->{done}{$todo}) {
        $req->{done}{$todo}=1;
        if($add) {
            DEBUG and 
                warn "$req->{RID}: Adding ==== $todo ==== to TODO list\n";
            push @{$req->{todo}}, $todo;
        } else {
            DEBUG and 
                warn "$req->{RID}: ==== $todo ==== is allowed to happen\n";
        }
        return $todo;
    }
    return;
}

##################################################
sub script_prev
{
    my($self, $req)=@_;
    return if $req->{script_already_prev}++;

    $req->post_ctx('remove',            # remove previous widgets
                    {obj=>'widgets.'.$req->{script}[$req->{script_position}]});
    # blech, someone could back up to that particular form...


    $req->{script_position}--;
    $req->{script_done}=0;              # can't be done now 

    DEBUG and 
        warn "$req->{RID}: Backing up: position=$req->{script_position}\n";

    if($req->{script_position} < 0) {
        $POE::Kernel::poe_kernel->yield(app_error=>
                      "Script position is negative: $req->{script_position}");
    }
    DEBUG and 
        warn "$req->{RID}: Step is called '$req->{script}[$req->{script_position}]'\n";

}




1;

__END__

=head1 NAME

JAAS::Multipart - Application action entry point for a multi-part form

=head1 SYNOPSIS

    use JAAS::Multipart;

    # most of the time, the entry point will be set up in 
    # JAAS::Server->spawn();  If not, then read the source.


=head1 DESCRIPTION

This is an entry point for a multi-part form.  An example of a multi-part is
the "software wizards" that some programs have.  This is a term I dislike,
so I use "multipart form" or "forced march".

A multi-part form is a form that's devided into multiple page or parts or
steps.  A list of steps is called the form's "script".  JAAS::Multipart
deals with moving backwards and forwards in the steps, saving useful
information as it goes.  Each step has a "name".  If no script is
configured, a script consisting of the step 'start' is used.  Steps can be
added or removed at run time, allowing you to skip steps.  Each step is
split in three : step_pre method, response generation then a step_post
method.  step_pre is called when we enter the step, step_post is called when
the user has responed to this steps response.


Here's an attempt at showing what's going on

    step: start (first step)
        start_pre()
        response send
        (user clicks next)
        input validated and captured
        start_post
    move to next step (foo)
        foo_pre()
        response send
        (user clicks next)
        input validated and captured
        foo_post
    move to next step (billy)
        billy_pre()
        response send
        (user clicks prev)
    move to prev step (foo)
        reponse from step foo is sent
        (user clicks next)
        input validated and captured
        foo_post
    move to next step (billy)
            
And so on.  It is sort of implicit that the response takes the form of a
bunch of widgets (see JAAS::Entrypoint) but it doesn't really have to be.




=head1 CONFIGURATION


  <Application ...>
    <EntryPoint name>
        package JAAS::Multipart
        <args>
            main object-name
            default_script start
            default_script foo
            default_script billy
        </args>
    </EntryPoint>
  </Application>


=head2 EntryPoint name

Name of this entry point.  See C<spawn>.

    poe://KernelName/app/name

=head2 package

What package implements this entry point.  In the future, there will be
several flavours of entry points.  

=head2 main

Name of the main business object.  See C<spawn>.

=head2 default_script

String or array of strings that lists the steps that must be followed in
this multipart form.



=head1 METHODS

=head2 spawn

    JAAS::Entrypoint->spawn(default_script=>[...], ... );

Creates a new POE session for this entry point.  See C<JAAS::Entrypoint>
Additional arguments are :

=over 4

=item default_script

Array reference or scalar that contains the script when the user first
proceeds down the twisty road.  Can be changed while in progress.






=head1 STATES

JAAS::Multipart inheirits all the states from JAAS::Entrypoint, with some
additions and modifications :


=head2 get_action

Only 4 actions make sense to JAAS::Multipart: empty (''), next, prev,
cancel.  If the action is the same as the name of the form, this is a
special case of empty.

=head2 script_add

Appends an arrayref of steps to the script.

=head2 script_get

Returns the full list of steps in the script.

=head2 position_get
=head2 get_script
=head2 script_done


=head1 TODO

A foreign kernel is going to want to post more then one request per
connection, so we should try to optimise the cleaning up and so on for each
request.


=head1 AUTHOR

Philip Gwyn, jaas at pied.nu

=head1 SEE ALSO

perl(1).

=cut










$Log: Multipart.pm,v $
Revision 1.10  2002/04/15 15:10:28  fil
Added debuging stuff

Revision 1.9  2001/10/27 03:11:59  fil
Fiddled warnings a bit.
Added support for more params to IKC::Server... still more todo, though

Revision 1.8  2001/10/11 02:30:15  fil
Fixed "prev" problem
Fixed many other problems
I hope I've got it right.

Revision 1.7  2001/09/27 02:00:04  fil
Minor fiddling

Revision 1.6  2001/09/26 05:20:03  fil
Multipart handles named widgets better.

Revision 1.5  2001/09/26 02:20:52  fil
Script can now tell Multipart that it's DONE.  This causes an extra
kludged on _post to happen.

Revision 1.4  2001/09/25 20:12:14  fil
Fixed widget handeling.  Widgets now have a unique ID.

Revision 1.3  2001/09/21 18:49:40  fil
Fixed Multipart / Entrypoint interaction.
Fixed script fetching for Multipart.
Interaction w/ CTX no longer goes through IKC.  This needs to be corrected
at some point.

Revision 1.2  2001/09/20 01:05:07  fil
Cleanup debug messages, fixed the cancel operation.

Revision 1.1  2001/09/18 22:16:07  fil
Added Multipart, which almost works.  Needs doco
Moved tests around a bit.

