package JAAS::Server;

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

use POE::Component::IKC::Server;
use POE::Component::IKC::Responder;
use JAAS::Config;
use JAAS::Object::Context;
use JAAS::Factory;
use JAAS::Application;
use Carp;

$VERSION = '0.03';
my $ID="JAAS000";

BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () {1}; }
}

###############################################################################
sub spawn
{
    my($package, $config)=@_;
    croak "NO CONFIG" unless $config;

    unless(ref $config) {
        DEBUG and warn "Promoting $config to object\n";
        $config=JAAS::Config->new($config);
        die $@ unless $config;
        DEBUG and warn "Spawning factory\n";
        my $factory=JAAS::Factory->new($config);
        my $q=$config->for('Cache');
        my $cache;
        if($q) {
            DEBUG and warn "Creating context cache\n";
            $cache=$factory->make('Cache');
        }

        DEBUG and warn "Spawning context\n";
        JAAS::Object::Context->spawn('Context', $factory, $cache);
        DEBUG and warn "Spawning services\n";
        JAAS::Object::Services->spawn($config);

    }

    POE::Component::IKC::Responder->spawn();

    $package->create_servers($config);
    $package->create_applications($config);
}


##################################################
# Build a IKC servers from the config
sub create_servers
{
    my($package, $config)=@_;

    my $server=$config->get("Server");
    my($name, @listen, $aliases);

    # get an array of everywhere we are going to listen
    if($server) {
        my $ls=$package->sane_array(delete $server->{listen});
        foreach my $l (@$ls) {
            if($l =~ m/unix:(.+)$/) {               # unix:/path/to/socket
                push @listen, {unix=>$1};
                DEBUG and warn "$package: server on $listen[-1]{unix}\n"
            }
            elsif($l=~ m/(\w+)?:(\d+)/) {           # ip:port
                push @listen, {port=>$2};
                $listen[-1]{ip}=$1 if $1;
                DEBUG and warn "$package: server on $listen[-1]{ip}:$listen[-1]{port}\n"
            }
    
            else {                                  # ip (or nada)
                push @listen, {ip=>$l, port=>20000+$<};
                DEBUG and warn "$package: server on $listen[-1]{ip}:$listen[-1]{port}\n"
            }
        }

        $aliases=$package->sane_array(delete $server->{name});
        $name=shift @$aliases;
    } 
    else {
        DEBUG and warn "No servers\n";

    }

    # have to have a name and listen to SOMETHING
    $name||=$ID++;
    @listen=({port=>20000+$<}) unless @listen;
    DEBUG and warn "$package: Creating ", 0+@listen, " IKC servers\n";

    foreach my $l (@listen) {                       # spawn IKC Server
        POE::Component::IKC::Server->spawn(
                            name=>$name, aliases=>$aliases, %$l, %$server);
    }
}


##################################################
sub create_applications
{
    my($package, $config)=@_;

    my $server_name=$package->sane_scalar($config->get("Server/name"));
    $server_name||='JAECA';

    DEBUG and warn "$package: Creating applications for $server_name:\n";
    my $apps=$config->get("Application");
    return unless $apps;
    $apps=$package->sane_hash($apps);
    while(my($name, $def)=each %$apps) {
        $package->create_application($server_name, $name, $def);
    }
}

##################################################
sub create_application
{
    my($package, $server_name, $name, $def)=@_;

    DEBUG and warn "$package: Creating $name\n";
    my $uses    = $package->sane_array($def->{Uses});
    my $entries = $package->sane_array($def->{EntryPoint});
    my $aliases = $package->sane_array($def->{Alias});
    my $main    = $def->{Main}||$uses->[0];

    # prepare entry-point config for the application
    my %EPs;
    my $points=$package->sane_hash($def->{EntryPoint});
    foreach my $name (@$entries) {
        my $ep=$package->sane_hash($points->{$name});
        if($ep) {
            $EPs{$name}=$ep->{args};
            $EPs{$name}{package}=$ep->{package}||$ep->{class};
        }    
        $EPs{$name}{package}||= 'JAAS::Entrypoint';
        $EPs{$name}{main}   ||= $main;
        $EPs{$name}{server} ||= $server_name;

        # JAAS::Application::create_entries is going to do something like
        # the following during JAAS::Application::_start
        # $EPs{$name}{package}->spawn(name=>$name, objects=>$uses, 
        #                             session=>$app_session, %{$EPs{$name}});
    }

    # spawn the application session
    JAAS::Application->spawn(name=>$name, aliases=>$aliases, 
                             objects=>$uses, entry_points=>\%EPs);
}


##################################################
sub sane_array
{
    my($package, $data)=@_;
    return [] unless defined $data;
    $data=[$data] unless ref $data;
    $data=[keys %$data] if 'HASH' eq ref $data;
    return $data;
}

##################################################
sub sane_hash
{
    my($package, $data)=@_;
    return {} unless defined $data;
    $data={$data=>1} unless ref $data;
    if('ARRAY' eq ref $data) {
#        carp "Odd number of elements in hash" if @$data & 1;
        $data={map {$_=>1} @$data};
    }
    return $data;
}

##################################################
sub sane_scalar
{
    my($package, $data)=@_;
    return ''           unless defined $data;
    return $data        unless ref $data;
    return $data->[0]   if 'ARRAY' eq ref $data;
    return $data->{(keys %$data)[0]};       # any old value
}


1;

__END__
# Below is the stub of documentation for your module. You better edit it!

=head1 NAME

JAAS::Server - JAAS server object

=head1 SYNOPSIS

    use JAAS::Server;

    JAAS::Server->spawn('some.conf');

    # or create a JAAS::Context, JAAS::Factory and start
    # JAAS::Context and JAAS::Services as needed

    JAAS::Server->spawn($config);


=head1 DESCRIPTION

A server is a collection of applications.  Each application has one or more
entry points.  A server listens at various places for IKC requests, then
routes them to the appropriate.




=head1 CONFIGURATION

    <Server>
        name=KernelName
        name=KernelAlias
        listen host:port
        listen unix:/path/to/socket
    </Server>


    <Application name>
        # see JAAS::Application and JAAS::Entrypoint for details
    </Application>

Server blocks aren't named.

=head2 name

List of whatever names and/or aliases you want the kernel to be known as at
the IKC layer.

    poe://KernelName/

=head2 listen

Tells the server where you want IKC listening so that it can receive requests 
or something.



=head1 METHODS

=head2 spawn

    JAAS::Server->spawn('some.conf');
    JAAS::Server->spawn($config);


Spawn digs through the configuration, pulling out the bits and pieces that
are relevant.  It creates all required applications and their entry points. 
See C<JAAS::Entrypoing> and C<JAAS::Application> for details.


=head1 AUTHOR

Philip Gwyn, jaas at pied.nu

=head1 SEE ALSO

perl(1).

=cut

$Log: Server.pm,v $
Revision 1.8  2002/06/07 08:30:25  fil
Save ID between _start and _stop, because of something strange in POE.
Action can now have an additional number at the end of it.  SMELLS TODO
Rearranged get/set field a bit, so that more warnings happen

Revision 1.7  2001/10/27 03:11:59  fil
Fiddled warnings a bit.
Added support for more params to IKC::Server... still more todo, though

Revision 1.6  2001/08/09 16:24:48  fil
Improved handling of more then one response.  IE, an error during cleanup, say.

Revision 1.5  2001/08/09 03:42:25  fil
Widget handling actually works now :)

Revision 1.4  2001/08/03 07:14:14  fil
Server/name now propagates to Entrypoint

Revision 1.3  2001/08/01 05:36:30  fil
Added documentation!
First pass at keepalive.
Muddled with how entry/respond promotes things to a hashref.
