package JAAS::Widget::List;

use strict;

##################################################
sub new {
	my $class = shift;
	my %parm = @_;
	my $self = {
		name => '',
		title => '',
		source => '',
		validation => '',
		type => '',
		content => [],
        static => 0,
        straight => 0,
        orientation => 'horizontal',
	};
	bless $self, $class;
	$self->set(%parm);
    $self->{title} ||= $self->{name};
	return $self;
}

#################################################
# Accessors

sub populate {
	my ($self, $obj) = @_;
	my $data = $obj->get_field($self->{source});
	$self->check($data);
}

sub capture {
	my ($self, $obj) = @_;
    return if $self->{static};
	my @result;
	my $item_ary = $self->{item_names};
	my $i = 0;
	foreach my $item (@$item_ary) {
		push @result, $item if $self->{content}->[$i++];
	}		
	if (@result == 1) {
		$obj->set_field($self->{source}, $result[0]);
	}
	else {
		$obj->set_field($self->{source}, \@result);
	}
}

sub set {
	my ($self, %parm) = @_;
	foreach my $key (keys %parm) {
		next if $key eq 'tags';
		if ($key eq 'items') {
			my $item_ary = $parm{items};
			$item_ary = [$item_ary] if !ref($item_ary);
			my (@items, %tags);
			foreach my $i (@$item_ary) {		
				my ($item, $tag) = split /:/, $i, 2;
				push @items, $item;
				$tags{$item}=$tag if $tag;
			}
			$self->{item_names} = \@items;
			$self->{item_tags} = \%tags;		
		}
		else {
	 	 	$self->{$key} = $parm{$key} if exists $self->{$key};
		}
 	}
}

sub get {
	my $self = shift;
	my %result;
	foreach my $key (keys %$self) {
		$result{$key} = $self->{$key};
	}
	return %result;
}

sub to_mail {
	my ($self, $i18n) = @_;
	my $q = 0;
	my @text=($i18n->get($self->{title}, $self->{bits}).":");
	foreach my $n (@{$self->{item_names}||[]}) {
		if ($self->{content}->[$q++]) {
            push @text, $i18n->get($self->{title}.'.'.$n, $self->{bits});
		}
	}
    my $j=($self->{type} eq 'single' or @text==2) ? ' ' : "\n";
	return join $j, @text;
}

#############################################
# $data is an arrayref (promoted as needed from scalar)
# we create {content} which is an arrayref of flags, indexes are the same
# as those in {item_names}
sub check {
	my ($self, $data) = @_;
    $self->{content}=[];
    return unless $data;
    $data=[$data] unless ref $data; 
    my %check;
    @check{@$data}=@$data;

	foreach my $item (@{$self->{item_names}}) {
        push @{$self->{content}}, (exists($check{$item}) ? 1 : 0);
        delete $check{$item};
	}
    return unless keys %check;
    warn "Attempting to set content of ", ($self->{title}||$self->{name}), 
        " to unknown values ", join ', ',keys %check;
}

#############################################
# Validation
sub validate {
	my ($self, $obj) = @_;
    return if $self->{static};

	$self->check($obj->param($self->{name}));

 	my $invalid;
	my $val_ary = $self->{validation}||[];
	$val_ary = [$val_ary] if !ref($val_ary);
 	foreach my $elem (@$val_ary) {
		my ($validator, $parm) = split /:/, $elem, 2;
        next unless $self->can($validator);
        $invalid = $self->$validator($parm);
    	return ($invalid, $self->{title}) if $invalid;
 	}
    return;
}

sub required {
    my($self, $count)=@_;
	my $cont_ary = $self->{content};

    # count number of non-blank entries
	my $i = 0;
	foreach my $item (@$cont_ary) {
		$i++ if $item;
	}		
	if ($self->{type} eq 'single') {
        $count=1;
	}
    $count ||= '1+';

    # note that $i will never be negative
	if ($count =~ /\+/) {
	    chop $count;
		return $i==0 ? 'REQUIRED' : 'REQUIRED_MORE' if $i < $count;
	}
    elsif ($count =~ /\-/) {
        my ($min, $max) = split /-/, $count;
		return $i==0 ? 'REQUIRED' : 'REQUIRED_MORE' if $i < $min;
		return 'REQUIRED_LESS'                      if $i > $max;
	} 
    else {
        return 'REQUIRED_LESS'                      if $i > $count;
        return $i==0 ? 'REQUIRED' : 'REQUIRED_MORE' if $i < $count;
    }
    return;

    # REQUIRED == there are none, give us some
    # REQUIRED_MORE == there are some, but we still need more
    # REQUIRED_LESS == there are some, but to many
}

1;

=head1 NAME

JAAS::Widget::List - List-type widget object

=head1 SYNOPSIS

	use JAAS::Widget::List;

        my $w = JAAS::Widget::List->new(name=>'lang',
                                title=>'Language',
                                source=>'right/here',
	                        validation=>['required'],
				type=>'single',
				items=>['fr:Francais', 'en:English', 'ru:Russki']);

	# Fill out the widget using data from an object
        $w->populate($something);

	# Set widget parameters
	$w->set(title=>'Langage' items=>['fr:Francais', 'es:Espanol']);

	# Fill widget with 'checked' values
	$w->check('fr');

	# Get widget parameters
        my %parm = $w->get();
	print "$parm{title}: $parm{content}\n";

        # Validate data
	my $val = $w->validate($obj);

	# Give widget data back to the object
        $w->capture($something);

=head1 DESCRIPTION

This widget is used for lists, like checkboxes, radio buttons, menu list,
etc. The widget has two kinds of text fields: the list title and the tags
before list items. The input data is as follows:

	- name: Unique field name
	- title: List title
	- source: Object that the widget is linked with in the JAAS
	        environnement. The form is 'name/field'. For example, a
	        'name' text widget could be linked to a 'user' object, 
	        therefore the source would be 'user/name'.
	- validation: A scalar or list of validating routines. See below.
	- type: Item checking style. 'single' permits only one selection,
		just like radio buttons. 'multi' permits multiple choices.
	- items: A scalar or arrayref of items, in a 'name:text' format.

On output (with the C<get()> method), all data is given back, except for
C<items> which is broken down in three lists: C<item_names>, C<item_tags>
and C<content>. Those lists follow each other, in the sense that the first
item in the list has its name in the first element of C<item_names>, its
text in the first element of C<item_tags> and its checked status in the
first element of C<content>. Checked status is 0 if unchecked, 1 if
checked.

=head1 METHODS

=head2 new
          
        $w = JAAS::Widget::L:ist->new(%parameters);

Constructor. The parameters can be any of the data members listed above.

=head2 populate

	$w->populate($obj);

Fills out the widget with data from the object C<$obj>, a JAAS object
wrapper. Data from the object should be an arrayref
of the form ['item1', 'item2', ...], listing each pre-checked item. A
scalar is OK if the object contains only one value.

=head2 capture

	$w->capture($obj);

Gives back the widget value to the object C<$obj>, a JAAS object wrapper.
Data form is same as C<populate()>.

=head2 set

	$w->set(%parameters);

Sets parameters, just like C<new()> but without having to re-initialize the
object over.

=head2 check

	$w->check($data);

Sets the 'checked' status of the widget. $data is either a scalar for a single
value or an arrayref for multiple values. Values are the names for each
item.

=head2 get

	%parameters = $w->get();

Gets all parameters listed in the description. 

=head2 validate

	$valid = $w->validate($obj);

Runs the validation routines, as listed in C<validation>, on the data from
the JAAS object wrapper.
Returns 0 if the data validates, otherwise returns a failiure message.

Here is a (partial) list of validator routines. When filling out the
C<validation> field in C<new()> or C<set()>, use the names of those 
routines. Currently, only the C<required> routine is implemented.

=over 4

=item required

Checks if widget contains data. With type 'single', checks if only one
element is selected. With type 'multi', takes as parameter the number of
selected elements to verify. The syntax is:

C<required:x> - Checks for exactly 'x' selections.

C<required:x+> - Checks for 'x' or more selections.

C<required:x-y> - Checks for nb. of selections between 'x' and 'y'.

Default is '1+' if used without arguments.

With 'single' lists, returns 'REQUIRED' if no selection has been made.
With 'multi' lists, returns 'REQUIRED_MORE' if not enough selections have
been made, 'REQUIRED_LESS' if too many have been made.

=back

=head1 AUTHOR

Philip Gwyn <jaas at awale.qc.ca>, 
Learry Gagn <mou-jaas at awale.qc.ca>.

=head1 SEE ALSO

perl(1).

=cut


