# $Id: Text.pm,v 1.24 2002/06/07 08:26:18 fil Exp $
package JAAS::Widget::Text;

use strict;
use JAAS::Widget::Text::PostalCode;

use vars qw($VERSION);
$VERSION="0.01";

BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () {0}; }
}

##################################################
sub new {
	my $class = shift;
	my %parm = @_;
	my $self = {
		name => '',
		title => '',
		source => '',
		validation => [],
		content => '',
		class => '',
        static => 0,
        error => 0,
        lines => 0,
        size => 0,
	};
	bless $self, $class;
	$self->set(%parm);
    $self->{validation}||=[];
    $self->{validation}=[$self->{validation}] unless ref $self->{validation};
    $self->{title} ||= $self->{name};
	return $self;
}

#################################################
# Accessors

sub populate {
	my ($self, $obj) = @_;
	$self->{content} = $obj->get_field($self->{source});
}

sub capture {
	my ($self, $obj) = @_;
    if($self->{static}) {
        warn $self->{name}, " is static, won't capture input!";
        return;
    }
    DEBUG and warn "Capturing $self->{source}, $self->{content}\n";
	$obj->set_field($self->{source}, $self->{content});
}

sub set {
	my ($self, %parm) = @_;
	foreach my $key (keys %parm) {
 	 	$self->{$key} = $parm{$key} if exists $self->{$key};
 	}
}

sub get {
	my $self = shift;
	my %result;
	foreach my $key (keys %$self) {
		$result{$key} = $self->{$key};
	}
	return %result;
}

#############################################
sub to_mail {
	my ($self, $i18n) = @_;
	my $cont;
	my $text = $i18n->get($self->{title}, $self->{bits});
	if ($self->{content}) {
		if ($self->{class} eq 'money') {
			$cont = $self->to_money($i18n);
		}
		else {
			$cont = $self->{content};
		}
		$text .= ": " if $text;
        $text .= $cont;
	}
	return $text;
}

#############################################
sub to_money {
	my ($self, $i18n) = @_;
	my $p=0+$self->{content};

	my $dol = int($p/100);
	my $cen = $p - $dol*100;
 
 	$cen="0$cen" if $cen < 1;	#Padding zeros
        $cen="0$cen" if $cen < 10;

	my $ret = $i18n->get('money', {d=>$dol, c=>$cen});

	return $ret;
}

#############################################
# Validation

sub validate {
	my ($self, $obj) = @_;

    delete $self->{error};
    return if $self->{static};

	$self->{content} = $obj->param($self->{name});
    if(ref $self->{content}) {                # lame way to avoid dup values
        warn "duplicate $self->{name} input elements, using first one\n";
        $self->{content}=$self->{content}[0];
    }
    DEBUG and warn "Validating $self->{content}\n";

 	foreach my $elem (@{$self->{validation} || []}) {
    	my ($validator, $parm) = split /:/, $elem, 2;
        unless($self->can($validator)) {
            warn "Unknown validation method: ", ref($self), "->$validator\n";
            next;
        }
        my @invalid = $self->$validator($parm);
        next unless @invalid;   

        push @invalid, $self->{title} if 1==@invalid;
        $self->{error}=1;
        return @invalid;
    }
#    $self->{content}=delete $self->{content};
	return;
}

sub required {
 	my $self = shift;
	return if defined $self->{content} and $self->{content} =~ /\S/;
 	return 'REQUIRED';
}

sub regex_no {
        my ($self, $regex) = @_;
	return 'REGEX_INVALID' unless eval {"" =~ /$regex/; 1};
 	return unless $self->{content} and $self->{content} =~ /\S/;
        return if ($self->{content} !~ /$regex/);
        DEBUG and warn "$self->{content} matched $regex when it shouldn't";
 	return 'REGEX';
}
		
sub regex {
	my ($self, $regex) = @_;
	return 'REGEX_INVALID' unless eval {"" =~ /$regex/; 1};
	return unless $self->{content} and $self->{content} =~ /\S/;
	return if ($self->{content} =~ /$regex/);
	DEBUG and warn "$self->{content} didn't match $regex";
	return 'REGEX';
}

sub phone {
	my $self = shift;
	my $digit = 0;
	return unless $self->{content} and $self->{content} =~ /\S/;
	return 'PHONE' unless ($self->{content} =~ /^[-\d ()]+$/);
	$digit++ while ($self->{content} =~ /\d/g);
        return if ($digit >= 10);     
	return 'REGCODE' if ($digit == 7);
	return 'PHONE';
}

sub integer {   
    my $self = shift;
    return unless $self->{content} and $self->{content} =~ /\S/;
    return if ($self->{content} =~ /^-?[0-9]+$/);
    return 'INTEGER';
}

sub float {
    my $self = shift;
    return unless $self->{content} and $self->{content} =~ /\S/;
    return if ($self->{content} =~ /^-?[0-9]+(\.[0-9]+)?$/);
    return 'FLOAT';
}

sub amount {
    my $self = shift;
    return unless $self->{content} and $self->{content} =~ /\S/;
    return 'AMOUNT' unless ($self->{content} =~ /^[0-9]+(\.\d\d)?$/);
	$self->{content} *= 100;
	return;	
}	

sub quantity {
	my $self = shift;
    return unless $self->{content} and $self->{content} =~ /\S/;
	$self->{content} ||= 0;		#defaults to 0
	$self->{content} =~ s/\.\d+$//;	#strip decimals
	return if ($self->{content} =~ /^\s*[0-9]+\s*$/);
	return 'QUANTITY'
}
	
sub strip {			
	my $self = shift;
	my $data = $self->{content};
    return unless defined $data;
	$data =~ tr/\n/ /;
	$data =~ s/^\s+//;
	$data =~ s/\s+$//;
	$data =~ s/\s{2,}/ /g;
	$self->{content} = $data;
	return;
}


1;

__END__

=head1 NAME

JAAS::Widget::Text - Text widget object

=head1 SYNOPSIS

	use JAAS::Widget::Text;

	my $w = JAAS::Widget::Text->new(name=>'cc',
                                title=>'Credit Card',
                                source=>'user/cc',
                                validation=>['required', 'cc']);

	# Fill out the widget using data from an object
	$w->populate($something);

	# Set widget parameters
	$w->set(title=>'Your Credit Card Number', content=>'1111222233334444');

	# Get widget parameters
	my %parm = $w->get();
	print "$parm{title}: $parm{content}\n";

	# Validate data
	my $val = $w->validate($item);
	print "Validate: @val\n";

	# Give widget data back to the object
	$w->capture($something);

=head1 DESCRIPTION

This widget is used for simple text capture, like name, address, credit card
number, etc. It's not much more than a data repository with validation
routines. The object has 5 data members:

	- name: Unique field name
	- title: Text that goes along with the widget content
	- source: Object that the widget is linked with in the JAAS
		environnement. The form is 'name/field'. For example, a
		'name' text widget could be linked to a 'user' object,
		therefore the source would be 'user/name'.
	- validation: A scalar or list of validating routines. See below.
	- content: Widget value.

=head1 METHODS

=head2 new
	
	$w = JAAS::Widget::Text->new(%parameters);

Constructor. The parameters can be any of the data members listed above.

=head2 populate

	$w->populate($obj);

Fills out the widget with data from the object C<$obj>, a wrapper for
objects in JAAS.

=head2 capture

	$w->capture($obj);

Gives back the widget value to the object C<$obj>, a wrapper for objects in
JAAS

=head2 set

	$w->set(%parameters);

Sets parameters, just like C<new()> but without having to re-initialize the
object over.

=head2 get

	%parameters = $w->get();

Gets all parameters listed in the description.


=head2 validate

	$valid = $w->validate($obj);

Runs the validation routines, as listed in C<validation>, on the data from
the JAAS object wrapper.
Returns 0 if the data validates, otherwise returns a failiure message.

When filling out the C<validation> field in C<new()> or C<set()>, 
use the names of those routines.

Check out the source for individual validation description, most are simple regex
matches. Special ones are:

=over 4

=item regex

The special validator 'regex' is provided for creating your own
matches. Its syntax is C<regex:pattern>, for example C<regex:^\d+$> to
validate on digits-only data.

=item regex_no

Same as regex, except it validates if there's no match (e.g. !~ instead of =~).

=item amount

Apart from doing a pattern match on a valid dollar and cents amount, it
multiplies the value by 100 so the float gets converted to an integer.

=item quantity

Validates on a positive integer. Data is set to 0 if empty. Decimals are
stripped. If you want to invalidate an empty field, configure the validation
array as C<['required', 'quantity']>. That way, a 'required' validation will
run first.

=item strip

Not a validator 'per se', but strips newlines, leading/trailing spaces and
extra spaces between words in the data. Always returns 0 (valid).

=back

=head1 AUTHOR

Philip Gwyn <jaas at awale.qc.ca>, 
Learry Gagn <mou-jaas at awale.qc.ca>.

=head1 SEE ALSO

JAAS::Widget::Text::CreditCard, perl(1).

=cut


$Log: Text.pm,v $
Revision 1.24  2002/06/07 08:26:18  fil
Bumped version
Added orientation,title to Widget::Group
Widget::Hidden can now be static :)
Widget::List item_tags is now a hashref name => tag
Widget::List checks that capture is on a value in the list of known values
Widget::Text can be multilined
Widget::Text won't capture info if static
Wiget::G::Action now has _title, a hashref of name => title pairs

Revision 1.23  2002/04/23 19:22:37  fil
eep eep!  {bits} wasn't getting to i18n->get() in to_mail.

Revision 1.22  2001/10/18 02:15:43  mou
Added 'regex_no' validator (validate if no match).

Revision 1.21  2001/10/12 04:25:40  fil
Added tags
