# $Id: Config.pm,v 1.8 2001/11/14 04:32:41 fil Exp $
package JAAS::Config;

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

use Carp;
use Config::General::Extended;

$VERSION = '0.02';
BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () {0}; }
}

###########################################################
sub new
{
    my($package, $filename)=@_;
    my $self=bless {config=>{}, delta=>{}}, $package;
    if($filename and not $self->load($filename)) {
        return;
    }
    return $self;
}

###########################################################
sub load
{
    my($self, $filename)=@_;

    $filename||=$self->{filename};

    my $new=eval {
        croak "No filename to load!" unless $filename;
        my $cge=Config::General::Extended->new(
                        -hash=>{},
                        -UseApacheInclude=>1,
                        # -LowerCaseNames=>1
                        );
        $cge->{configfile}=$filename;
        $cge->_open($filename);
        return $cge->_parse({}, $cge->{content});
    };
    return if $@;
    _merge($self->{config}, $new);
    $self->{filename}||=$filename;
    return 1 unless -r "$filename.delta";

    return $self->load("$filename.delta");
}

###########################################################
sub add
{
    my($self, $content)=@_;
    $content=[split "\n", $content] unless ref $content;

    my $new=eval {
        my $cge=Config::General::Extended->new(
                        -hash=>{},
                        -UseApacheInclude=>1,
                        # -LowerCaseNames=>1
                        );
        $cge->{configfile}="scalar";
        return $cge->_parse({}, $content);
    };
    return if $@;
    _merge($self->{config}, $new);
    return 1;
}

###########################################################
sub _merge
{
    my($old, $new)=@_;
    while(my($k, $v)=each %$new) {
        unless(exists $old->{$k}) {
            $old->{$k}=$v;
            next;
        }

        _merge_val($old, $k, $v);
    }
}
sub _merge_val
{
    my($old, $k, $v)=@_;

    my $ov=$old->{$k};
    my $conv=($ov ? (ref($ov)||'SCALAR') : 'UNDEF')."+".
             ($v ? (ref($v)||'SCALAR') : 'UNDEF');
    if($conv eq 'SCALAR+SCALAR') {         # redefine
        $old->{$k}=$v;
    } 
    elsif($conv eq 'SCALAR+HASH') {        # promote, add
        $old->{$k}=$v;
        $old->{$k}{$ov}=1 unless exists $v->{$ov};
    }
    elsif($conv eq 'SCALAR+ARRAY') {       # promote, push
        $old->{$k}=[$ov, @$v];
    }
    elsif($conv eq 'HASH+SCALAR') {        # add
        $ov->{$v}=1;
    }
    elsif($conv eq 'HASH+HASH') {          # merge
        _merge($ov, $v);
    } 
    elsif($conv eq 'HASH+ARRAY') {         # convert to array, push
        $old->{$k}=[$ov, @$v];
        @{$ov}{@$v}=(1) x @$v;
    }
    elsif($conv eq 'ARRAY+SCALAR') {       # push
        push @{$ov}, $v;
    }
    elsif($conv eq 'ARRAY+HASH') {         # push
        push @$ov, $v;
    }
    elsif($conv eq 'ARRAY+ARRAY') {        # push
        push @{$ov}, @$v;
    }
    else {
        carp "Don't know how to merge $conv for '$k'";
        $old->{$k}=$v;
    }
}

###########################################################
sub save
{
    my($self, $ext)=@_;
    $ext||=".delta";
    unless($self->{filename}) {
        $@="No file was loaded, can't save the delta";
        return;
    }
    $@='';
    eval {
        my $cge=Config::General::Extended->new(-hash=>$self->{delta});
        $cge->save($self->{filename}.$ext);
        return 1;
    } or return;
    $self->{saved}=1;
    return 1;
}

###########################################################
sub clear
{
    my($self)=@_;
    $self->{config}={};
    $self->{delta}={};
    delete $self->{filename};
    delete $self->{saved};
}

###########################################################
sub get
{
    my($self)=@_;
    my $cur=$self->{config};
    foreach my $q (1..($#_-1)) {            # ignore 0th and last one
        $cur=$cur->{$_[$q]};
        return unless $cur;
        if('HASH' ne ref $cur) {
            carp join('/', @_[1..$q]), " is not a hash ref";
            return;
        }
    }
    return $cur->{$_[-1]};
}

###########################################################
sub get_hash
{
    my $self=shift @_;
    my $data=$self->get(@_);
    return {} unless defined $data;
    $data={$data=>1} unless ref $data;
    $data={@$data} if 'ARRAY' eq ref $data;
    return $data;
}

###########################################################
sub get_array
{
    my $self=shift @_;
    my $data=$self->get(@_);
    return [] unless defined $data;
    $data=[$data] unless ref $data;
    $data=[keys %$data] if 'HASH' eq ref $data;
    return $data;
}

###########################################################
sub get_scalar
{
    my $self=shift @_;
    my $data=$self->get(@_);
    return unless defined $data;
    return $data unless ref $data;
    return $data->[0] if 'ARRAY' eq ref $data;
    my @q=keys %$data;
    return $data->{$q[0]};
}

###########################################################
sub for
{
    my($self, $name)=@_;
    return $self->{config}{Object}{$name}
                ||
            $self->{config}{Session}{$name};
}

###########################################################
sub set
{
    shift(@_)->_set(1, @_);
}

sub default
{
    shift(@_)->_set(0, @_);
}

###########################################################
sub _set
{
    my($self, $do_d)=@_;
    return unless @_ >= 4;


    my $cur=$self->{config}||={};
    my $delta=$self->{delta} if $do_d;
    foreach my $q (2..($#_-2)) {            # ignore 0th and last two
        $cur=$cur->{$_[$q]};
        return unless $cur;
        if('HASH' ne ref $cur) {
            carp join('/', @_[1..$q]), " is not a hash ref";
            return;
        }
        $delta=$delta->{$_[$q]}||={} if $do_d;
    }
    delete $self->{saved}        if $do_d;
    _merge($delta, {@_[-2, -1]}) if $do_d;
#    use Data::Dumper;
#    warn Dumper [$cur, {@_[-2, -1]}];
    _merge($cur, {@_[-2, -1]});
    return 1;
}

1;
__END__

=head1 NAME

JAAS::Config - JAAS config file loader

=head1 SYNOPSIS

    use JAAS::Config;

    my $config=JAAS::Config->new();
    $config=JAAS::Config->new("filename");
    $config->load("filename");
    $config->clear();

    my $hashref=$config->get("something");
    
    $hashref=$config->for('object name');
    # equiv to 
    $hashref=$config->get("object", "object name");

    $config->set("object", "object name", "args", some=>'thing');

    # save changes to an extra file
    $config->save();

=head1 DESCRIPTION

JAAS configuration files use an Apache-style format, thanks to
Config::General.  The config files will/can contain anything.  

The configuration is changable at run time, via C<set>.  JAAS::Config keeps
tabs of all changes made after a file is loaded.  These changes can then be
saved into a "delta" file via C<save>.  This file will then be reloaded when
the original config is loaded.  While multple files can be loaded, but only
the first one is considered "supreme", as it were, and the delta is
saved/loaded for that file.

Rough overview of file syntax.

    # this is a comment
    key value
    key=value           # 'key' is now a array
    something=multi \
            line \
            value
    <block>
        key=value       # 'key' is not the same key as previously
        <sub-block>
            more data
            goes here
        </sub-block>
    </block>
    # the next two are roughly equivalent, only one of the os is named
    # name1 and the other name2
    <o name1>               
        more values
    </o> 
    <o name2>
       more values
    </o>

    something <<HERE
    You can put a lot of data here
    plus newlines and so on.  And it will work.
    HERE


The preceeding is equivalent to the following perl structure

    {
        'key' => [ 'value', 'value' ],
        'block' => {
            'key' => 'value',
            'sub-block' => {
                'more' => 'data',
                'goes' => 'here'
            }
        },
        'something' => [
            'multi line value',
            'You can put a lot of data here
plus newlines and so on.  And it will work.'
        ],
        'o' => {
            'name1' => { 'more' => 'values' },
            'name2' => { 'more' => 'values' }
        }
    }

Of special importance (now) are <object> blocks </object>.  These are use to
run JAAS::Factory and have a special shortcut, C<for>.  Each object must
have a C<name> after it, so that JAAS::Factory can reference it.

    <object name>
        like so
    </object>

DON'T DO THE FOLLOWING:

    object name

This will cause things to break.

=head1 METHODS

=head2 new

    my $conf=JAAS::Config->new();
    $conf=JAAS::Config->new($filename);

Creates a new JAAS::Config object.  If C<$filename> is specified, it is
loaded and parsed.  If there was an error during parsing, C<new()> returns
undef and $@ is set to whatever error Config::General sees fit to die on.

=head2 load

    $conf->load($filename);

Load a configuration file and merge it's contents with the current
configuration data.  Much magic happens during the merge.  An attempt is
made to make it DWIM without suprises, but there are some gotchas, due
mostly to magic inside of Config::General.

If you DON'T want to merge, but would rather start afresh, call
C<clear()> first.

=head2 save

    $conf->save();
    $conf->save($extention);

Saves any changes to the configuration to a "delta" file.  The file has the
same name as the first loaded file with C<$extention> appended. 
C<$extention> defaults to '.delta'.  If you use a different extention,
C<load()> won't be able to find it, however.  This is here because some
platforms don't like multple extentions... if you know of one, please tell
me how to detect it via C<$^O> and what the default should be.

=head2 add

    $config->add($string);
    $config->add($array_of_lines);

This allows you to add more information to the config without using a file.
The parameter can be either an arrayref of lines orscalar (in which case it
is split on newlines to form an arrayref).  The configuration information
is merged into the current configuration.

=head2 clear

    $conf->clear();

Drops everything from the object.  All configuration data, all changes 
is lost.  Clears the "supreme" filename used by C<save()>.

=head2 get

    my $value=$conf->get(@path);

Returns a single value from the configuration.  C<@path> is all the names
to be looked up, recursively to get to the value.  If one of the names isn't a
hash, well you get a warning and it returns C<undef()>.

    # assuming the config file defined in the DESCRIPTION
    my $here=$conf->('block', 'sub-block', 'goes');
    # $here = 'here'

Note that the returned value can be a scalar, hashref or arrayref.  Use
C<get_hash> or C<get_array> if you want to force the type.

=head2 get_hash

    my $hashref=$conf->get_hash(@path);

Returns a single value from the configuration.  C<@path> is all the names
to be looked up, recursively to get to the value.  If one of the names isn't a
hash, well you get a warning and it returns an empty hashref.

Note that the returned value is ALWAYS a hashref, even if empty.

=head2 get_array

    my $arrayref=$conf->get_array(@path);

Returns a single value from the configuration.  C<@path> is all the names
to be looked up, recursively to get to the value.  If one of the names isn't a
array, well you get a warning and it returns an empty arrayref.

Note that the returned value is ALWAYS an arrayref, even if empty.

=head2 set

    $conf->set(@path, $value);

Sets a new value in the configuration.  C<@path> is described previously. 
Some magic is done to merge C<$value> into the destination.  I don't feel
like fully describing this magic here.

=head2 default

    $conf->default(@path, $value);

Sets a default value in the configuration.  This differs from C<set> in that
a delta is not saved.  This allows you to provide default values for things
in your code and not have them saved in a delta.

=head2 for

    my $c=$conf->for($name);

As mentioned earlier, C<for> is a special accessor used by JAAS::Factory. 
It returns the <object>block</object> named C<$name>.  That is, the
following two are equivalent:

    $c=$conf->for($name);
    $c=$conf->get('object', $name);

See L<JAAS::Factory> for a full description of what should be in the object
configuration.  Here is a small sample.

    <object name>
        class=My::Package  # or package=
        ctor=new           # constructor.  'new' and 'spawn' are default
        <args>
            hashref passed
            to ctor
        </args>
    </object>

=head1 AUTHOR

Philip Gwyn <perl at pied.nu>

=head1 SEE ALSO

Config::General, perl(1).

=cut


$Log: Config.pm,v $
Revision 1.8  2001/11/14 04:32:41  fil
Added ->add() method + tests for it.

Revision 1.7  2001/08/01 05:30:14  fil
Updated the doco.
Fixed the tests
Fixed manifying...  MakeMaker-- GRR GRR GRR

Revision 1.6  2001/07/27 21:38:21  fil
Small changes to help JAECA

Revision 1.5  2001/07/27 00:34:42  fil
Added default

Revision 1.4  2001/07/06 06:09:56  fil
Documented JAAS::Config::Session
Added JAAS::Config->{saved}, used for status.

Revision 1.3  2001/07/06 05:05:09  fil
Added new-fangled DEBUG stuff
Added JAAS::Config::Session (not documented...)

Revision 1.2  2001/07/06 02:11:46  fil
Added documentation to JAAS::Config!

Revision 1.1.1.1  2001/07/06 00:02:58  fil
JAAS config file and object factories.

