# $Id: Session.pm,v 1.4 2001/08/01 05:30:14 fil Exp $
package JAAS::Config::Session;

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

use POE::Session;

$VERSION = '0.01';
BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()} }
    else { *DEBUG=sub () {0} }
}

###########################################################
sub spawn
{
    my($package, $params)=@_;
    $package||=__PACKAGE__;
    my $heap={};
    $heap->{name}   = $params->{name}||"JAAS_Config";
    $heap->{config} = $params->{config}||JAAS::Config->new();

    POE::Session->create(
        package_states=>[$package=>[qw(_start get get_pb set 
                                        save load clear status)]],
        heap=>$heap,
    );
}

###########################################################
sub _start
{
    my($kernel, $heap)=@_[KERNEL, HEAP];
    $kernel->alias_set($heap->{name});
    # this could silently fail
    $kernel->post(IKC=>'publish', $heap->{name}=>[
                qw(get_pb set save load clear status)]); 
}

###########################################################
sub get
{
    my($heap, $path)=@_[HEAP, ARG0];
    $path=[split '/', $path] unless ref $path;
    DEBUG and warn "Fetching ", join '/', @$path;
    my $rv=$heap->{config}->get(@$path);
    DEBUG and warn "Value is $rv\n";
    return $rv;
}

###########################################################
sub get_pb
{
    my($kernel, $sender, $heap, $path, $postback)=
                            @_[KERNEL, SENDER, HEAP, ARG0, ARG1];

    $path=[split '/', $path] unless ref $path;
    DEBUG and warn "Fetching ", join('/', @$path), " for ", $sender->ID;
    my $rv=$heap->{config}->get(@$path);
    DEBUG and warn "Value is $rv\n";
    $kernel->post($sender, $postback, $rv);
    DEBUG and warn "Posted to ", $sender->ID, "/$postback";
}

###########################################################
sub set
{
    my($heap, $path)=@_[HEAP, ARG0];
    $path=[split '/', $path] unless ref $path;
    DEBUG and warn "Setting ", join('/', @{$path}[0..($#$path-1)]), 
                                " to $path->[-1]";
    $heap->{config}->set(@$path);
}

###########################################################
sub load
{
    my($heap, $file)=@_[HEAP, ARG0];
    DEBUG and warn "Loading config file $file";
    $@='';
    $heap->{config}->load($file);
    DEBUG and $@ and warn $@;
}

###########################################################
sub save
{
    DEBUG and warn "Saving config file";
    $@='';
    $_[HEAP]->{config}->save();
    DEBUG and $@ and warn $@;
}

###########################################################
sub clear
{
    DEBUG and warn "Clearing configuration";
    $_[HEAP]->{config}->clear();
}

###########################################################
sub status
{
    my($kernel, $sender, $heap, $pb)=@_[KERNEL, SENDER, HEAP, ARG0];
    DEBUG and warn "Getting status for ", $sender->ID;
    my $conf=$heap->{config}->clear();
    my $f=$conf->{config}{filename}||'NOT SET';

    my $status="Configuration file: $f\n";
    if(keys %{$conf->{config}{delta}}) {
        $status.="Configuration has changed\n";
        $status.="Configuration changs have been saved\n"
                    if $conf->{config}{saved};
    }
    $kernel->post($sender=>$pb, $status);
}

1;
__END__

=head1 NAME

JAAS::Config - JAAS config session

=head1 SYNOPSIS

    use JAAS::Config::Session;

    JAAS::Config::Session->spawn({config=>$config, name=>'Config'});

    my $something=$poe_kernel->call(Config=>'get', \@path);
    $poe_kernel->post(Config=>'get_pb', \@path, 'postback);
    $poe_kernel->post(Config=>'set', [@path, $value]);
    $poe_kernel->post(Config=>'save');
    $poe_kernel->post(Config=>'clear');
    $poe_kernel->post(Config=>'load', $filename);
    $poe_kernel->post(Config=>'status', 'statusback');

=head1 DESCRIPTION

This is a small wrapper session around JAAS::Config.  This is only really
useful in an IKC process, where you want to be able to change the config
"on-the-fly" via the mythical C<JAAS::Console> or C<IKC::Onliner>.


=head1 METHODS

=head2 spawn

    JAAS::Config::Session->spawn();
    JAAS::Config::Session->spawn({config=>$config, name=>'alias'});

Spawns a POE::Session that will provide an interface to a JAAS::Config
object.  

The following events are published via IKC: get, get_pb, set, load, clear,
save, status.

=head1 EVENTS

=head2 get

    my $something=$poe_kernel->call(Config=>'get', \@path);
    my $otherthing=$poe_kernel->call(Config=>'get', "some/path/to/key");

Returns the value stored at C<\@path>, which could be a scalar, hashref or
array.  See L<JAAS::Config::get>.  This event is not published via IKC for
obvious reasons.

=head2 get_pb

    $poe_kernel->post(Config=>'get_pb', \@path, 'postback);
    $poe_kernel->post(Config=>'get_pb', "some/path/to/key", 'postback');

Gets the value stored at C<\@path> and posts it to the 'postback' event in
the current session.  See L<JAAS::Config::get>.  This event is published
via IKC.


=head2 set

    $poe_kernel->post(Config=>'set', [@path, $value]);
    $poe_kernel->post(Config=>'set', "some/path/to/key/$value");

Sets or merges the value stored at C<\@path> to/with $value.
See L<JAAS::Config::set>.  This event is published
via IKC.



=head2 load

    $poe_kernel->post(Config=>'load', $filename);

Causes the configuration to load C<$filename> See L<JAAS::Config::load>. 
This event is published via IKC.


=head2 save

    $poe_kernel->post(Config=>'save');

Causes the configuration to save all changes to a delta-file. See
L<JAAS::Config::save>.  This event is published via IKC.


=head2 clear

    $poe_kernel->post(Config=>'clear');

Clears everything in the configuration. See L<JAAS::Config::clear>.  This
event is published via IKC.

=head2 status

    $poe_kernel->post(Config=>'status');
    $poe_kernel->post(Config=>'status', 'statusback');

Gets a status report on the configuration and posts it back to 'statusback'.
This event is published via IKC.


=head1 AUTHOR

Philip Gwyn <perl at pied.nu>

=head1 SEE ALSO

Config::General, perl(1).

=cut


$Log: Session.pm,v $
Revision 1.4  2001/08/01 05:30:14  fil
Updated the doco.
Fixed the tests
Fixed manifying...  MakeMaker-- GRR GRR GRR

Revision 1.3  2001/07/06 06:12:36  fil
Small bug in JAAS::Config::Session

Revision 1.2  2001/07/06 06:09:56  fil
Documented JAAS::Config::Session
Added JAAS::Config->{saved}, used for status.

Revision 1.1  2001/07/06 05:05:09  fil
Added new-fangled DEBUG stuff
Added JAAS::Config::Session (not documented...)

Revision 1.2  2001/07/06 02:11:46  fil
Added documentation to JAAS::Config!

Revision 1.1.1.1  2001/07/06 00:02:58  fil
JAAS config file and object factories.

