# $Id: Factory.pm,v 1.10 2001/10/12 04:28:33 fil Exp $
package JAAS::Factory;

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

$VERSION = '0.01';
BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () {0}; }
}


###########################################################
sub new
{
    my($package, $config)=@_;
    
    unless(UNIVERSAL::can($config, 'can')) {    # is it an object?
        $config=JAAS::Factory::Config->new($config);
    }
    bless {config=>$config, deleted=>{}}, $package;
}

###########################################################
sub make
{
    my($self, $name, @args)=@_;
    DEBUG and warn "Making $name";

    my $c=$self->{config}->for($name);
    unless($c) {
        $@="No object named $name in config";
        return 0;
    }
    DEBUG and warn "Config is $c";

    my $package=$c->{class} || $c->{package};
    DEBUG and warn "Package is $package";

    unless($self->load_package($package)) {
        $@||="Can't load $package";
        return;
    }
    DEBUG and warn "Loaded $package";

    my $constructor=$self->find_ctor($package, $c->{ctor});
    unless($constructor) {
        $@="Can't find constructor for $package";
        return
    }
    DEBUG and warn "Constructor for $package is $constructor";

    unless($c->{args}) {
        $c->{args}={name=>$name};
    }
    elsif((ref($c->{args})||'') eq 'HASH') {
        $c->{args}{name}||=$name;
    }

    if($c->{includes}) {
        push @args, $self->mk_children($c->{includes});
    }

    my $object=eval {
            $self->build_object($package, $constructor, $c->{args}, \@args);
        };
    unless($object) {
        # DEBUG and 
            warn "Failed to build $name: $@\n";
        return;
    }

    DEBUG and warn "Object is $object";
    return $object;
}



###########################################################
sub find_ctor
{
    my($self, $package, $ctor)=@_;
    my @funcs;
    if($ctor) {
        @funcs=($ctor);
    } else {
        @funcs=qw(spawn new);
    }
    my $func;
    foreach my $c (@funcs) {
        $func=$package->can($c);
        last if $func;
    }
    return $func if $func;
    local $"=', ';
    DEBUG and warn "Can't find contructor for $package: @funcs";        
    return;
}

###########################################################
sub build_object
{
    my($self, $package, $ctor, $args, $more)=@_;
    
    my @under=($package);
    my $r=ref $args;
    if($r and $r eq 'ARRAY') {
        push @under, @$args;
        push @under, @$more if @$more;
    } else {
        push @under, $args if defined $args;
        push @under, @$more if @$more;
    }
    $ctor->(@under);
}

###########################################################
sub _get_table
{
    my($package)=@_;
    my $table=$::{'main::'};
    foreach my $b (split '::', $package) {
        return unless exists $table->{$b.'::'};
        $table=$table->{$b.'::'};
    }
    return $table;
}

###########################################################
sub load_package
{
    my($self, $package)=@_;

    return 1 if defined _get_table($package); # see if it exists in symtable
    # nope... have to load it
    my $path="$package.pm";
    $path=~s{::}{/}g;
    eval { require $path } and return 1;

    # this next line might not be necessary
    delete $INC{$path};
    DEBUG and warn "Can't load $path: $@\n";
    return;
}


###########################################################
sub make_object
{
    my($self, $package, @args)=@_;
    unless($self->load_package($package)) {
        $@||="Can't load $package";
        return;
    }
    my $constructor=$self->find_ctor($package);
    unless($constructor) {
        $@="Can't find constructor for $package";
        return;
    }
    return eval {
        $self->build_object($package, $constructor, undef(), \@args);
    };
}

###########################################################
sub mk_children
{
    my($self, $includes)=@_;

    my $children;
    $includes=[$includes] unless ref $includes;

    if('ARRAY' eq ref $includes) {
        my %t;
        @t{@$includes}=@$includes;
        $includes=\%t;
    }

    my $object;
    while(my($name, $package)=each %$includes) {
        $object=$self->make($package);
        $object=$self->make_object($package) if defined $object and $object==0;
        unless($object) {
            # DEBUG and 
                    warn "Failed to build $name: $@\n";
            next;
        }

        $children->{$name}=$object;
    }
    return $children;
}


###########################################################
package JAAS::Factory::Config;
use strict;
sub DEBUG () {0}

sub new
{
    my($package, $config)=@_;

    my %self;
    if(ref $config eq 'ARRAY') {
        @self{@$config}=map {{package=>$_}} @$config;
        DEBUG and warn "Promoting config to ", ref $config;
    } 
    elsif(ref $config eq 'HASH') {
        %self=%{$config};               # shallow copy....     
    } 
    else {                            # assume scalar
        $self{$config}={package=>$config};  
    }
    return bless \%self, $package;
}

sub for
{
    my($self, $name)=@_;
    return $self->{$name};
}

1;

__DATA__

=head1 NAME

JAAS::Factory - Builder of objects

=head1 SYNOPSIS

    use JAAS::Factory;
    my $factory=JAAS::Factory->new($config);

    # create a new object
    my $obj=$factory->make($object_name);

=head1 DESCRIPTION

JAAS::Factory is an object that, given an configuration, will
construct objects when asked politely.  Construction is controled by
C<$config>.  Objects are asked for by name.  This name is associated with a
package, and optionnal constructor arguments by C<$config>.  Packages can
loaded on the fly.  After construction, the factory doesn't care about the
objects and it's up to the caller to muddle with them.  Run-time control is
achieved via C<$config>.

There is an additional mechanism for overloading some behaviour.

=head1 METHODS

=head2 new

    my $factory=JAAS::Factory->new($config)

Object constructor.  Requires one parameter, C<$config>, which controls how
a given object is created.  C<$config> is an object that has one method
(C<for()> that, when called with a name as parameter, returns a hashref for
that name.  This hashref contains information on how to construct the
objects.  If C<$config> isn't an object, it is assumed to be a hashref, keyed
on object name, that contains the construction information.

Construction information:

=over 4

=item C<class>, C<package>

Scalar value that indicates what package the constructor will be call in. 
In other words, what object will be created.  If both are present C<class>
has priority over C<package>.

=item C<args>

Arguments to the constructor.  If it is an arrayref, it is dereferenced
before passing to constructor, otherwise it is passed as is.

=item C<ctor>

Normally, constructors are call C<spawn> (for C<POE::Session>s) or C<new>
(for standard objects), but MAYBE you like doing things different?  If so,
set the C<ctor> to whatever your constructor is called.  Yes, this means you
can call arbitrary package methods.  Not that you should, eh.

=back


    JAAS::Factory->new({
                # passing an array to Net::Telnet->new();
                Telnet=>{   class=>'Net::Telnet'
                            args=>[Timeout=>10, Prompt=>'/bash\$ $/']
                        },
                # passing a scalar to IO::File->new();
                File=>{     class=>'IO::File',
                            args=>"filename"
                      },
                # Calling an abritrary package method
                Dumper=>{   class=>'Data::Dumper',
                            ctor=>'Dump'
                            args=>[[$foo, $bar], [qw(foo *ary)]]
                        }
                }
            );


If C<$config> is an arrayref, it is assumed to be a list of package names
and is promoted to an object accordingly.

    JAAS::Factory->new([qw(IO::File Net::TCP)]);

Is equivalent to 

    JAAS::Factory->new({IO::File=>{package=>'IO::File'},
                                Net::TCP=>{package=>'Net::TCP'}});

If C<$config> is a scalar, it is assumed to be package name
and is promoted to an object accordingly.

    JAAS::Factory->new('IO::File');

Is equivalent to 

    JAAS::Factory->new({IO::File=>{package=>'IO::File'}});



C<JAAS::Factory->new> should never blow up.







=head2 make

    my $name='File';
    my $something=$factory->make($name, @args);

This method constructs an instance of the named object that is the required
and sole parameter.  Instructions for creating the object are described in
the C<$config> parameter passed to C<new>.

C<make> will load the package if it's not already.  If loading fails, it
returns C<undef()> and possibly something in $@.  C<make> then calls the
constructor, passing it the package name as first param, and any other
parameters defined in C<$config>.  If the constructor throws an exception,
C<make> returns C<undef()> and the exception is in C<$@>.

C<make> will also return C<undef()> if the named object doesn't exists in
C<$config>.

If the arguments in C<$config> are a hashref, the C<$name> parameter is
added to it.  This is because the object must know it's name, eh.  If the
arguments aren't defined in C<$config>, they are set to C<$name>, for the
same reason.  This second bit might not be such a good idea.

If C<@args> is specified, it is added to the constructor call.



=head2 make_object

    my $package='My::Package::Is';
    my $something=$factory->make_object($package, @args);
    die $@ unless $something;

This method constructs an instance of the package named C<$package>

C<make_object> will load the package if it's not already.  If loading fails,
it returns C<undef()> and possibly something in $@.  C<make> then calls the
constructor, passing it the package name as first param, and any other
parameters defined via C<@args>.  If the constructor throws an exception,
C<make_object> returns C<undef()> and the exception is in C<$@>.




=head1 OVERLOADING

It is also possible to change some of JAAS::Factory's behaviour by
deriving a class from it, and overloading the following methods:

=head2 load_package

    sub load_package {
        my($self, $package)=@_;
        ....
    }

This method make sure that $package has been properly loaded.  Current
behaviour is to check the symbol table.  If it's absent, it does a
C<require> (wraped in C<eval()>) on the package.

Returns true if C<$package> is now usable, and false if it isn't.

You might want to overload C<load_package> if you want to pull classes out
of new and exciting places, rather then perl's standard C<require>
mechanism.


=head2 build_object

    sub build_object {
        my($self, $package, $ctor, $args, $more)=@_;
        return $ctor->($package, $args, $more);
    }

This is the heart of the action.  It is expected to call the coderef
C<$ctor> in C<$package>, with the parameters in C<$args>.  C<$package> has
already been loaded by L<load_package>.   

You might want to overload this method to change the parameters handed to
the constructor.


=head2 find_ctor

    sub find_ctor {
        my($self, $package, $ctor)=@_;
        $ctor||='new';
        return $package->can($ctor);
    }

This method returns a coderef of the method C<$ctor> in package C<$package>. 
Returns C<undef()> on failure.  C<$ctor> might be empty, in which case you
should look for a default constructor.

You might want to overload this method if you don't like the fact that the
default method looks for 'spawn' before 'new'.  Or again, maybe you want to
pull your code out of new and exciting places.


=head1 AUTHOR

Philip Gwyn <gwynp@artware.qc.ca>

=cut



$Log: Factory.pm,v $
Revision 1.10  2001/10/12 04:28:33  fil
Factory::build_object wasn't dereferencing $more.

Revision 1.9  2001/10/12 03:05:49  fil
Factory now recognises includes for data member objects.

Revision 1.8  2001/09/21 18:38:23  fil
JAAS::Factory->make now returns 0 (false but defined) if object wasn't
defined in the config.  This is so that you can guess it.

Revision 1.7  2001/08/09 16:26:32  fil
Small debuging and comment change

Revision 1.6  2001/08/02 02:47:02  fil
Forgot test4.conf... oops

Revision 1.5  2001/08/01 05:30:14  fil
Updated the doco.
Fixed the tests
Fixed manifying...  MakeMaker-- GRR GRR GRR

Revision 1.4  2001/07/27 21:38:21  fil
Small changes to help JAECA

Revision 1.3  2001/07/27 00:34:42  fil
Added default

Revision 1.2  2001/07/06 05:05:09  fil
Added new-fangled DEBUG stuff
Added JAAS::Config::Session (not documented...)

Revision 1.1.1.1  2001/07/06 00:03:00  fil
JAAS config file and object factories.

Revision 1.1  2001/06/22 20:42:20  fil
Added Factory, Cache/*
Reworked internals... Still have problems, though.

