package JAAS::Object;

use strict;
use vars qw($VERSION);

$VERSION = '0.01';

use Carp;
use Data::Dumper;
use JAAS::Object::Relationships;
use JAAS::Object::Services;
use POE::Kernel;
use POE::Session;

sub DEBUG () {0}

################################################################
sub create
{
    my $package=shift @_;
    my %args=@_;
    
    my %create;
    foreach my $k (qw(_options)) {
        next unless exists $args{$k};
        $create{$k}=delete $args{$k};
    }

    my $object_conf=delete $args{_object};
    my $object;
    $object=$object_conf;
    croak "No object" unless $object;

    my %methods;
    my @states=qw(_default);
    foreach my $method (keys %args) {
        croak "Please don't call your method $method" if $method=~ /^_/;
        $methods{$method}=$args{$method};
#        push @states, $method;
    }
    
    POE::Session->create(
            args=>[\%methods, $object],
            package_states=>[
                $package=>[qw(_start _stop), @states],
            ],
            %create,
        );
}


################################################################
sub _start
{
    my($kernel, $heap, $methods, $object)=@_[KERNEL, HEAP, ARG0, ARG1];

    $heap->{methods}=$methods;
    $heap->{object}=$object;
    $heap->{name}=$object->name();
    croak "No name" unless $heap->{name};
    $kernel->alias_set($heap->{name});
}

################################################################
sub _default
{
    my @POE=splice @_, 0, ARG0; # gratuitous use of splice() detected at line 133 of file objects.perl

    my($method, $args)=@_;
    $POE[STATE]=$method;                # fix this bug please

    my($package, $heap)=@POE[OBJECT, HEAP];
    my $object=$heap->{object};

    DEBUG and warn "_default =>= $heap->{name}/$method\n";
    return if $method =~ /^_/;

    die "No object" unless $object;
    unless($heap->{methods}{$method}){
        warn $object;
        warn join ', ', keys %{$heap->{methods}};
        warn "Object $object hasn't exposed method '$method'\n";
        return;
    }
    my $about_method=$heap->{methods}{$method};

    unless($object->can($method)) {
        warn "Can't call method $method on ", ref $object;
        return;
    }

    my $postback;
    if($about_method->{respond}) {
        $postback=pop @$args;
        $postback={to=>$postback} unless ref $postback;
        # TODO : This is 100% right.  Maybe SENDER has many names...
        # how can we check that $session is in fact equiv to SENDER?
        $postback->{default_session}=$POE[SENDER]->ID;
    }

    # warn Dumper \@_;
    if(defined($about_method->{args_n}) and $about_method->{args_n}!=@$args) {
        warn ref($object), 
            "->$method must be posted with $about_method->{args_n} argument(s), not ".(@$args)."\n";
        return;
    }

    $jaas_services->_POE(\@POE);
    $@='';
    my $ret=eval{
            my $code=$object->can($method);
            die "Unable to find method $method in class ", ref($object), "\n"
                        unless $code;
            DEBUG and warn "Invoking ", ref($object), "->$method\n";

            # only give scalar context if we actually want something
            # return $code->($object, @$args) if $postback;
            # NO this muddles up reply-to

            # otherwise, void context
            $code->($object, @$args);
            # return;
        };
    if($@) {
        DEBUG and warn $@;
        $jaas_services->inform_monitor('on_error', $heap->{name}, $@);
        $@='';
    }
    elsif($postback) {
        $jaas_services->post($postback, $ret);
    }
    $jaas_services->_POE();
    return $ret;
}

################################################################
sub _stop
{
    my($kernel, $heap)=@_[KERNEL, HEAP];
    $jaas_relationships->remove($heap->{name});
}


1;
__END__

=head1 NAME

JAAS::Object - JAAS object layer for POE

=head1 SYNOPSIS

    use JAAS::Object;
    
    my $thermo=new Household::Thermometer->new(port=>'/dev/ttyC10',
                                               name=>'TAttic');

    JAAS::Object->create(
                _object=>$thermo,
                read=>{args_n=>0, respond=>1},
                interval=>{args_n=>1},
                clear=>{},
            );

    # clear the min/max in the thermometer, from anywhere in POE-land
    $poe_kernel->post(TAttic=>'clear');

    # or inside of JAAS
    $jaas_services->post('clear@TAttic');



    # This package controls a hypothetical temperature probe connected
    # to a cyclades card
    package Household::Thermometer;

    #  ... new() and IOcommand() are left as an exercise for the reader

    sub read {
        my($self)=@_;
        return $self->Icommand('READ PROBE');
    }

    sub interval {
        my($self, $t)=@_;
                                                    
        die "$t is MUCH to big" if $t > 3600;       # Exceptions get caught

        $self->Ocommand("SET INTERVAL=$t");
    }

    sub clear {
        my($self)=@_;
        $self->Ocommand("CLEAR");
    }


=head1 DESCRIPTION

The JAAS object layer is an object model that interfaces with POE.  It sits
on top of POE and provides services to all objects that I knows about. 
Services include posting messages to other objects, monitoring objects,
postbacks and responses.  It also provides a tuple-space for creating and
maintaining relationships between objects.

JAAS::Object actually only handles receiving messages from POE-space for a
particular object.  Posting a message to POE-space is handled by
JAAS::Object::Services.  Object relationships are, naturally enough,
encapsulated in JAAS::Object::Relationships.

=head2 ADDRESSING

Message and object addressing issues are going to be sticky.  Currently, I
use method@object, but this probably won't last.  Note that object is any
POE alias, including sessions that have nothing to do with JAAS and that
'method' could be a POE event.

    If you subscribed to a session via IKC
    event@poe://Kernel/Session

    Mythical Stem interoperability
    target@stem:Hub/Cell


=head1 METHODS

=head2 create
    
    JAAS::Object->create(%parameters);

This creates a POE::Session that deals with receiving messages (aka events)
from POE.  C<%parameters> is a hash with public method names as keys, and
method "prototypes" (see below) as values.  A method name corresponds to a
POE event with the same name.  Method names must not start with '_' because
POE uses that for internal use.  And it would be bad form anyway.  There are
also 2 specials methods:

=over 4

=item C<_object>

Value is your blessed object.  Only one method is called on the object:
C<name>, which is used to set the session's alias, so that it can be
externally addressable and POE won't GC the session.  It is kept internally
(in the session's HEAP), so it won't ever DESTROY until the session is shut
down.

=item C<_options>

Options to be passed to C<POE::Session>->C<create()>.  Useful mainly for
debugging purposes.

=back

Method prototypes are a hashref that may contain the following:

=over 4

=item C<args_n>

Number of arguments the method expects.  Method will not be called if you do
not supply this exactly this amount.  Currently, a warning is displayed and
the method is not called.  This will change when I get exception handling
sorted out.

COMING SOON: args_n could also be an arrayref of [min, max] so that one
could specify a range.

=item C<respond>

Flag that indicates that at the last parameter at invocation is in fact the
address of a message that the method will send it's output to.  This is one
way of avoiding the dreaded C<call()>.  The other is 'reply-to' (see
L<JAAS::Object::Services>).

To use the C<read> method in the example given in SYNOPSIS, you do the
following:

    $jaas_services->post('read@TAttic', 'read');
    # method 'read' in current object will receive the current temperature
    # reading


=back

=head1 AUTHOR

Philip Gwyn <perl at pied.nu>

=head1 SEE ALSO

perl(1).

=cut
