package JAAS::Object::Services;

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK $jaas_services);

require Exporter;

@ISA = qw(Exporter);
@EXPORT = @EXPORT_OK = qw($jaas_services $jaas_relationships);
$VERSION = '0.01';
use POE::Session;
use JAAS::Object::Relationships;

sub DEBUG () {0}

################################################################
sub new
{
    my($package)=@_;
    bless {}, $package;
}


################################################################
sub _POE
{
    my($self, $poe)=@_;
    $self->{POE}=$poe;
}




################################################################
sub post
{
    my($self, $name, @args)=@_;
    die "Not called from within a JAAS method\n" unless $self->{POE};
    
    my $address=$self->_parse($name);
    my $dest=$address->{to};

    my $kernel=$self->{POE}[KERNEL];
    
    DEBUG and do {
        local $"=", ";
        DEBUG and warn "Post: $dest->{object}/$dest->{method}(@args)\n";
    };
    unless($address->{"reply-to"}) {
        # let POE handle routing for now
        $kernel->post($dest->{object}, $dest->{method}, @args);
    }
    else {
        my $back=$address->{"reply-to"};

        my(@ret, $yes);
        # this won't work for IKC... GRRRRR
        if($dest->{wantarray}) {                    # silly
            @ret=$kernel->call($dest->{object}, $dest->{method}, @args);
            $yes=1 if @ret > 0;
        } else {
            $yes=$kernel->call($dest->{object}, $dest->{method}, @args);
            if(defined $yes) {
                $ret[0]=$yes;
                $yes=1;
            }
        }
        if($yes) {
            # die "defined" if defined $ret[0];
            DEBUG and do {
                local $"=", ";
                warn "Reply-to: $back->{object}/$back->{method}(@ret)\n";
            };
            $kernel->post($back->{object}, $back->{method}, @ret);
        }
    }
}


################################################################
sub inform_monitor
{
    my($self, $cause, $who, $about)=@_;
    $about={message=>$about} unless ref $about;
    $about->{cause}=$cause;

    my $what=$jaas_relationships->find($cause, $who);
    foreach my $worried (values %$what) {
        $self->post($worried, $about);
    }
}

################################################################
# crude attempt at grokking as many ways of addressing as possible
sub _parse 
{
    my($self, $what)=@_;
    $what={to=>$what} unless ref $what;
    $what->{default_object} ||= $self->{POE}[SESSION]->ID;

    foreach my $t (qw(reply-to from to)) {
        next unless $what->{$t};
        unless(ref $what->{$t}) {
            my $q=$what->{$t};
            $what->{$t}={};
            @{$what->{$t}}{qw(method object)}=split '@', $q, 2;
        }
        $what->{$t}{object} ||= $what->{default_object};
        # Here is where we detect Stem and IKC and so on
        $what->{$t}{ikc}=1 if $what->{$t}{object} =~ /^poe:/;
    }
    @{$what->{from}}{qw(line file package)}=(caller(2))[2,1,0];
    return $what;
}

BEGIN {
    $jaas_services||=new __PACKAGE__;
    $jaas_relationships||=new JAAS::Object::Relationships;
}

$jaas_relationships->add_types({on_error=>[qw(store)]});


1;
__END__
# Below is the stub of documentation for your module. You better edit it!

=head1 NAME

JAAS::Object::Services - Provide POE and other services to JAAS::Objects

=head1 SYNOPSIS

    use JAAS::Object::Services;

    $jaas_services->post('method@object');
    $jaas_relationships->nail($postback, 'subscribed', "one_true_list");

=head1 DESCRIPTION

Coordinates objects communicating back into POE-space.  And amoungst each
other, really.






=head1 METHODS

=head2 post

    $jaas_services->post($address, @arguments);

Send a message to another object... or any POE session, really.  In the
interest of Fun and DWIMitud, C<$address> can be one of the following
formats:

=over 4

=item plain string

This is taken as the destination address for the message.  Current format
for addresses is method@object or event@session.  This means that
event@poe://subscribed/session will work.

=item hashref

Contains the following keys:  

'to' destination address of the message.

'from' source address of the message (not implemented)

'reply-to' yet another way of not using C<POE::Kernel::call()>.  The return
value of the method is sent back to this address.  Yes, it uses C<call()>
internally, but ONLY SO THAT YOU DON'T HAVE TO.

Values of these keys can be either a plain string, or a hash with 3 pairs:
'ikc', 'object' and 'method'.  Yes, I should add kernel, though I currently
don't need to.  'ikc' is a flag indicating that this is an IKC address.
    
=item default_object

If any of the addresses in the structure doesn't have a session, this one is
used.  This is used internally for 'respond'-type methods.  If not
specified, current object is used.


=back

So, this means:

    'foo@bar'
    equiv to
    {   to=>{object=>'bar', method=>'foo'},
        default_object=>42,
    }


    'foo@poe://kernel/session'
    equiv to
    {   to=>{method=>'foo', object=>'poe://kernel/session', ikc=>1}, 
        default_object=>42,
    }

    {to=>'foo@bar', 'reply-to'=>'wibble'}
    equiv to 
    {   to=>{object=>'bar', method=>'foo'},
        'reply-to'=>{object=>42, method=>'wibble'},
        default_object=>42,
    }

More smarts to be added as needed.  Like adding 'from' when I can.


=head1 MONITORING

You can set up monitoring of other objects via relationships.

    $jaas_relationships->nail('me', 'on_error', 'other_object', 'error@me');

So, when a method in 'other_object' throws an exception, 'error@me' gets
called with a secret-sacred-roach-hashref as the first argument.

Only 'on_error' is currently defined.  This form of monitoring is going to
change next week.

=head1 OVERLOADING

If you want to overload C<$jaas_services>'s behaviour, you have to do the
following:

    # define anything needed by new()
    BEGIN {
        $JAAS::Object::Services::jaas_services=new MyPackage;
    }

    use JAAS::Object::Services;

Note that the BEGIN block comes before the C<use>, and that
C<MyPackage::new> must be defined before the BEGIN block.

The following methods are also used internally:

=head2 _POE

Handed an arrayref containing all POE-specific parameters for a message. 
This is saved so that C<post()> will be able to post messages into
POE-space.

=head2 _parse

Turn a address into an address hashref.  It can be handed a simple string,
or a half filled-in hashref.  See C<post> to find out what an address
structure can look like.

=head2 inform_monitor

Read the source.

=head1 AUTHOR

Philip Gwyn <perl at pied.nu>

=head1 SEE ALSO

perl(1).

=cut
