# $Id: Relationships.pm,v 1.5 2001/09/18 19:57:17 fil Exp $
package JAAS::Object::Relationships;

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK $jaas_relationships %KNOWN_FLAGS);
use Carp;

$VERSION = '0.01';
require Exporter;

@ISA = qw(Exporter);
@EXPORT = @EXPORT_OK = qw($jaas_relationships);
BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () {0}; }
}

###############################################################
sub new
{   
    my($package, $types)=@_;
    my %types;
    my $self=bless {nails=>{}, types=>{}}, $package;
    $self->add_types($types) if $types;
    return $self;
}

###############################################################
sub add_types
{
    my($self, $types)=@_;
    $self=$jaas_relationships unless ref $self;

    my $conf=$self->{types};
    # $self->add_types([qw(foo bar baz)]);
    if('ARRAY' eq ref $types) {
        @{$conf}{@$types}=({}) x @$types;
    }
    elsif('HASH' eq ref $types) {
        foreach my $type (keys %$types) {
            my @flags;
            # $self->add_types({foo=>[qw(store bidirectional)]);
            if('ARRAY' eq ref $types->{$type}) {
                @flags=@{$types->{$type}};
            } 
            # $self->add_types({foo=>{store=>1}});
            elsif('HASH' eq ref $types->{$type}) {
                @flags=grep {$types->{$type}{$_}} keys %{$types->{$type}};
            }
            else {
                croak "Flags of $type must be arrayref or hashref";
            }
            @{$conf->{$type}}{@flags}=(1) x @flags;
        }
    } else {
        croak "Relationship types must be defined via a arrayref or hashref";
    }

    return $self;
}

###############################################################
sub nail
{
    my($self, $from, $rel, $to, $what)=@_;
    $self=$jaas_relationships unless ref $self;

    $from=$from->name if ref $from;
    $to=$to->name if ref $to;

    my $type=$self->{types}{$rel};
    croak "I don't know about '$rel' relationships" unless $type;
    $what=1 unless defined($what) and $type->{store};

    $self->{nails}{$to}{$rel}{$from}=$what;
    DEBUG and warn "Nailing $to $rel $from\n";
    if($type->{bidirectional}) {
        $self->{nails}{$from}{$rel}{$to}=$what;
        DEBUG and warn "Nailing $from $rel $to\n";
    }
    return $self->_temporary($from, $rel) if defined wantarray;
}

###############################################################
sub break
{
    my($self, $from, $rel, $to)=@_;
    $self=$jaas_relationships unless ref $self;

    $from=$from->name if ref $from;
    $to=$to->name if ref $to;

    my $type=$self->{types}{$rel};
    delete $self->{nails}{$to}{$rel}{$from};
    delete $self->{nails}{$to}{$rel} unless keys %{$self->{nails}{$to}{$rel}};
    delete $self->{nails}{$to} unless keys %{$self->{nails}{$to}};
    DEBUG and warn "Breaking $to $rel $from\n";

    if($type->{bidirectional}) {
        delete $self->{nails}{$from}{$rel}{$to};
        delete $self->{nails}{$from}{$rel} 
                        unless keys %{$self->{nails}{$from}{$rel}};
        delete $self->{nails}{$from} unless keys %{$self->{nails}{$from}};
        DEBUG and warn "Breaking $from $rel $to\n";
    }

    return $self->_temporary($from, $rel) if defined wantarray;
    
}

###############################################################
sub remove
{
    my($self, $who)=@_;
    $self=$jaas_relationships unless ref $self;
    my $nails=$self->{nails};

    my $n;
    if($nails->{$who}) {
        DEBUG and carp "Deleting $who";
        delete $nails->{$who};        # this is the easy one
        $n++;
    }
                                    # scan for all those on the far end
    foreach my $from (keys %$nails) {
        my $rels=$nails->{$from};
        foreach my $rel (keys %$rels) {
            next unless $rels->{$rel}{$who};
            DEBUG and warn "Deleting $from < $rel < $who";
            delete $rels->{$rel}{$who};
            delete $rels->{$rel} unless keys %{$rels->{$rel}};
            $n++;
        }
        delete $nails->{$from} unless keys %{$nails->{$from}};
    }
    return $n;
}

###############################################################
sub status
{
    my($self, $who)=@_;
    $self=$jaas_relationships unless ref $self;
    my @ret;

    if($who) {
        push @ret, $self->_status($who);
    } else {
        foreach my $who (sort keys %{$self->{ref($self)}{nails}}) {
            push @ret, $self->_status($who);
        }
    }
    return @ret if wantarray;
    return join "\n", @ret;
}

sub _status
{
    my($self, $who)=@_;
    my $types=$self->{types};
    my $nails=$self->{nails};

    my @ret=("$who:");
    foreach my $rel (sort keys %{$nails->{$who}}) {
        my $q="    ";
        $q .= $types->{$rel}{bidirectional} ? "-<->-" : "-->--";
        foreach my $to (sort keys %{$nails->{$who}{$rel}}) {
            push @ret, qq($q $to);
        }
    }

    return @ret;
}


###############################################################
sub find
{
    my($self, $rel, $who)=@_;
    $self=$jaas_relationships unless ref $self;
    croak "NO WHO" unless defined $who;
    return unless exists $self->{nails}{$who};
    return unless exists $self->{nails}{$who}{$rel};

    my %ret=%{$self->{nails}{$who}{$rel}||{}};
    return unless keys %ret;
    return keys %ret if wantarray;
    return \%ret;
}

###############################################################
sub rfind
{
    my($self, $rel, $who)=@_;
    $self=$jaas_relationships unless ref $self;

    my %ret;
    while(my($from, $rels)=each %{$self->{nails}}) {
        next unless $rels->{$rel};
        foreach my $to (keys %{$rels->{$rel}}) {
            $ret{$from}=$rels->{$rel}{$to} if $to eq $who;
        }
    }
    return unless %ret;
    return keys %ret if wantarray;
    return \%ret;
}


###############################################################
sub _temporary
{
    JAAS::Object::Relationships::Temporary->new(@_);
}

BEGIN {
    %KNOWN_FLAGS=qw(bidirectional 1 store 1);
}





###############################################################
package JAAS::Object::Relationships::Temporary;
use strict;

sub new
{
    my $package=shift @_;
    return bless [@_], $package;
}

sub nail
{
    my($self, $to)=@_;
    $self->[0]->nail($self->[1], $self->[2], $to);
    return $self;
}

sub break
{
    my($self, $to)=@_;
    $self->[0]->break($self->[1], $self->[2], $to);
    return $self;
}

1;
__END__

=head1 NAME

JAAS::Object::Relationshops - POE object relationship collection

=head1 SYNOPSIS

    use JAAS::Object::Relationships;

    my $notary=JAAS::Object::Relationships->new({
            subscribed=>{},
            married=>{bidirectional=>1},
        });

        # create some relationships
    $notary->nail('pete'  , 'subscribed', 'Montreal Gazette');
    $notary->nail('betty' , 'subscribed', 'Montreal Gazette');
    $notary->nail('george', 'subscribed', 'Montreal Gazette')
                ->nail('Journal de Montreal');  # syntatic sugar

        # search relationships
    my @subscribers=$notary->find('subscribed', 'Montreal Gazette');
        # returns george, pete, betty

        # married is bidirectional, so this creates 2 relationships
    $notary->nail('betty', 'married', 'pete');
    my $spouse=$notary->find('married', 'betty');

        # She'd dead, Jim!
    $notary->remove('betty');
        # no more wife!
    my $wife=$notary->find('married', 'pete'); 
    
        # break a relationship
    $notary->break('pete', 'subscribed', 'Montreal Gazette');

        # only george left
    @subscribers=$notary->find('subscribed', 'Montreal Gazette');

        # slower then find, because of internal representation
    @subscriptions=$notary->rfind('subscribed', 'george');


=head1 DESCRIPTION

Relationships are a way of associating various objects with each other.  A
relationship is a one way link between 2 objects.  Objects are referenced by
name (a simple scalar).  The link is one of various types of configurable
relationships.

An example relationship is object "foo" is in kernel "bar" (foo kernel bar). 
You can then search for all the objects in kernel "bar" (? kernel bar)
(C<find()>) or all the kernels that have an object "foo" (foo kernel ?)
(C<rfind()>).

You can use relationships to build up lists.  This will take care of all the
house-keeping for you.  Say you want a list of postbacks for a given event.

    $notary->nail(qw(poe://kernel/session/event subscribed something));
    $notary->nail(qw(method@object subscribed something));
    $notary->nail(qw(yibble@foobar subscribed something));

Then, when you want to post the 'something' event, you search like so:

    @subscribed=$notary->find('subscribed', 'something');

What's more, if you no longer want to post the 'something' event, you just
remove it collection.

    $notary->remove('something');

So now the list has disapeared.  

This is also useful, in IKC, for when a kernel disconnects.  Removing the
kernel name from a collection will break all relationships that include the
kernel.  Of course, this means that you have only one collection of
relationships within the application, and that we've solved how to map
message address to actual kernel name.  One way is to create a "located"
maping.

    $notary->nail("poe://Bot32/mouth/talk", 'located', 'Bot32');
    $notary->nail("poe://Bot32/mouth/talk", 'subscribed', 'sound-output');

So that when Bot32 disconnects, we can do the following:

    foreach my $spec ($notary->rfind('located', 'Bot32')) {
        $notary->remove($spec);
    }
    $notary->remove('Bot32');

This could be generalised into some form of parent/child relationship and
C<remove()> would be made smarter.

I've realised that relationships are in fact 3-way tuples.  The current
implementation offers limited search capabilities.  There is no tuple module
on CPAN.  I should either find a cool way to implement one, or poke someone
(Damian!) into writing one.



=head2 NAMESPACE

There is only ONE namespace.  If you have a kernel named 'foo' and an object
named 'foo' and you then remove('foo'), BOTH will disapear.  Giving objects
and kernels different names is easy.  But 

If/when I get a real tuple-space, I could then add type: to the front of
names (say), so that (foo kernel bar) becomes (object:foo in kernel:bar) and
you would search for (? in kernel:bar).  This wouldn't solve the problem of
having multiple objects of the same name in different kernels.  Maybe I
should use full addresses.




=head1 METHODS

=head2 new

    JAAS::Object::Relationship->new([$types]);

Creates a new relationship collection.  C<$types> is a description of
possible relationships.  See the description under C<add_types>.

=head2 add_types

    $notary->add_types($types);

Adds new types of possible relationships to the collection.  C<$types> is a
hashref, keyed on relationship type.  Values are a hashref of flags:

=over 4

=item bidirectional

Means that a given relationship is 2-directional.  Creating a relationship
in one direction causes the other to be created.  Breaking one causes the
other to be broken.  This means that

    $notary->nail(qw(a B c));    
    $notary->nail(qw(c B a));    

Is equivalent to

    $notary->add_types({B=>{bidirectional=>1}});
    $notary->nail(qw(a B c));    

=item store

Allows you to store arbitrary data with each relationship.  This is useful
if you want to keep some OOB data in the relationship.

    $notary->add_types({subscribed=>{store=>1}});
    $notary->nail($callback, 'subscribed', 'my_event', $args);
    # ie, we want to $callback->(@$args)

Relationships with "stores" are 4-way tuples.

=back


As syntatic sugar, C<$types> also be an arrayref:

    $notary->add_types([qw(foo bar baz)]);
    # equiv to $notary->add_types({foo=>{}, bar=>{}, baz=>{}});

Also, flags can be an arrayref:

    $notary->add_types({foo=>[qw(bidirectional store)]});
    # equiv to $notary->add_types({foo=>{bidirectional=>1, store=>}});


=head2 nail

    $notary->nail($who, $type, $what[, $data]);

Creates a relationship of type C<$type> from C<$who> to C<$what>.  C<$type>
must be predefined via C<add_types()> or an exception is thrown.  C<$data>
is only valid if the C<$type> has the flag store set.

If not in void context, C<nail> returns value is a temporary object that
will allow you to enumerate things:

    $notary->nail($who, $type, $what1)->nail($what2);
    # equiv to 
    $notary->nail($who, $type, $what1);
    $notary->nail($who, $type, $what2);

Yes, they can be chained

    $notary->nail('beer', 'good', 'St-Ambroise Oatmeal Stout')
                        ->nail('Boreal Noir')
                        ->nail('Unibroue Trois Pistol');

If C<$type> is bidirectional, you will in fact be creating 2 relationships.

=head2 break

    $notary->break($who, $type, $what);

Deletes a relationship from the collection.  Does not check to see if the
relationship actually existed.

If C<$type> is bidirectional, you will in fact be breaking 2 relationships.




=head2 remove

    $notary->remove($who);

Does a batch delete of all the relationships that C<$who> is in.



=head2 find

    $hashref=$notary->find($type, $what);
    @array=$notary->find($type, $what);

Searches the collection for objects that are in related via C<$type> to
C<$what>.  Come again?  

    $notary->nail(qw(a b c));
    print $notary->find(qw(b c));
    # will print 'a'

In scalar context, returns a hashref, keyed on object name, value is either
the stored data, or 1.  In array context, returns an array of all object
names.


=head2 rfind

    $hashref=$notary->rfind($type, $who);
    @array=$notary->rfind($type, $who);

Does the oposite of C<find()>.  Searches the collection for objects that
C<$who> have a C<$type> relationship to.

    $notary->nail(qw(a b c));
    print $notary->rfind(qw(a b));
    # will print 'c'

Because I haven't found a clever way to represent the relationships, rfind
is MUCH slower then find.  The reason C<rfind> is seperate from C<find> is
that relationships aren't bidirectional.  Just because foo subscribed to
bar doesn't mean bar subsribed to foo and you need to distinguish searches
for one or the other.

=head2 status

    $everything=$notary->status();
    @everything=$notary->status();

Returns a string that shows all  relationships in a
collection.

    $all_about_susan=$notary->status('susan');

Returns a string that shows all relationships that a given person is in.

=head1 AUTHOR

Philip Gwyn <perl at pied.nu>

=head1 SEE ALSO

perl(1).

=cut


$Log: Relationships.pm,v $
Revision 1.5  2001/09/18 19:57:17  fil
Added the OOB feature

Revision 1.4  2001/07/26 22:47:16  fil
Fixed error handling

Revision 1.3  2001/07/24 21:26:30  fil
We now ask the objects for methods, rather then in the constructor.  See
_attach, _detach and so on

Revision 1.2  2001/06/29 14:14:04  fil
Implemented and documented first draft.
