package JAAS::Object::Services;

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK $jaas_services $alarm_id);

require Exporter;

@ISA = qw(Exporter);
@EXPORT = @EXPORT_OK = qw($jaas_services $jaas_relationships);
$VERSION = '0.01';
use POE::Session;
use JAAS::Object::Relationships;
use Carp;

BEGIN {
    if($INC{"JAAS/Config.pm"})
    {   *DEBUG=sub () {__PACKAGE__->debug}}
    else {
        *DEBUG=sub () {0}
    }
}

$alarm_id=0;

################################################################
sub spawn
{
    my($package, $config)=@_;
    my $self=bless {config=>$config}, $package;
    $jaas_services||=$self;
    return $self;
}


################################################################
sub _POE
{
    my($self, $poe)=@_;
    $self->{POE}=$poe;
}

################################################################
sub _OOB
{
    my($self, $oob)=@_;
    $self->{OOB}=$oob;
}

################################################################
sub exec_ctx
{
    my($self, $new)=@_;
    if(1==@_) {
        return {%{$self->{CTX}||{}}};
    }
    $self->{CTX}=$new;
}




################################################################
sub build_continuation
{
    my($self, $req)=@_;

    my $kernel = $self->{POE}[KERNEL] || $POE::Kernel::poe_kernel; # grr
    $req=$self->_parse($req, $kernel->get_active_session->ID);

    # propagate things like wantarray, reply-to, and from
    if($self->{OOB}) {

        DEBUG and print STDERR "Continuation via OOB $self->{OOB}\n";
        if($req->{'reply-to'} and $self->{OOB}{'reply-to'}) {
            $self->{OOB}{__todo} ||= [];
            my $t=delete $self->{OOB}{'reply-to'};
            push @{$self->{OOB}{__todo}}, $t;
            DEBUG and print STDERR "   Stack ", ">"x (0+@{$self->{OOB}{__todo}}), 
                        " $t->{method}\@$t->{object}\n";
        } 
        elsif(0) {
            use Data::Dumper;
            print STDERR "****** req=", Dumper $req unless $req->{'reply-to'};
            print STDERR '****** OOB=', Dumper $self->{OOB} unless $self->{OOB}{'reply-to'};
        }

        foreach my $k (keys %{$self->{OOB}}) {
#            next if $k eq 'from';
            $req->{$k}=$self->{OOB}{$k} unless exists $req->{$k};
        }

        $self->{OOB}{continuing}=1;
        if(DEBUG) {
            my @stack;
            push @stack, @{$self->{OOB}{__todo}} if $self->{OOB}{__todo};
            # push @stack, $self->{OOB}{'reply-to'} if $self->{OOB}{'reply-to'};
            push @stack, $req->{'reply-to'} if $req->{'reply-to'};
            print STDERR "Return stack: ", join(', ', 
                            map {join '@', @{$_}{qw(method object)}}
                                                     @stack), "\n" if @stack;
        }
    }
    return $req;
}

sub continuation
{
    my $self=shift;
    $self=$jaas_services if not ref $self;
    my $req=$self->build_continuation(shift @_);
    DEBUG and warn "continuation with ", 0+@_, " arguments";
    $self->post($req, @_);
    return;
}


################################################################
sub _pop_todo
{
    my($self)=@_;

    return unless $self->{OOB};
    my $OOB=$self->{OOB};
    # print STDERR "We have OOB $OOB\n";

    my $t;
    # NO!  Don't pretend 'reply-to' is in the stack at this point, because
    # strange things happen.
#    if($OOB->{'reply-to'}) {
#        $t=delete $OOB->{'reply-to'};
#    } 
#    els
    if(@{$OOB->{__todo}||[]}) {
        $t=pop @{$OOB->{__todo}};
        DEBUG and print STDERR "   Stack ", "<"x (1+@{$OOB->{__todo}}), 
                            " $t->{method}\@$t->{object}\n";
        delete $OOB->{__todo} if 0==@{$OOB->{__todo}};
    }
    else {
        return;
    }

    if(DEBUG) {
        my @stack;
        push @stack, @{$OOB->{__todo}} if $OOB->{__todo};
        # push @stack, $OOB->{'reply-to'} if $OOB->{'reply-to'};
        print STDERR "Return stack: ", join(', ', 
                            map {join '@', @{$_}{qw(method object)}}
                                                     @stack), "\n" if @stack;
    }
    return $t;    
}

################################################################
sub post
{
    my($self, $req, @args)=@_;

    # all we need is the KERNEL, really
#    croak ref($self), 
#        "->post not called from within a JAAS method\n" unless $self->{POE};
    
    unless(ref $req) {                 # add a bit of brains
        $req={to=>$req};
        # note that to=>scalar gets promoted to a hashref in _parse
    }

    #######################
    my $kernel = $self->{POE}[KERNEL] || $POE::Kernel::poe_kernel;
    my $current;
    if(@{$self->{POE}||[]}) {
        confess "NO SESSION" unless $self->{POE}[SESSION];
        $current=$self->{POE}[SESSION]->ID;
    } else {
        $current=$kernel->get_active_session()->ID;
    }
    $req->{sender}=$current;
    $req->{default_object} ||= $current;


    ##########################
    my $address=$self->_parse($req);   # add even more brains

    my $dest=$address->{to};

    use Data::Dumper;
    confess "I wasn't given a object proper address to post! ", Dumper $address 
                    unless $dest->{object} and $dest->{method};
    DEBUG and 
    do {
        local $"=", ";
        print STDERR "Post: $dest->{method}\@$dest->{object}(@args) from session $req->{sender}\n";
    };


    ################################################
    # Pop the TODO stack
    $req->{'reply-to'} ||= $self->_pop_todo();


    ################################################
    # Create OOB, then the far end must deal with it. 
    # it's up to the caller to prevent this from happening
    unless($req->{noOOB}) {
        DEBUG and warn "Using OOB data\n";
        push @args, bless {wantarray=>$req->{wantarray},
                            from=>$req->{from}, 
                            __todo=>($req->{__todo}||[]),
                            'reply-to'=>$req->{'reply-to'}, OOB=>1, 
                            'error-to'=>$req->{'error-to'}}, 'JAAS::OOB';

        if(DEBUG and $req->{'reply-to'}) {
            warn "$dest->{method}\@$dest->{object} is going to reply to $req->{'reply-to'}{method}\@$req->{'reply-to'}{object} using OOB\n";
        }
    }
    else {
        # print STDERR __PACKAGE__, "::post OOB --\n";
    }

    ################################################
    # routing goes here
    unless($req->{noOOB} and $address->{"reply-to"}) {
        # let POE handle routing for now
        DEBUG and do {
            local $"=", ";
            print STDERR "kernel->post($dest->{object}, $dest->{method}, @args)\n";
        };
        $kernel->post($dest->{object}, $dest->{method}, @args);
    }

    else {
        # If we don't have OOB, but we want a reply, we can't depend on 
        # JAAS::Object or whatever else is on the other end

        # this mess is partially moved into JAAS::Object::_default, but
        # the caller has to allow us to use OOB
        DEBUG and warn "reply-to in action for $dest->{method}";
        my $back=$address->{"reply-to"};
        $back->{'wantarray'}||=$address->{'wantarray'};

        my($ret, $yes);

        if($dest->{ikc}) {
            
            DEBUG and 
                print STDERR __PACKAGE__, " IKC call\n";
            # ICK/call will only allow one return, so we ignore wantarray
            $kernel->call(IKC=>'call', 
                          'poe:'.$dest->{object}.'/'.$dest->{method}, 
                            \@args, 
                          'poe:'.$back->{object}.'/'.$back->{method}
                            );
            # If $back is a method, it will have been published
            # If it's an event, hope that the session has published something
        } 
        else  {                     # probably local kernel

            DEBUG and do {
                local $"=", ";
                print STDERR "kernel->call($dest->{object}, $dest->{method}, @args)\n";
            };
            # NB : reply-to doesn't work in conjunction w/ respond... AT ALL
            $ret=$kernel->call($dest->{object}, $dest->{method}, @args);

            # it would be cool to do yield() here and then do the following
            # some other way
            $self->_postback($kernel, $back, $ret);
        }
    }
}

################################################################
sub _postback
{
    my($self, $kernel, $back, $ret)=@_;

    my @ret;
    if(defined $ret) {
        @ret=($ret);            # silly
    } 
    else {
        DEBUG and 
            print STDERR __PACKAGE__, " returned undef\n";
        return;
    }

    # call doesn't propagate wantarray(), we do it the "hard" way
    if($back->{'wantarray'}) {
        if(ref $ret eq 'ARRAY') {
            @ret=@$ret;                 # ARRAY + wantarray == hapiness
        } elsif(DEBUG) {
            print STDERR __PACKAGE__, " $ret isn't an arrayref\n";
        }
    } elsif(DEBUG) {
        print STDERR __PACKAGE__, " scalar context\n";
    }

    if(@ret) {                              # actually got a response
        DEBUG and do {
            local $"=", ";                  # zealous debugging
            print STDERR "Reply-to: $back->{method}\@$back->{object}(@ret)\n";
        };
                                            # send the response back
        # Because post() might call _postback() again, so we could recurse 'til
        # the cows came home (if everything has noOOB).  So we mess with the
        # call stack using goto
        @_=($self, {to=>$back}, @ret);
        my $post=$self->can('post');        # still OO
        goto &$post;

        # Must use object services, so that __todo etc is looked at
        # $self->post({to=>$back}, @ret);
    } elsif(DEBUG) {
        print STDERR __PACKAGE__, " nothing to return\n";
    }
}

################################################################
sub sender
{
    my($self)=@_;
    return $self->{POE}[SENDER]->ID;
}

################################################################
sub monitor_inform
{
    my($self, $cause, $who, $about)=@_;
    # TODO: sanner handling of blank $who
    croak "NO WHO" unless $who;
    $about={message=>$about} unless ref $about;
    $about->{cause}=$cause;

    DEBUG and warn "informing $who $cause\n";
    my $what=$jaas_relationships->find($cause, $who);
    my $count=0;
    foreach my $worried (values %$what) {
#        my $more=$worried->{data} if ref $worried eq 'HASH' and
#                                    exists $worried->{data};
        DEBUG and warn "Posting $worried, $about\n";
        $self->post($worried, $about);
        $count++;
    }
    $what=$jaas_relationships->find($cause, '*');
    foreach my $worried (values %$what) {
        DEBUG and warn "Posting $worried, $about\n";
        $self->post($worried, $about);
        $count++;
    }
    use Data::Dumper;
    print STDERR __PACKAGE__, " No one cares about this: ", Dumper $cause, $who, $about, $jaas_relationships unless $count;
}

################################################################
# crude attempt at grokking as many ways of addressing as possible
sub _parse 
{
    my($self, $what, $default)=@_;
    $what={to=>$what} unless ref $what;
    die "$what must be a hash" unless 'HASH' eq ref $what;
#    $what->{default_object} ||= $self->{POE}[SESSION]->ID;

    foreach my $t (qw(reply-to from to)) {
        next unless $what->{$t};
        unless(ref $what->{$t}) {
            my $q=$what->{$t};
            $what->{$t}={};
            @{$what->{$t}}{qw(method object)}=split '@', $q, 2;
        }
        $what->{$t}{object} ||= $what->{default_object}||$default;
        # Here is where we detect Stem and IKC and so on
        $what->{$t}{ikc}=1 if $what->{$t}{object} and 
                                $what->{$t}{object} =~ /^poe:/;
    }
    my $q=1; 
    $q++ while(caller($q) eq __PACKAGE__);
    @{$what->{from}}{qw(line file package)}=(caller($q))[2,1,0] 
                                    unless $what->{from};
    @{$what->{poster}}{qw(line file package)}=(caller($q))[2,1,0];
    return $what;
}

################################################################
BEGIN {
#     $jaas_services||=new __PACKAGE__;
     $jaas_relationships||=new JAAS::Object::Relationships;
}


$jaas_relationships->add_types({on_error=>[qw(store)], 
                                on_warning=>[qw(store)]});





################################################################
sub config_set
{
    my($self, $config)=@_;
    $self=$jaas_services unless ref $self;
    $self->{config}=$config;
}

################################################################
sub config_get
{
    my($self)=@_;
    $self=$jaas_services unless ref $self;
    return $self->{config};
}

################################################################
sub debug
{
    my($self, $package)=@_;
    $self=$jaas_services unless ref $self;
    return 0 unless $self->{config};

    $package||=caller;
    return $self->{config}->get('debug', $package);
}







################################################################
sub alarm_set
{
    my($self, $req, $delay, @args)=@_;

    my $kernel = $self->{POE}[KERNEL] || $POE::Kernel::poe_kernel; # grr

    unless(ref $req) {                 # add a bit of brains
        $req={to=>$req};
        # note that to=>scalar gets promoted to a hashref in _parse
    }

    #######################
    my $current=$kernel->get_active_session()->ID;
    $req->{sender}=$current;
    $req->{default_object} ||= $current;

    ##########################
    my $post=$self->_parse($req);   # add even more brains

    $alarm_id++;

    $self->{alarms}{$alarm_id}=$post->{to}{object};

    $kernel->call($post->{to}{object}, '_alarum_set', 
                        $alarm_id, $post, $delay, \@args);

    return $alarm_id;
}

################################################################
sub delay_set
{
    my ($self, @foo)=@_;
    $foo[1] += time;
    $self->alarm_set(@foo);
}

################################################################
sub alarm_remove
{
    my($self, $id)=@_;

    my $kernel = $self->{POE}[KERNEL] || $POE::Kernel::poe_kernel; # grr
    $kernel->call(delete $self->{alarms}{$id}, '_alarum_remove', $id);
}

################################################################
sub alarm_adjust
{
    my($self, $id, $sec)=@_;

    my $kernel = $self->{POE}[KERNEL] || $POE::Kernel::poe_kernel; # grr
    $kernel->call($self->{alarms}{$id}, '_alarum_adjust', $id, $sec);
}


1;
__END__

=head1 NAME

JAAS::Object::Services - Provide POE and other services to JAAS::Objects

=head1 SYNOPSIS

    use JAAS::Object::Services;

    JAAS::Object::Services->spawn($config);

    $jaas_services->post('method@object');
    $jaas_relationships->nail($postback, 'subscribed', "one_true_list");

    $jaas_services->debug(__PACKAGE__) and warn "OOPSIE!";

    my $sender=$jaas_services->sender();
    $jaas_services->post("something@$sender", $data);

=head1 DESCRIPTION

Coordinates objects communicating back into POE-space.  And amoungst each
other, really.






=head1 METHODS

=head2 post

    $jaas_services->post($address, @arguments);

Send a message to another object... or any POE session, really.  In the
interest of Fun and DWIMitud, C<$address> can be one of the following
formats:

=over 4

=item plain string

This is taken as the destination address for the message.  Current format
for addresses is method@object or event@session.  This means that
event@poe://subscribed/session will work.

=item hashref

Contains the following keys:  

'to' destination address of the message.

'from' source address of the message (not implemented)

'reply-to' yet another way of not using C<POE::Kernel::call()>.  The return
value of the method is sent back to this address.  'reply-to' superceeds
C<respond> (see JAAS::Object) unless you deactivate OOB.

If you have deactiveated OOB, via C<noOOB>, it uses C<call()> internally,
but ONLY SO THAT YOU DON'T HAVE TO.

Values of these keys can be either a plain string, or a hash with 3 pairs:
'ikc', 'object' and 'method'.  Yes, I should add kernel, though I currently
don't need to, because you can subscribe to the object via IKC.  'ikc' is a
flag indicating that this is an IKC address.
    

'wantarray' Used in conjunction with 'reply-to', takes a stab at creating an
array context for the method.  But, because C<POE::Kernel::call> only ever
provides scalar context, this means that a returned arrayref is converted into
an array before 'reply-to' is called.

'default_object'  If any of the addresses in the structure doesn't have a session, this one is
used.  This is used internally for 'respond'-type methods.  If not
specified, current object is used.

'noOOB'  By default, C<post> adds some information to the parameter list of 
the method in the hopes that either the recipient is a JAAS::Object which will
make use of the information, or it won't get confused.  

The following cases can confuse things :

If you don't post to an event w/ all parameters filled (even if it's just
undef(), one of the parameters could be the OOB data.  Or, if the method
accepts a variable amount of parameters :

 sub my_event {
    my @parms=@_[ARG0..$#_];
    ...
  }

Setting C<noOOB> will prevent this.  Note that if the destination is a
JAAS::Object, you don't have to worry.


=back

So, this means:

    'foo@bar'
    equiv to
    {   to=>{object=>'bar', method=>'foo'},
        default_object=>42,
    }


    'foo@poe://kernel/session'
    equiv to
    {   to=>{method=>'foo', object=>'poe://kernel/session', ikc=>1}, 
        default_object=>42,
    }

    {to=>'foo@bar', 'reply-to'=>'wibble'}
    equiv to 
    {   to=>{object=>'bar', method=>'foo'},
        'reply-to'=>{object=>42, method=>'wibble'},
        default_object=>42,
    }



=head2 continuation

    $jaas_services->continuation($req, @names);

Deffer responding to the current message to another method.  C<$req> can be
whatever would be passed to C<post>.  Things like 'from', 'reply-to' and
'wantarray' are propagate as one would expect.


=head2 sender

    my $id=$jaas_services->sender

Returns an opaque value that points back to the object/session that called
the current method.



=head2 debug

  BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () {0}; }
  }

Check the configuration to see if debuging is turned on for this package.



=head2 exec_ctx

    $jaas_services->exec_ctx({something=>1});   # new information

    $jaas_services->exec_ctx({});               # clear context

Allows outside objects to set values in the execution context.  This would
allow you to set extra information for logging/monitoring methods.



=head2 alarm_set

    my $id=$jaas_services->alarm_set($address, $time, @arguments);

Sets an alarm.  $address is a JAAS specifier as you would give to post.
$time is the epoch seconds at which point the $address will be posted.  The
ID that is returned can be used to clear the alarm later with
C<alarm_remove>.  

Yes, you can set alarms in other objects.  Please don't.  If you do it to 
remote objects, it won't work and I will bite you.

=head2 delay_set

    my $id=$jaas_services->delay_set($address, $seconds, @arguments);

Casses $address to be called in no less then $seconds seconds.  The ID that
is returned can be used to clear the alarm later with C<alarm_remove>.


=head2 alarm_remove

    $jaas_services->alarm_remove($id);

Kills the alarm or delay.  Kills it dead.  Gone.

=head2 alarm_adjust

    $jaas_services->alarm_adjust($id, $delta_seconds);

Adds $delta_seconds to the time at which $id was going to do something.  If
$delta_seconds is negative, this will be sooner.





=head1 MONITORING

You can set up monitoring of other objects via relationships.

    $jaas_relationships->nail('me', 'on_error', 'other_object', 'error@me');

So, when a method in 'other_object' throws an exception, 'error@me' gets
called with a secret-sacred-roach-hashref as the first argument.

Only 'on_error' is currently defined.  This form of monitoring is going to
change next week.







=head1 OVERLOADING

If you want to overload C<$jaas_services>'s behaviour, you have to do the
following:

    # define anything needed by new()
    BEGIN {
        $JAAS::Object::Services::jaas_services=new MyPackage;
    }

    use JAAS::Object::Services;

Note that the BEGIN block comes before the C<use>, and that
C<MyPackage::new> must be defined before the BEGIN block.

The following methods are also used internally:

=head2 _POE

Handed an arrayref containing all POE-specific parameters for a message. 
This is saved so that C<post()> will be able to post messages into
POE-space.

=head2 _OOB

Handed a harhref containing the OOB data for this parameter.  
This is saved so that C<post()> will be able to use it to implement various
featuers.

=head2 _parse

Turn a address into an address hashref.  It can be handed a simple string,
or a half filled-in hashref.  See C<post> to find out what an address
structure can look like.

=head2 monitor_inform

Read the source.

=head1 AUTHOR

Philip Gwyn <perl at pied.nu>

=head1 SEE ALSO

perl(1).

=cut
