# $Id: Context.pm,v 1.10 2001/10/27 03:15:24 fil Exp $
package JAAS::Object::Context;

use strict;
use POE;
use POE::Component::IKC::Responder;
use JAAS::Object::Cache::Simple;
use Storable qw(dclone);

use vars qw($VERSION);

$VERSION="0.02";

$JAAS::Object::Context::VERSION="0.2";
BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () {0}; }
}

############################################################
sub spawn 
{
    my($package, $name, $factory, $cache)=@_;

    $name   ||='Context';
    $factory||=JAAS::Object::Context::SimpleFactory->new();
    $cache  ||=JAAS::Object::Cache::Simple->new();

    POE::Session->create (
        args=>[$name, $factory, $cache],
        package_states => [
            $package => [qw(_start register unregister shutdown
                            setup clear delete
                            get release putback 
                            add remove
                            list locklist blocklist block do_block unblock)],
        ]
    );
}


############################################################
sub _start {
    my($kernel, $heap, $name, $factory, $cache)=
                    @_[KERNEL, HEAP, ARG0, ARG1, ARG2];
    if(DEBUG) {
        die "NO NAME" unless $name;
        die "NO FACTORY" unless $factory;
        die "NO CACHE" unless $cache;
    }

    $heap->{_factory}=$factory;
    $heap->{_cache}=$cache;

    DEBUG and warn "Context ($name) _start\n";

    $kernel->alias_set($name);
    POE::Component::IKC::Responder->spawn();
    $kernel->post(IKC=>'publish', 
        $name=> 
        [qw(setup clear delete 
            get release putback 
            add remove
            list locklist blocklist block unblock)]);

    $kernel->post('IKC', 'monitor', '*', {register=>'register'});

}


###########################################################
# Remote kernel connected to us
sub register {
    my ($kernel, $heap, $name, $cannonical)=@_[KERNEL, HEAP, ARG1, ARG2];
#    if ($cannonical) {    # Get only proper name (no dots!)
        $kernel->post('IKC' => 'monitor', 
                      $name => {unregister=>'unregister'});
        DEBUG and warn "$name has registered\n";
#        $heap->{_xlat}{$name}=$name;
#    } 
}

###########################################################
# remote kernel disconnected
sub unregister {
    my ($kernel, $heap, $session, $name)=@_[KERNEL, HEAP, SESSION, ARG0];

    # Release all locks held by the quitting client
    my $list = $kernel->call($session, 'list');
    foreach my $id (@$list) {
        # BRUTE FORCE IT!
        $kernel->yield('clear', {id=>$id, name=>$name});
    }
    DEBUG and warn "$name has quit\n";
}

###########################################################
# IKC is going DOWN
sub shutdown
{
    my($kernel, $session, $heap)=@_[KERNEL, SESSION, HEAP];
    DEBUG and warn "Shutdown of all contexts\n";
    $session=$session->ID if ref $session;
    my $list=$kernel->call($session, 'list');

    # remove all aliases
    foreach my $a ($kernel->alias_list($session)) {
        $kernel->alias_remove($a);
    }
    # retract all published states
    $kernel->post(IKC=>'retract');

    # equiv to clear() on all our "known" contexts
    $heap->{_cache}->shutdown($list) if @$list;
    delete @{$heap}{@$list};
}




###########################################################
# Someone wants to use a context.  
sub setup
{
    my ($kernel, $heap, $args)=@_[KERNEL, HEAP, ARG0];
    my $id=$args->{id};
    my $name=$args->{name};

    unless (exists $heap->{$id}) {
        my $ctx=$heap->{$id}={_block=>0, _refcount=>1, _name=>$name};

        DEBUG and 
            warn "Context: $id created (ref: 1)\n";
    } else {
        $heap->{$id}{_refcount}++;
        DEBUG and 
            warn "Context: $id already exists (ref: $heap->{$id}{_refcount})\n"
    }
    $heap->{_cache}->setup($id, $name);
}

###########################################################
# Someone no longer cares about a context
sub clear
{
    my ($kernel, $session, $heap, $args)=@_[KERNEL, SESSION, HEAP, ARG0];
    my $id=$args->{id};
    my $name=$args->{name};
    unless (exists $heap->{$id}) {
        DEBUG and warn "Context: $id never existed!\n";
        return;
    }

    $heap->{$id}{_refcount}--;
    if($heap->{$id}{_refcount} > 0) {
        DEBUG and 
            warn "Context: $id still wanted (ref: $heap->{$id}{_refcount})\n";
        return;
    } else {
        DEBUG and 
            warn "Context: $id no longer wanted (ref: $heap->{$id}{_refcount})\n";
    }

    # REMOVE ALL LOCKS
    $_[OBJECT]->clear_locks($kernel, $session, $heap, $id, $name);

    # clear all blocking
    if(($heap->{$id}{_blocked_by}||'') eq $name) {
        $kernel->yield(unblock=>{id=>$id, name=>$name});
    }

    # maybe clear from memory?
    if($heap->{_cache}->clear($id, $name)) {
        # Doing this causes a problem when
        $kernel->yield('delete', $id);
    }
}

###########################################################
# Kill a context (called when everyone does a clear/)
sub delete
{
    my($kernel, $heap, $session, $id)=@_[KERNEL, HEAP, SESSION, ARG0];
    $id=$id->{id} if ref $id;

    if(exists $heap->{$id}) {
        if($heap->{$id}{_refcount} > 0) {
            # This happens if a 'setup' comes in between 'clear' and 'delete'
            # It's a sucky way of doing it, but cleaner then alternatives.
            # Also, I don't think it will mess up the cache and persistance
            # DEBUG and 
                warn "Context: $id got a reprieve (ref: $heap->{$id}{_refcount})\n";
            return;
        }

        DEBUG and warn "Going to delete $id\n";
        my $ctx=$heap->{$id};
        $ctx->{_deleted}=1;

        if($ctx->{_blocked_by}) {           # make sure it's not blocked
            DEBUG and warn "Context: $id is blocked, deffering delete\n";
            return;
        }

        my $hold=0;
        my $objlist=$kernel->call($session=>'list', $id);
        foreach my $obj (@$objlist) {        # make sure it's not locked
            if($ctx->{$obj}{lock}) {
                DEBUG and 
                    warn "Context: $id/$obj is locked $ctx->{$obj}{lock}, deffering delete\n";
                $hold=1;
                next;
            }
            $heap->{_cache}->delete($id, $obj);
        }

        unless($hold) {
            delete $heap->{$id};
            DEBUG and
                warn "Context: $id deleted\n";
        }
    } else {
        # DEBUG and
            warn "Context: $id doesn't exist\n"
    }
}


###########################################################
# Put an object into the context
sub add
{
    die "Don't call me!";
    my($heap, $args)=@_[HEAP, ARG0];
    my $id = $args->{id};               # context ID
    my $obj = $args->{obj};             # object name
    my $ctx = $heap->{$id};
    my $name = $args->{name};
    return unless $ctx;
    my $work;

    if(($ctx->{_blocked_by}||$name) ne $name) {
        warn "Context $id isn't blocked by $name\n";
        return;
    }

    if(exists $ctx->{$obj}) {
        my $name = $args->{name};
        $work = $ctx->{$obj};    

        if(($work->{lock}||$name) ne $name) {
            warn "Object $id/$obj isn't locked by $name\n";
            return;
        }
    } else {
        $work=$heap->{$id}{$obj}||={};        # new object
        $work->{lock}='';
    }
     
    $heap->{_cache}->store($obj, $args->{in});
}

###########################################################
# Remove an object from the context
sub remove
{
    my($heap, $args)=@_[HEAP, ARG0];
    my $id = $args->{id};               # context ID
    my $obj = $args->{obj};             # object name
    my $ctx = $heap->{$id};
    return unless $ctx;
    my $work = $ctx->{$obj};
    my $name=$args->{name};

    if(($ctx->{_blocked_by}||$name) ne $name) {
        warn "Context $id isn't blocked by $name\n";
        return;
    }

    unless(($work->{lock}||$name) eq $name) {
        warn "Object $id/$obj isn't locked by $name\n";
        return;
    }
    $heap->{_cache}->delete($id, $obj);
    delete $heap->{$id}{$obj};
    DEBUG and warn "Context: deleting $id / $obj\n";
}

###########################################################
# Get or create an object in the context
sub get
{
    my ($kernel, $heap, $args)=@_[KERNEL, HEAP, ARG0];
    my $id=$args->{id};
    my $obj=$args->{obj};
    my $name=$args->{name};
    die "NO NAME" unless $name;

    DEBUG and 
        warn "Object: $id/$obj wanted for '$name'... ";

    my $ctx=$heap->{$id};
    unless($ctx) {
        # DEBUG and 
            warn "Context: $id doesn't exist\n";
        return -1;                              # ERR -1: no context
    }
    if($ctx->{_deleted}) {
        # DEBUG and 
            warn "Context: $id is deleted\n";
        return -1;                              # ERR -1: no context
    }

    unless(($ctx->{$obj}{lock} || $name) eq $name) {
        DEBUG and 
            warn "was locked by $ctx->{$obj}{lock}!\n";
        return 0;                               # ERR 0: Obj. locked
    }
    unless(($ctx->{_blocked_by} || $name) eq $name) {
        DEBUG and warn "context blocked by $ctx->{_blocked_by}!\n";
        return -2;                              # ERR -2: blocked
    }

    ######################################################
    my $out=$heap->{_cache}->fetch($id, $obj);
    unless($out or $args->{no_create}) {
        DEBUG and 
            warn "creating...";
        $out=$heap->{_factory}->make($obj);
        unless($out) {
            DEBUG and warn "can't!\n";
            DEBUG and warn "$!\n" if $!;
            DEBUG and warn "$@\n" if $@;
            warn $@ if $@;                      # TODO: monitoring
            return -1;                          # ERR -1: can't create
        }
        $heap->{_cache}->store($id, $obj, $out)
    }

    unless($out) {                              # ERR -1: Obj. doesn't exist
        # DEBUG and 
            warn "Context: $id/$obj doesn't exist\n";
        return -1;
    }
    DEBUG and warn "OK\n";

    $_[OBJECT]->get_lock($kernel, $heap, $id, $obj, $name);
    
    return dclone $out;
}

###########################################################
# Release locks on an object
sub release 
{
    my ($kernel, $heap, $args)=@_[KERNEL, HEAP, ARG0];
    my $name = $args->{name};                   # remote kernel
    my $id = $args->{id};                       # context ID
    my $ctx = $heap->{$id};                     # Context
    my $work = $ctx->{$args->{obj}};            # object

    unless($ctx) {
        DEBUG and warn "Context $id doesn't exist\n";
        return;
    }
    unless($work) {
        DEBUG and warn "Object: $id/$args->{obj} doesn't exist\n";
        return;
    }
    if ($work->{lock} ne $name) {
        DEBUG and warn "Object: $id/$args->{obj} isn't locked by $name\n";
        return;
    }

    $_[OBJECT]->release_lock($kernel, $heap, $id, $args->{obj}, $name);
}

###########################################################
# Return an object (modified) to the context.  Release locks.
sub putback {
    my ($kernel, $heap, $args)=@_[KERNEL, HEAP, ARG0];
    my $name = $args->{name};                   # remote kernel
    my $id = $args->{id};                       # context ID
    my $ctx = $heap->{$id};                     # Context
    my $obj = $args->{obj};                     # object name

    unless($ctx) {
        DEBUG and warn "Context $id doesn't exist\n";
        return;
    }
    my $new;
    unless(exists $ctx->{$obj}) {
        if($args->{new}) {
            $new=1;
        } else {
            DEBUG and warn "Object: $id/$obj doesn't exist\n";
            return;
        }
    }

    my $work = $ctx->{$obj}||={};                    # object
    if(not $work->{lock} and $args->{new}) {
        $new=1;
        $work->{lock}=$name;
    } 
    elsif(($work->{lock}||'') ne $name) {
        DEBUG and warn "Object: $id/$args->{obj} isn't locked by $name\n";
        return;
    }

    DEBUG and warn "Object: $id/$args->{obj} ", 
                    ($new ? "added" : "changed"), " by $name\n";

    $heap->{_cache}->store($id, $obj, $args->{in});

    $_[OBJECT]->release_lock($kernel, $heap, $id, $obj,  $name);
}




###########################################################
# Helper method for reserving a lock
sub get_lock
{
    my($package, $kernel, $heap, $id, $obj, $name)=@_;
    DEBUG and warn "Lock: $id/$obj reserved for $name\n";

    $heap->{$id}{$obj}{lock} ||= $name;
}

###########################################################
# Helper method for releasing a lock
sub release_lock
{
    my($package, $kernel, $heap, $id, $obj, $name)=@_;
    my $ctx=$heap->{$id};
    $ctx->{$obj}{lock}='';

    DEBUG and warn "Lock: $id/$obj released for $name\n";

    $kernel->yield('do_block', $id) if $ctx->{_blocked_by};
    $kernel->yield('delete', $id)   if $ctx->{_deleted};
}

###########################################################
# Helper method for releasing ALL locks
sub clear_locks
{
    my($package, $kernel, $session, $heap, $id, $name)=@_;
    # clear all locks
    my $locklist = $kernel->call($session, 'locklist', {id=>$id});
    foreach my $obj (keys %$locklist) {
        if ($locklist->{$obj} eq $name) {
            $package->release_lock($kernel, $heap, $id, $obj, $name);
        } elsif($locklist->{$obj}) {
            # DEBUG and warn "$name isn't $locklist->{$obj}\n";
        }
    }
}


###########################################################
# Helper method for listing locks
sub list {
    my ($kernel, $heap, $id)=@_[KERNEL, HEAP, ARG0];
    $id=$id->{id} if ref $id;

    return [grep !/^_/, keys %{$id ? $heap->{$id} : $heap}];
}
    
###########################################################
# Helper method for listing locks
sub locklist {
    my ($kernel, $heap, $args)=@_[KERNEL, HEAP, ARG0];
    my $id = $args->{id};               # context ID
    my %locklist = (_err => 0);
    if (exists $heap->{$id}) {
        my $objlist = $heap->{$id};
        foreach my $obj (keys %$objlist) {
            next if $obj =~ /^_/;
            my $work = $heap->{$id}->{$obj};
            $locklist{$obj} = $work->{lock} 
                            if ($work->{lock} || $args->{type});
        }
    }
    else {
        DEBUG and warn "Locklist: context $id doesn't exist\n";
        $locklist{_err} = 1;    
    }
    return \%locklist;
}



###########################################################
# Helper method for listing blocked
sub blocklist {
    my ($kernel, $session, $heap, $args)=@_[KERNEL, SESSION, HEAP, ARG0];
    my %list;
    foreach my $id (@{$kernel->call($session => 'list')}) {
        next unless $heap->{$id}{_blocked_by};
        $list{$id}=$heap->{$id}{_blocked_by};
    }
    return \%list;
}

###########################################################
# Allows someone to block an entire context
sub block {
    # $heap->{$id}->{_block} is the flag:
    # 0 = normal, unblocked
    # 1 = session in process of being blocked
    #   (i.e. gives no further obj and waits for return of used obj)
    # 2 = session completely blocked
    # $heap->{$id}->{_blocked_by} is remote ID of who did blocking

    my ($kernel, $heap, $session, $args)=@_[KERNEL, HEAP, SESSION, ARG0];
    my($id, $name)=@{$args}{qw(id name)};
    return unless exists $heap->{$id};
    my $ctx=$heap->{$id};

    $ctx->{_block_wait} = $kernel->call($session=>'list', $id);
    die "No name trying to block $id" unless $name;
    $heap->{$id}->{_blocked_by} = $name;

    $kernel->yield('do_block', $id);
}

##############################################################
# Do the actual work of blocking off a session
sub do_block
{
    my($kernel, $heap, $id)=@_[KERNEL, HEAP, ARG0];
    return unless exists $heap->{$id};
    my $ctx=$heap->{$id};
    my $name=$ctx->{_blocked_by};
    die "No name" unless $name;

    unless(@{$ctx->{_want_to_block}||[]}) {         # current list done
                                                    # continue deffered list
        return unless $ctx->{_block_wait};
        $ctx->{_want_to_block}=delete $ctx->{_block_wait};
    }

    my $list=$ctx->{_want_to_block};
    my $obj=shift @$list;
    my $work = $ctx->{$obj};

    if(($work->{lock}||$name) ne $name) {
        $ctx->{_block_wait}||=[];
        push @{$ctx->{_block_wait}}, $obj;
        DEBUG and warn "Block: $id/$obj was locked by $work->{lock}, deferring\n";
    } else {
        DEBUG and warn "Block: $id/$obj blocked for $name\n";
    }

    if(@$list) {
        $kernel->yield('do_block', $id)
    } else {
        if($ctx->{_block_wait}) {
            DEBUG and warn "Block: $id waiting for deferred locks\n";
        } else {
            DEBUG and warn "Block: $id fully blocked for $name\n";
            $kernel->yield('unblock', {id=>$id, name=>$name})
                if $ctx->{_unblock_wait};
            $kernel->yield('delete', $id) if $ctx->{_deleted};
        }
        delete $ctx->{_want_to_block};
        return;
    }
}           

##############################################################
# Remove the block on a context
sub unblock {
    my ($kernel, $heap, $args)=@_[KERNEL, HEAP, ARG0];
    my($id, $name)=@{$args}{qw(id name)};

    return unless exists $heap->{$id};
    my $ctx=$heap->{$id};
    return unless $ctx->{_blocked_by};

    unless($ctx->{_blocked_by} eq $name) {
        DEBUG and warn "Unblock: $id wasn't blocked by $name\n";
        return;
    }

    if($ctx->{_block_wait} or $ctx->{_want_to_block}) {
        # Wait for blocking to end
        DEBUG and warn "Unblock: Waiting for full blocking...\n";
        $ctx->{_unblock_wait}=1;
    } else {
        delete @{$ctx}{qw(_block _unblock_wait
                                    _blocked_by _want_to_block _block_wait)}; 
        DEBUG and warn "Unblock: $id unblocked\n";
        $kernel->yield('delete', $id) if $ctx->{_deleted};
    }
}




##############################################################################
package JAAS::Object::Context::SimpleFactory;
use strict;

sub new { bless {}, $_[0]}
use Carp;
sub make
{
    my($self, $name)=@_;
    return if exists $self->{deleted}{$name};
    return eval{$name->new()};
}
sub delete
{
    my($self, $name)=@_;
    $self->{deleted}{$name}=1;
}






1;
__END__

=head1 NAME

JAAS::Object::Context - JAAS object repository

=head1 SYNOPSIS

    # on the server side:
    JAAS::Object::Context->spawn('Context', $factory);

    # On the client side, subscribe to the Context session in the server
    # aka poe://Server/Context;

    # Create a new context
    $kernel->post('poe://Server/Context' => 'create', $ID);
    # $ID is a unique ID for this application instance

    # Get an object from Context:
    $kernel->post(IKC=>'call', 'poe://Server/Context/get',
                {id=>'Session1234', obj=>'Cart', name=>'Client1'}, 
                    'callback');
    # now callback is called and ARG0 is the object

    # Give the object back to Context:
    $kernel->post('poe://Server/Context'=>'putback',
        {id=>'Session1234', obj=>'Cart', name=>'Client1', in=>$object});

    # Unlock an object after use:
    $kernel->post('poe://Server/Context'=>'release',
        {id=>'Session1234', obj=>'Cart', name=>'Client1'});

    # Get a list of context IDs currently running:
    $kernel->post(IKC=>'call', 'poe://Server/Context'=>'list',  
                        '', 'poe:/me/callback');
    # or
    @list=$kernel->call('poe://Server/Context'=>'list');

    # Get a hash of locked objects:
    $kernel->call('IKC', 'call', 'poe://Server/Context/locklist',
                 {id=>$ID}, 'poe:/me/callback');

    # Get a hash of all objects, locked or not:
    $kernel->call('IKC', 'call', 'poe://Server/Context/locklist',
                 {id=>'Session1234', type=>1}, 'poe:/me/callback');

    # Block a context:
    $kernel->post('poe://Server/Context'=>'block', $ID);

    # Unblock a context:
    $kernel->post('poe://Server/Context'=>'unblock', $ID);


=head1 DESCRIPTION

JAAF::Object::Context is a POE IKC server acting as an object repository.
Clients connect to it, setup working contexts, get objects, work on them and
put them back in the repository, then clear the context.  While a client has
an object, a lock is registered under its name, preventing anyone else from
getting the object.  When a client disconnects, all its locks are
automatically released and all interest is cleared.

Being "interested" is a silly pattern : when an external session wants to
use a context, it first posts 'setup'.  The server will create the context
if needed.  The session will then play with objects in the session as
needed.  When done, the session calles 'clear' to revoke it's interest. 
When no one is interested in the context, it is deleted from memory.

The 'working context' is a collection of objects, always the same, as
created by C<$factory>.  Each context is maintained by the C<$cache>. For
example, an e-comm config could define objects 'Cart', 'User' and 'Catalog';
any new context would create new instances of these objects as needed (via
C<$factory>) and store in C<$cache> under a particular context name. 
Persistance is implemented via the C<$cache>.

Because no magic is done about syncing objects across proccess,
JAAS::Object::Context should only be spawned in a non-forking server.


=head1 METHODS

=head2 spawn

    JAAS::Object::Context->spawn($name, $factory, $cache);

Creates the a session to act at the object repository.  C<$name> is the name
of the session.  C<$factory> is an object that will be used to create and
configure objects in the contexts.  If it is omited, a very simple object is
used, that assumes that object names are packages and calls C<new> on them.
C<$cache> is an object that will be used to store objects related to a
context.  If it is omited, a very simple object is used, that keeps
everything in a hash and deletes all objects when no one is interested in the
context anymore.


=head1 STATES

=head2 setup

    $kernel->call('IKC', 'post', 'poe://Server/Context/setup', 
                {id=>$ID, name=>$me});

C<$me> wants to use a context called C<$ID>.  A new context is created if it 
doesn't exist.  Context is kept around until everyone has "cleared" their
interest.  This is a crude form of GC.

=head2 clear

    $kernel->call('IKC', 'post', 'poe://Server/Context/setup', 
                {id=>$ID, name=>$me});

C<$me> no longer wants to use a context called C<$ID>.  This means that the
repository can get it out of memory (depending on the cache).


=head2 delete

    $kernel->post('IKC', 'post', 'poe://Server/Context/delete', $ID)

Purposefully delete a context.  This holds until all blocks and locks are
removed, including your own.


=head2 shutdown

    $kernel->post(Context=>'shutdown');

Removes the Context session from the kernel.  This also calls C<shutdown()>
on the cache.

=head2 get

    $kernel->call('IKC', 'call', 'poe://Server/Context/get',
            {id=>$ID, obj=>$obj, name=>$name}, 'poe:/me/callback');

Gets object named C<$obj> from context C<$ID> on Context, then passed on to
the callback handler. C<get> locks the object under the name C<$name>. If
the object is already locked or if the entire context was blocked, C<get>
returns 0 to callback.  If the object doesn't exist, or the context doesn't,
or the factory can't create the object, callback is called with -1.

If you add C<no_create> to the arguements, no object will be created.

=head2 putback

        $kernel->post('IKC', 'post', 'poe://Server/Context/putback',
                {id=>$ID, obj=>$obj, name=>$name, in=>$frozen_object});

Gives the object back to Context.  A C<putback> automatically unlocks the
object.

=head2 release

        $kernel->post('IKC', 'post', 'poe://Server/Context/release',
                {id=>$ID, obj=>$obj, name=>$name});

Releases the lock on an object. This message is preferred to C<putback> if
the client hasn't modified anything on the object.

=head2 list

        $kernel->post('IKC', 'post', 'poe://Server/Context/list',
                [$id], 'callback');

Fetches a list of active context IDs on Context, as an arrayref, to the
callback handler.  If C<$id> is given, fetches a list of active objects in
that context.

=head2 locklist

    $kernel->call('IKC', 'call', 'poe://Server/Context/locklist',
         {id=>$ID, type=>$type}, 'callback');

Gives a list of active objects contained in C<$ID>, as a hashref, to the
callback handler. There are two types of lists, as specified by the C<type>
argument.

=item Type 0

Returns a hash of locked object names as keys, owner names as values.

=item Type 1

Returns a hash of all objects, in the same format as Type 0, but with empty
values for unlocked objects.

=item Error Handling

In the hash for both types is a C<_err> key that contains 1 if the requested context
doesn't exist, 0 otherwise. So if you iterate through the keys of the
hash, be sure to skip C<{_err}>.


=head2 block

    $kernel->call('IKC', 'post', 'poe://Server/Context/block', 
            {id=>$ID, name=>$name});

Blocks a whole context by preventing any client from getting objects. A
context isn't fully locked until all of the object-owning clients to unlock
their objects.  As soon as a client releases a lock, the blocks is retried.
One can not get a new lock while a block is in place.

'block' is usually called before shutting down a context, so a process (not
implemented) could collect data and kill the context properly. It could also
be used to 'pause' a context while doing something else, as you can
'unblock' contexts (see below).

=head2 unblock

    $kernel->call('IKC', 'post', 'poe://Server/Context/block', 
            {id=>$ID, name=>$name});

Unblocks a context that was previously blocked.  If the blocking event
hasn't finished its business (i.e. still waits for clients to release their
lock), it waits for it to finish. It thus ensures that all locks are clear
after an 'unblock'.

=head1 CLIENT BEHAVIOR

These are the guidelines for the clients connecting to the Context server.

- If the client is in a different process, it must C<use> the packages for
the objects requested from Context, so it can use their methods.

- A client should create for itself a unique name, like
'C<catalog-lookup$$>' for example. That name _must_ be passed in the
subscription phase, in the C<create_ikc_client> method, as it will be used
for unlocking objects when the client disconnects.

- Never use names starting with underscore (_) for ID and client names,
these are reserved names for Context. Unless you're a hacker. A good one.

=head1 BUGS

Clients are expected to behave properly, as error-handling is VERY
primitive, at times non-existant. In particular, they are expected to get
their session ID and object names right.

As The Rock says, "Know Your Role!".

=head1 AUTHOR

Learry Gagne <gagnel@artware.qc.ca>, 
Philip Gwyn <gwynp@artware.qc.ca>

=cut

$Log: Context.pm,v $
Revision 1.10  2001/10/27 03:15:24  fil
Fixed some warning messages.

Revision 1.9  2001/10/11 02:16:56  fil
Added comments
Added refcounts to setup/clear so that a context can be setup after a clear
but before being deleted from memory.

Revision 1.8  2001/09/27 04:35:05  fil
context/get now dclones the objects it returns, to avoid references messing
w/ the cache.

Revision 1.7  2001/09/21 18:52:15  fil
Debugging mess

Revision 1.6  2001/09/20 01:10:44  fil
Added no_create parameter

Revision 1.5  2001/09/13 00:37:12  fil
Fixed Persistant so that it doesn't look like a context exists
even if it was only quieried t/20_cache.t

Revision 1.4  2001/08/09 16:35:11  fil
Small tweaks

Revision 1.3  2001/08/09 03:29:45  fil
Added {new} to Context/putback
Context/delete is broken.  yow!

Revision 1.2  2001/07/27 21:41:20  fil
Minor debuging changes.

Revision 1.1  2001/07/24 21:12:31  fil
Fibble

Revision 1.3  2001/07/05 23:56:37  fil
ID in Context->list could be a hash ref
Doco fix in Cache::Persistant
Moved Factory over to JAAS::Config

Revision 1.2  2001/06/22 20:42:20  fil
Added Factory, Cache/*
Reworked internals... Still have problems, though.

Revision 1.1  2001/06/21 04:11:19  fil
Totaly rearranged things:
    Context.pl is now Context.pm and server.perl
    Blocking and locking now are revoked gracefully when remote closes
    Blocking loop is now unrolled.
    Removed all the delay()s.  When something is deffered, we check again
    when the condition is encountered.

