# $Id$
package JAAS::Object::Context;

use strict;
use POE;
use POE::Component::IKC::Responder;

sub DEBUG {0};

$JAAS::Object::Context::VERSION="0.1";

sub spawn 
{
    my($package, $name, $config)=@_;
    $name||='Context';
    POE::Session->create (
        args=>[$name, $config],
        package_states => [
            $package => [qw(_start register unregister shutdown
                            create delete
                            get release putback 
                            add remove
                            list locklist blocklist block do_block unblock)],
        ]
    );
}


############################################################
sub _start {
    my ($kernel, $heap, $name, $config)=@_[KERNEL, HEAP, ARG0, ARG1];
    if(ref $config eq 'ARRAY') {
        my %conf;
        @conf{@$config}=@$config;
        $config=\%conf;
    }
    $heap->{_config}=$config;

    DEBUG and print "Context ($name) _start\n";

    $kernel->alias_set($name);
    POE::Component::IKC::Responder->spawn();
    $kernel->post(IKC=>'publish', 
        $name=> 
        [qw(create delete 
            get release putback 
            add remove
            list locklist blocklist block unblock)]);

    $kernel->post('IKC', 'monitor', '*', {register=>'register'});

}


###########################################################
sub register {
    my ($kernel, $heap, $name, $cannonical)=@_[KERNEL, HEAP, ARG1, ARG2];
#    if ($cannonical) {    # Get only proper name (no dots!)
        $kernel->post('IKC' => 'monitor', 
                      $name => {unregister=>'unregister'});
        DEBUG and print "$name has registered\n";
#        $heap->{_xlat}{$name}=$name;
#    } 
}

###########################################################
sub unregister {
    my ($kernel, $heap, $session, $name)=@_[KERNEL, HEAP, SESSION, ARG0];

    # Release all locks held by the quitting client
    my $list = $kernel->call($session, 'list');
    foreach my $id (@$list) {
        my $locklist = $kernel->call($session, 'locklist', {id=>$id});
        foreach my $obj (keys %$locklist) {
            if ($locklist->{$obj} eq $name) {
                $kernel->yield(release=>{id=>$id, obj=>$obj, name=>$name});
            } elsif($locklist->{$obj}) {
                # DEBUG and print "$name isn't $locklist->{$obj}\n";
            }
        }
        if(($heap->{$id}{_blocked_by}||'') eq $name) {
            $kernel->yield(unblock=>{id=>$id, name=>$name});
        }
    }
    DEBUG and print "$name has quit\n";
}
###########################################################
sub create {
    my ($kernel, $heap, $id)=@_[KERNEL, HEAP, ARG0];
    $id=$id->{id} if ref $id;
    unless (exists $heap->{$id}) {
        my $ctx=$heap->{$id}={};
        foreach my $obj (keys %{$heap->{_config}}) {
            $ctx->{$obj}->{objref} = $heap->{_config}{$obj}->new();
            $ctx->{$obj}->{lock} = '';  
        }
        $ctx->{_block} = 0;
        DEBUG and print "Context: $id created\n";
    } else {
        DEBUG and print "Context: $id already exsists\n"
    }
}

###########################################################
sub delete
{
    my($kernel, $heap, $session, $id)=@_[KERNEL, HEAP, SESSION, ARG0];
    $id=$id->{id} if ref $id;

    if(exists $heap->{$id}) {
        DEBUG and print "Going to delete $id\n";
        my $ctx=$heap->{$id};
        $ctx->{_deleted}=1;

        if($ctx->{_blocked_by}) {           # make sure it's not blocked
            DEBUG and print "Context: $id is blocked, deffering delete\n";
            return;
        }
        
        my $objlist=$kernel->call($session=>'list', $id);
        foreach my $obj (@$objlist) {        # make sure it's not locked
            next unless $ctx->{$obj}{lock};
            DEBUG and print "Context: $id/$obj is locked $ctx->{$obj}{lock}, deffering delete\n";
            return;
        }

        delete $heap->{$id};
        DEBUG and print "Context: $id deleted\n";
    } else {
        DEBUG and print "Context: $id doesn't exsist\n"
    }
}

###########################################################
sub shutdown
{
    my($kernel, $session, $heap)=@_[KERNEL, SESSION, HEAP];
    DEBUG and print "Shutdown of all contexts\n";
    $session=$session->ID if ref $session;
    foreach my $a ($kernel->alias_list($session)) {
        $kernel->alias_remove($a);
    }
    $kernel->post(IKC=>'retract');
    $heap={};
}


###########################################################
sub add
{   
    my($heap, $args)=@_[HEAP, ARG0];
    my $id = $args->{id};               # context ID
    my $obj = $args->{obj};             # object name
    my $ctx = $heap->{$id};
    my $name = $args->{name};
    return unless $ctx;
    my $work;

    if(($ctx->{_blocked_by}||$name) ne $name) {
        warn "Context $id isn't blocked by $name\n";
        return;
    }

    if(exists $ctx->{$obj}) {
        my $name = $args->{name};
        $work = $ctx->{$obj};    

        if(($work->{lock}||$name) ne $name) {
            warn "Object $id/$obj isn't locked by $name\n";
        } else {
            $work->{objref}=$args->{in};
        }
        return;
    }

    $work=$heap->{$id}{$obj}={};        # new object
    $work->{lock}='';
    $work->{objref}=$args->{in};
}

###########################################################
sub remove
{   
    my($heap, $args)=@_[HEAP, ARG0];
    my $id = $args->{id};               # context ID
    my $obj = $args->{obj};             # object name
    my $ctx = $heap->{$id};
    return unless $ctx;
    my $work = $ctx->{$obj};    
    my $name=$args->{name};

    if(($ctx->{_blocked_by}||$name) ne $name) {
        warn "Context $id isn't blocked by $name\n";
        return;
    }

    unless(($work->{lock}||$name) eq $name) {
        warn "Object $id/$obj isn't locked by $name\n";
        return;
    }
    delete $heap->{$id}{$obj};
}

###########################################################
sub get 
{
    my ($kernel, $heap, $args)=@_[KERNEL, HEAP, ARG0];

    my $name = $args->{name};
    DEBUG and print "Object: $args->{id}/$args->{obj} wanted by $name... ";

    my $ctx=$heap->{$args->{id}};
    
    my $work = $ctx->{$args->{obj}};
    unless($work->{objref} or $ctx->{_deleted}) { # ERR -1: Obj. doesn't exist
        DEBUG and print "doesn't exists\n";
        return -1;
    }

    unless(($work->{lock} || $name) eq $name) {
        DEBUG and print "was locked by $work->{lock}!\n";
        return 0;                                   # ERR 0: Obj. locked 
    }

    unless(($ctx->{_blocked_by} || $name) eq $name) {    # ERR -2: blocked
        DEBUG and print "context blocked by $ctx->{_blocked_by}!\n";
        return -2;
    }

    $work->{lock} ||= $name;
    DEBUG and print "OK\n";
    return $work->{objref};
}

###########################################################
sub release 
{
    my ($kernel, $heap, $args)=@_[KERNEL, HEAP, ARG0];
    my $name = $args->{name};                   # remote kernel
    my $id = $args->{id};                       # context ID
    my $ctx = $heap->{$id};                     # Context
    my $work = $ctx->{$args->{obj}};            # object

    unless($ctx) {
        DEBUG and print "Context $id doesn't exist\n";
        return;
    }
    unless($work) {
        DEBUG and print "Object $id/$args->{obj} doesn't exist\n";
        return;
    }
    if ($work->{lock} ne $name) {
        DEBUG and print "Object $id/$args->{obj} isn't locked by $name\n";
        return;
    }

    $work->{lock}='';
    $work->{objref}=$args->{in} if $args->{in};
    DEBUG and print "Object: $id/$args->{obj} released for $name\n";

    $kernel->yield('do_block', $id) if $ctx->{_blocked_by};
    $kernel->yield('delete', $id)   if $ctx->{_deleted};
}

###########################################################
sub putback {
    my ($kernel, $heap, $args)=@_[KERNEL, HEAP, ARG0];
    my $name = $args->{name};                   # remote kernel
    my $id = $args->{id};                       # context ID
    my $ctx = $heap->{$id};                     # Context
    my $work = $ctx->{$args->{obj}};            # object

    unless($ctx) {
        DEBUG and print "Context $id doesn't exist\n";
        return;
    }
    unless($work) {
        DEBUG and print "Object $id/$args->{obj} doesn't exist\n";
        return;
    }
    unless($work->{lock} eq $name) {
        DEBUG and print "Object $id/$args->{obj} isn't locked by $name\n";
        return;
    }

    DEBUG and print "Object: $id/$args->{obj} changed by $name\n";
    $work->{objref} = $args->{in};
    $kernel->yield('release', {id=>$id, obj=>$args->{obj}, name=>$name});
}


###########################################################
sub list {
    my ($kernel, $heap, $id)=@_[KERNEL, HEAP, ARG0];
    
    return [grep !/^_/, keys %$heap] unless $id;

    return [grep !/^_/, keys %{$heap->{$id}}];
}
    
###########################################################
sub locklist {
    my ($kernel, $heap, $args)=@_[KERNEL, HEAP, ARG0];
    my $id = $args->{id};               # context ID
    my %locklist = (_err => 0);
    if (exists $heap->{$id}) {
        my $objlist = $heap->{$id};
        foreach my $obj (keys %$objlist) {
            next if $obj =~ /^_/;
            my $work = $heap->{$id}->{$obj};
            $locklist{$obj} = $work->{lock} 
                            if ($work->{lock} || $args->{type});
        }
    }
    else {
        DEBUG and print "Locklist: context $id doesn't exist\n";
        $locklist{_err} = 1;    
    }
    return \%locklist;
}

###########################################################
sub blocklist {
    my ($kernel, $session, $heap, $args)=@_[KERNEL, SESSION, HEAP, ARG0];
    my %list;
    foreach my $id (@{$kernel->call($session => 'list')}) {
        next unless $heap->{$id}{_blocked_by};
        $list{$id}=$heap->{$id}{_blocked_by};
    }
    return \%list;
}

###########################################################
sub block {
    # $heap->{$id}->{_block} is the flag:
    # 0 = normal, unblocked
    # 1 = session in process of being blocked
    #   (i.e. gives no further obj and waits for return of used obj)
    # 2 = session completely blocked
    # $heap->{$id}->{_blocked_by} is remote ID of who did blocking

    my ($kernel, $heap, $session, $args)=@_[KERNEL, HEAP, SESSION, ARG0];
    my($id, $name)=@{$args}{qw(id name)};
    return unless exists $heap->{$id};
    my $ctx=$heap->{$id};

    $ctx->{_block_wait} = $kernel->call($session=>'list', $id);
    die "No name trying to block $id" unless $name;
    $heap->{$id}->{_blocked_by} = $name;

    $kernel->yield('do_block', $id);
}

##############################################################
sub do_block
{
    my($kernel, $heap, $id)=@_[KERNEL, HEAP, ARG0];
    return unless exists $heap->{$id};
    my $ctx=$heap->{$id};
    my $name=$ctx->{_blocked_by};
    die "No name" unless $name;

    unless(@{$ctx->{_want_to_block}||[]}) {         # current list done
                                                    # continue deffered list
        return unless $ctx->{_block_wait};
        $ctx->{_want_to_block}=delete $ctx->{_block_wait};
    }

    my $list=$ctx->{_want_to_block};
    my $obj=shift @$list;
    my $work = $ctx->{$obj};

    if(($work->{lock}||$name) ne $name) {
        $ctx->{_block_wait}||=[];
        push @{$ctx->{_block_wait}}, $obj;
        DEBUG and print "Block: $id/$obj was locked by $work->{lock}, deferring\n";
    } else {
        DEBUG and print "Block: $id/$obj blocked for $name\n";
    }

    if(@$list) {
        $kernel->yield('do_block', $id)
    } else {
        if($ctx->{_block_wait}) {
            DEBUG and print "Block: $id waiting for deferred locks\n";
        } else {
            DEBUG and print "Block: $id fully blocked for $name\n";
            $kernel->yield('unblock', {id=>$id, name=>$name})
                if $ctx->{_unblock_wait};
            $kernel->yield('delete', $id) if $ctx->{_deleted};
        }
        delete $ctx->{_want_to_block};
        return;
    }
}           

##############################################################
sub unblock {
    my ($kernel, $heap, $args)=@_[KERNEL, HEAP, ARG0];
    my($id, $name)=@{$args}{qw(id name)};

    return unless exists $heap->{$id};
    my $ctx=$heap->{$id};
    return unless $ctx->{_blocked_by};

    unless($ctx->{_blocked_by} eq $name) {
        DEBUG and print "Unblock: $id wasn't blocked by $name\n";
        return;
    }

    if($ctx->{_block_wait} or $ctx->{_want_to_block}) {
        # Wait for blocking to end
        DEBUG and print "Unblock: Waiting for full blocking...\n";
        $ctx->{_unblock_wait}=1;
    } else {
        delete @{$ctx}{qw(_block _unblock_wait
                                    _blocked_by _want_to_block _block_wait)}; 
        DEBUG and print "Unblock: $id unblocked\n";
        $kernel->yield('delete', $id) if $ctx->{_deleted};
    }
}
            
1;
__END__

=head1 NAME

JAAS::Object::Context - JAAS object repository

=head1 SYNOPSIS

    # on the server side:
    JAAS::Object::Context->spawn('Context', {Cart=>"My::Shop::Cart"});

    # On the client side, subscribe to the Context session in the server
    # aka poe://Server/Context;

    # Create a new context
    $kernel->post('poe://Server/Context' => 'create', $ID);
    # $ID is a unique ID for this application instance

    # Get an object from Context:
    $kernel->post(IKC=>'call', 'poe://Server/Context/get',
                {id=>'Session1234', obj=>'Cart', name=>'Client1'}, 
                    'callback');
    # now callback is called and ARG0 is the object

    # Give the object back to Context:
    $kernel->post('poe://Server/Context'=>'putback',
        {id=>'Session1234', obj=>'Cart', name=>'Client1', in=>$object});

    # Unlock an object after use:
    $kernel->post('poe://Server/Context'=>'release',
        {id=>'Session1234', obj=>'Cart', name=>'Client1'});

    # Get a list of context IDs currently running:
    $kernel->post(IKC=>'call', 'poe://Server/Context'=>'list',  
                        '', 'poe:/me/callback');
    # or
    @list=$kernel->call('poe://Server/Context'=>'list');

    # Get a hash of locked objects:
    $kernel->call('IKC', 'call', 'poe://Server/Context/locklist',
                 {id=>$ID}, 'poe:/me/callback');

    # Get a hash of all objects, locked or not:
    $kernel->call('IKC', 'call', 'poe://Server/Context/locklist',
                 {id=>'Session1234', type=>1}, 'poe:/me/callback');

    # Block a context:
    $kernel->post('poe://Server/Context'=>'block', $ID);

    # Unblock a context:
    $kernel->post('poe://Server/Context'=>'unblock', $ID);


=head1 DESCRIPTION

JAAF::Object::Context is a POE IKC server acting as an object repository.
Clients connect to it, create working contexts, get objects, work on them
and put them back in the repository. While a client has an object, a lock is
registered under its name, preventing anyone else from getting the object.
When a client disconnects, all its locks are automatically released.

The 'working context' is a collection of objects, always the same, as
defined in a config file (currently not implemented). For example, an e-comm config
could define objects 'Cart', 'User' and 'Catalog'; any new context would
create new instances of these objects under a particular context name.
Parameters for their new() constructor, if any, should be specified in the
config file. The client needs a 'use' statement for every object imported
from Context, so it can use its methods.

Because no magic is done about syncing objects across proccess,
JAAS::Object::Context should only be spawned in a non-forking server.


=head1 METHODS

=head2 spawn

    JAAS::Object::Context->spawn($name, $config);

Creates the a session to act at the object repository.  C<$name> is the name
of the session.  C<$config> is the object that will be used to create and
configure objects in the contexts.  


=head1 STATES

=head2 create

    $kernel->call('IKC', 'post', 'poe://Server/Context/create', $ID);

Creates a new working context called C<$ID> on Context. This creates
new holder of objects as specified in the initial config file which Context
loads at startup. If the context ID already exists, C<create> does nothing.

=head2 delete

    $kernel->post('IKC', 'post', 'poe://Server/Context/delete', $ID)


=head2 shutdown

    $kernel->post(Context=>'shutdown');

Removes the session for the kernel.

=head2 get

    $kernel->call('IKC', 'call', 'poe://Server/Context/get',
            {id=>$ID, obj=>$obj, name=>$name}, 'poe:/me/callback');

Gets object named C<$obj> from context C<$ID> on Context, then passed on to
the callback handler. C<get> locks the object under the name C<$name>. If
the object is already locked or if the entire context was blocked, C<get>
returns 0 to callback.  If the object doesn't exsist, or the context, even,
callback is called with -1.

=head2 putback

        $kernel->post('IKC', 'post', 'poe://Server/Context/putback',
                {id=>$ID, obj=>$obj, name=>$name, in=>$frozen_object});

Gives the object back to Context. Object must be previously frozen using
POE::Filter::Reference. A C<putback> automatically unlocks the object.

=head2 release

        $kernel->post('IKC', 'post', 'poe://Server/Context/release',
                {id=>$ID, obj=>$obj, name=>$name});

Releases the lock on an object. This handler is preferred to C<putback> if
the client hasn't modified anything on the object.

=head2 list

        $kernel->post('IKC', 'post', 'poe://Server/Context/list',
                '', 'callback');

Fetches a list of active context IDs on Context, as an arrayref, to the
callback handler.

=head2 locklist

    $kernel->call('IKC', 'call', 'poe://Server/Context/locklist',
         {id=>$ID, type=>$type}, 'callback');

Gives a list of objects contained in C<$ID>, as a hashref, to the callback
handler. There are two types of lists, as specified by the C<type> argument.

=item Type 0

Returns a hash of locked object names as keys, owner names as values.

=item Type 1

Returns a hash of all objects, in the same format as Type 0, but with empty
values for unlocked objects.

=item Error Handling

In the hash for both types is a C<_err> key that contains 1 if the requested context
doesn't exist, 0 otherwise. So if you iterate through the keys of the
hash, be sure to skip C<{_err}>.


=head2 block

    $kernel->call('IKC', 'post', 'poe://Server/Context/block', 
            {id=>$ID, name=>$name});

Blocks a whole context by preventing any client from getting objects. A
context isn't fully locked until all of the object-owning clients to unlock
their objects.  As soon as a client releases a lock, the blocks is retried.
One can not get a new lock while a block is in place.

'block' is usually called before shutting down a context, so a process (not
implemented) could collect data and kill the context properly. It could also
be used to 'pause' a context while doing something else, as you can
'unblock' contexts (see below).

=head2 unblock

    $kernel->call('IKC', 'post', 'poe://Server/Context/block', 
            {id=>$ID, name=>$name});

Unblocks a context that was previously blocked.  If the blocking event
hasn't finished its business (i.e. still waits for clients to release their
lock), it waits for it to finish. It thus ensures that all locks are clear
after an 'unblock'.

=head1 CLIENT BEHAVIOR

These are the guidelines for the clients connecting to the Context server.

- Clients must C<use> the packages for the objects requested from Context,
so they can use their methods.

- A client should create for itself a unique name, like 'C<catalog-lookup$$>' for
example. That name _must_ be passed in the subscription phase, in the
C<create_ikc_client> method, as it will be used for unlocking objects when
the client disconnects.

- Don't forget to C<use POE::Filter::Reference>, as C<get> and C<putback> will
need it.

- Never use names starting with underscore (_) for ID and client
names, these are reserved names for Context. Unless you're a hacker. A good
one.

=head1 BUGS

Clients are expected to behave properly, as error-handling is VERY
primitive, at times non-existant. In particular, they are expected to get
their session ID and object names right.

As The Rock says, "Know Your Role!".

=head1 AUTHOR

Learry Gagne <gagnel@artware.qc.ca>, 
Philip Gwyn <gwynp@artware.qc.ca>

=cut

$Log$
