# $Id: Persistant.pm,v 1.1 2001/06/22 20:44:07 fil Exp $
package JAAS::Object::Cache::Persistant;

use strict;
use JAAS::Object::Cache::Simple;

sub DEBUG {0}
use Carp;

###########################################################
sub new
{   
    my($package, $lifetimer)=@_;
    return bless {life=>$lifetimer,
                  interest=>{},
                  obj=>{}}, $package;
}

###########################################################
sub setup
{
    my($self, $id, $who)=@_;
    confess "No ID" unless $id;

    my $ret=0;
    DEBUG and warn "$who is interested in $id";
    unless($self->{interest}{$id}) {
        DEBUG and warn "restoring $id for $who";
        $self->{obj}{$id}={};
        $self->{life}->restore($self->{obj}{$id}, 'context', $id);
        $ret=1;
    }
    $self->{interest}{$id}{$who}=1;
    return $ret;
}

###########################################################
sub clear
{
    my($self, $id, $who)=@_;
    return unless exists $self->{interest}{$id}{$who};

    delete $self->{interest}{$id}{$who};
    DEBUG and warn "$who isn't interested in $id";

    return if keys %{$self->{interest}{$id}};

    DEBUG and warn "$id is GONE";
    # no more interest
    $self->{life}->save($self->{obj}{$id}, 'context', $id);
    
    delete $self->{obj}{$id};
    delete $self->{interest}{$id};
    return 1;
}

###########################################################
sub shutdown
{
    my($self, $ids)=@_;
    foreach my $id (@$ids) {
        foreach my $who (keys %{$self->{_interested}}) {
            $self->clear($id, $who);
        }
    }
}

###########################################################
sub fetch
{
    my($self, $id, $name)=@_;
    confess "No ID" unless $id;
    return unless exists $self->{obj}{$id}{$name};
    return $self->{obj}{$id}{$name}->[0] unless wantarray;
    return @{$self->{obj}{$id}{$name}};
}

###########################################################
sub store
{
    my($self, $id, $name, $data)=@_;
    confess "No ID" unless $id;
    $self->{obj}{$id}{$name}||=[];

    if(ref $data eq 'ARRAY') {
        DEBUG and warn "Adding ARRAY to $id/$name";
        return push @{$self->{obj}{$id}{$name}}, $data;
    }

    # use Data::Dumper;
    # warn Dumper $self->{obj}{$id};

    $self->{obj}{$id}{$name}->[0]=$data;
}



###########################################################
sub delete
{
    my($self, $id, $name)=@_;
    confess "No ID" unless $id;
    return delete $self->{obj}{$id}{$name};
}

###########################################################
sub list
{
    my($self, $id)=@_;
    return [keys %{$id ? ($self->{obj}{$id}||{}) : $self->{obj}}] unless $id;
}

sub _status
{
    my($self, $id)=@_;
    if($id) {
        return join "\n", "Contains of $id:",
                        map {"    $_ ".@{$self->{obj}{$id}{$_}}}
                        keys %{$self->{obj}{$id}};
    }
    return join "\n    ", "Contains contexts:", keys %{$self->{obj}};
}

###########################################################

1;

__DATA__

=head1 NAME

JAAS::Object::Cache::Persistant - Persistant object repository cache

=head1 SYNOPSIS

    my $cache=JAAS::Object::Cache::Persistant($lifetime);
    $cache->setup($id);
    $cache->store($id, $name, $obj);
    $cache->clear($id);

    $cache->setup($id);                     
    $obj$cache->fetch($id, $name);      # $id/$name still present
    $cache->clear($id);


=head1 DESCRIPTION


JAAS::Object::Cache::Persistant is an implementation of the repository cache
interface that adds persistance.  It does so via an object called a
"lifetimer" (JAAS::Object::Lifetime) that saves and restores the context
when the need arrises.

=head1 METHODS

=head2 new

    my $cache=JAAS::Object::Cache::Persistant->new($lifetime);

Object constructor.  C<$lifetime> is an object that is used for persistance.

=head2 setup

    $cache->setup($id);

If the context C<$id> doesn't exist in memory, the lifetimer is called as
follows:

    $lifetimer->restore($object, 'context', $id);

Where C<$object> is, as described in JAAS::Object::Liftime.

=head2 clear

    $cache->clear($id);

If no one else is interested in context C<$id>, the lifetimer is used to
save the context as follows:

    $lifetimer->save($object, 'context', $id);

Where C<$object> is described in JAAS::Object::Liftime.

=head1 BUGS

This interface is still under construction.  I will probably redo it very
soon.

=head1 AUTHOR

Philip Gwyn <gwynp@artware.qc.ca>

=cut

$Log: Persistant.pm,v $
Revision 1.1  2001/06/22 20:44:07  fil
Fixed some CVS problems :)

