# $Id: Simple.pm,v 1.1 2001/06/22 20:44:07 fil Exp $
package JAAS::Object::Cache::Simple;

use strict;

sub DEBUG {0};

use Carp;

###########################################################
sub new
{   
    my($package)=@_;
    return bless {}, $package;
}

###########################################################
sub setup
{
    my($self, $id, $who)=@_;
    my $ret=exists $self->{$id};
    $self->{$id}{_interested}{$who}=1;
    DEBUG and warn "$who is interested in $id";
    return $ret;
}

###########################################################
sub clear
{
    my($self, $id, $who)=@_;
    return unless exists $self->{$id}{_interested}{$who};
    delete $self->{$id}{_interested}{$who};
    DEBUG and warn "$who isn't interested in $id";

    return if keys %{$self->{$id}{_interested}};    

    DEBUG and warn "$id is GONE";
    # no more interest
    delete $self->{$id};
    return 1;
}

###########################################################
sub shutdown
{
    my($self, $ids)=@_;
    foreach my $id (@$ids) {
        foreach my $who (keys %{$self->{_interested}}) {
            $self->clear($id, $who);
        }
    }
}

###########################################################
sub fetch
{
    my($self, $id, $name)=@_;
    confess "No ID" unless $id;
    return $self->{$id}{$name};
}

###########################################################
sub store
{
    my($self, $id, $name, $data)=@_;
    confess "No ID" unless $id;
    return $self->{$id}{$name}=$data;
}

###########################################################
sub delete
{
    my($self, $id, $name)=@_;
    confess "No ID" unless $id;
    return delete $self->{$id}{$name};
}



1;
__END__

=head1 NAME

JAAS::Object::Cache::Simple - JAAS object repository cache

=head1 SYNOPSIS


    # build a new one
    my $cache=JAAS::Object::Cache::Simple->new();

    # setup a context
    $cache->setup($id);
    
    # fetch one object
    my $obj=$cache->fetch($id, 'IO::File');
    unless($obj) {          # woah, doesn't exist
        $obj='IO::File'->new();

        # store the item in the cache
        $cache->store($id, 'IO::File', $obj);
    }

    # remove an object from the cache
    $cache->delete($id, $obj_name);

    # clear our interst in the context.  Cache takes are of GC.
    $cache->clear($id);

    # batch clearing of interest
    $cache->shutdown($ids);



=head1 DESCRIPTION

An object cache keeps track of all objects in a Context, and all external
entities that are interested in the Context.  This simple implementation
keeps them in memory until everyone has C<clear>ed all their interest, then
kicks all the objects out of memory.

See L<JAAS::Object::Cache::Persistant> if you are interested in adding
persistance.


=head1 METHODS

=head2 new

    my $cache=JAAS::Object::Cache::Simple->new();

Object constructor.

=head2 setup

    $cache->setup($id, $who);

C<$who> is expressing interest in the context C<$id>.  If the context
doesn't exist, it is created.

=head2 clear

    $cache->clear($id, $who);

C<$who> is expressing disinterest in the context C<$id>.  If this is the
last entity to clear it's interest, the context is deleted.  


=head2 shutdown

    $cache->shutdown($ids);

Batch clearing of interests.  I thought maybe we could possibly share caches
somehow, and each user would do it's own C<shutdown>.  However, this clears
ALL interest in the Contexts listed.  Caveat emptor.

=head2 fetch

    my $obj=$cache->fetch($id, $name);

Fetch the object named C<$name> from the context C<$id>.  Returns C<undef()>
if the object doesn't exist in the context.

=head2 store

    $cache->store($id, $name, $object);

Stores an C<$object> named C<$name> in the context C<$id>, stomping any
object previously present in the cache.

=head2 delete

    $cache->delete($id, $name);

Removes the object C<$name> from the context C<$id>.

=head1 BUGS

=head1 AUTHOR

Philip Gwyn <gwynp@artware.qc.ca>

=cut

$Log: Simple.pm,v $
Revision 1.1  2001/06/22 20:44:07  fil
Fixed some CVS problems :)

Revision 1.1  2001/06/21 04:11:19  fil
Totaly rearranged things:
    Context.pl is now Context.pm and server.perl
    Blocking and locking now are revoked gracefully when remote closes
    Blocking loop is now unrolled.
    Removed all the delay()s.  When something is deffered, we check again
    when the condition is encountered.

