# $Id: Factory.pm,v 1.1 2001/06/22 20:42:20 fil Exp $
package JAAS::Object::Factory;

use strict;

sub DEBUG {0}

###########################################################
sub new
{
    my($package, $config)=@_;

    if(ref $config eq 'ARRAY') {
        my %conf;
        @conf{@$config}=map {{package=>$_}} @$config;
        $config=\%conf;
        DEBUG and warn "Promoting config to ", ref $config;
    }
    bless {config=>$config, deleted=>{}}, $package;
}

###########################################################
sub make
{
    my($self, $name)=@_;
    DEBUG and warn "Making $name";

    my $c=$self->{config}{$name};
    DEBUG and warn "Config is $c";
    return unless $c;

    my $package=$c->{class} || $c->{package};
    DEBUG and warn "Package is $package";

    $self->load_package($package) or return;
    DEBUG and warn "Package is $package";

    my $constructor=$self->find_ctor($package, $c->{ctor});
    return unless $constructor;
    DEBUG and warn "Constructor for $package is $constructor";

    my $object=eval {
            $self->build_object($package, $constructor, $c->{args});
        };
    return unless $object;

    DEBUG and warn "Object is $object";
    return $object;
}



###########################################################
sub find_ctor
{
    my($self, $package, $ctor)=@_;
    my @funcs;
    if($ctor) {
        @funcs=($ctor);
    } else {
        @funcs=qw(spawn new);
    }
    my $func;
    foreach my $c (@funcs) {
        $func=$package->can($c);
        last if $func;
    }
    return $func if $func;
    DEBUG and warn "Can't find contructor: @funcs";        
    return
}

###########################################################
sub build_object
{
    my($self, $package, $ctor, $args)=@_;
    
    my @under=($package);
    my $r=ref $args;
    if($r and $r eq 'ARRAY') {
        push @under, @$args;
    } else {
        push @under, $args;
    }
    $ctor->(@under);
}

###########################################################
sub _get_table
{
    my($package)=@_;
    my $table=$::{'main::'};
    foreach my $b (split '::', $package) {
        return unless exists $table->{$b.'::'};
        $table=$table->{$b.'::'};
    }
    return $table;
}

###########################################################
sub load_package
{
    my($self, $package)=@_;

    return 1 if defined _get_table($package); # see if it exists in symtable
    # nope... have to load it
    my $path="$package.pm";
    $path=~s(::)(/)g;
    eval { require $path } and return 1;
    return;
}


1;

__DATA__

=head1 NAME

JAAS::Object::Factory - Builder of objects

=head1 SYNOPSIS

    use JAAS::Object::Factory;
    my $factory=JAAS::Object::Factory->new($config);

    # create a new object
    my $obj=$factory->make($object_name);

=head1 DESCRIPTION

JAAS::Object::Factory is an object that, given an configuration, will
construct objects when asked politely.  Construction is controled by
C<$config>.  Objects are asked for by name.  This name is associated with a
package, and optionnal constructor arguments by C<$config>.  Packages can
loaded on the fly.  After construction, the factory doesn't care about the
objects and it's up to the caller to muddle with them.  Run-time control is
achieved via C<$config>.

There is an additional mechanism for overloading some behaviour.

=head1 METHODS

=head2 new

    my $factory=JAAS::Object::Factory->new($config)

Object constructor.  Requires one parameter, C<$config>, which controls how
a given object is created.  C<$config> can be either a hashref or an arrayref.
The hashref is keyed on object names, values are hashrefs that contain
information on how to construct the objects:

=over 4

=item C<class>, C<package>

Scalar value that indicates what package the constructor will be call in. 
In other words, what object will be created.  If both are present C<class>
has priority over C<package>.

=item C<args>

Arguments to the constructor.  If it is an arrayref, it is dereferenced
before passing to constructor, otherwise it is passed as is.

=item C<ctor>

Normally, constructors are call C<spawn> (for C<POE::Session>s) or C<new>
(for standard objects), but MAYBE you like doing things different?  If so,
set the C<ctor> to whatever your constructor is called.  Yes, this means you
can call arbitrary package methods.  Not that you should, eh.

=back

    JAAS::Object::Factory->new({
                # passing an array to Net::Telnet->new();
                Telnet=>{   class=>'Net::Telnet'
                            args=>[Timeout=>10, Prompt=>'/bash\$ $/']
                        },
                # passing a scalar to IO::File->new();
                File=>{     class=>'IO::File',
                            args=>"filename"
                      },
                # Calling an abritrary package method
                Dumper=>{   class=>'Data::Dumper',
                            ctor=>'Dump'
                            args=>[[$foo, $bar], [qw(foo *ary)]]
                        }
                }
            );


If C<$config> is an arrayref, it is assumed to be a list of package names
and is promoted to a hashref accordingly.

    JAAS::Object::Factory->new([qw(IO::File Net::TCP)]);

Is equivalent to 

    JAAS::Object::Factory->new({IO::File=>{package=>'IO::File'},
                                Net::TCP=>{package=>'Net::TCP'}});

C<JAAS::Object::Factory->new> should never blow up.


=head2 make

    my $something=$factory->make('File');

This method constructs an instance of the named object that is the required
and sole parameter.  Instructions for creating the object are described in
the C<$config> parameter passed to C<new>.

C<make> will load the package if it's not already.  If loading fails, it
returns C<undef()> and possibly something in $@.  C<make> then calls the
constructor, passing it the package name as first param, and any other
parameters defined in C<$config>.  If the constructor throws an exception,
C<make> returns C<undef()> and the exception is in C<$@>.

C<make> will also return C<undef()> if the named object doesn't exists in
C<$config>.



=head1 OVERLOADING

It is also possible to change some of JAAS::Object::Factory's behaviour by
deriving a class from it, and overloading the following methods:

=head2 load_package

    sub load_package {
        my($self, $package)=@_;
        ....
    }

This method make sure that $package has been properly loaded.  Current
behaviour is to check the symbol table.  If it's absent, it does a
C<require> (wraped in C<eval()>) on the package.

Returns true if C<$package> is now usable, and false if it isn't.

You might want to overload C<load_package> if you want to pull classes out
of new and exciting places, rather then perl's standard C<require>
mechanism.


=head2 build_object

    sub build_object {
        my($self, $package, $ctor, $args)=@_;
        return $ctor->($package, $args);
    }

This is the heart of the action.  It is expected to call the coderef
C<$ctor> in C<$package>, with the parameters in C<$args>.  C<$package> has
already been loaded by L<load_package>.   

You might want to overload this method to change the parameters handed to
the constructor.


=head2 find_ctor

    sub find_ctor {
        my($self, $package, $ctor)=@_;
        $ctor||='new'
        return $package->can($ctor);
    }

This method returns a coderef of the method C<$ctor> in package C<$package>. 
Returns C<undef()> on failure.  C<$ctor> might be empty, in which case you
should look for a default constructor.

You might want to overload this method if you don't like the fact that the
default method looks for 'spawn' before 'new'.  Or again, maybe you want to
pull your code out of new and exciting places.


=head1 BUGS

C<$config> is (nominaly) a hashref.  All we currently do is call C<FETCH()> on
it.  It could almost as easily have been an object.

=head1 AUTHOR

Philip Gwyn <gwynp@artware.qc.ca>

=cut



$Log: Factory.pm,v $
Revision 1.1  2001/06/22 20:42:20  fil
Added Factory, Cache/*
Reworked internals... Still have problems, though.

