package JAAS::Object::Lifetime;

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);
use Carp;
use Fcntl qw(:flock);

@ISA = qw();
@EXPORT = qw(
	
);
$VERSION = '0.02';

BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () {0}; }
}


#########################################################
sub new
{
    my $package=shift @_;
    my $params;
    if(@_==1 or @_==2) {                # fart around so that we can get
        $params=$_[0];                  # params from config file and
        $params->{factory}=$_[1];       # at run time
    } else {
        $params={@_};
    }

    my $self=bless {}, $package;
    delete $params->{name} unless 'HASH' eq ref $params->{name};
    $self->{id}=    delete $params->{key};
    $self->{when}=  delete($params->{when}) || 'save';
    $self->{factory}= delete $params->{factory};
    foreach my $when (keys %$params) {
        my $what={%{$params->{$when}}};
        $self->{stores}{$when}{expire}=delete $what->{expire};
#        $self->{stores}{$when}{class}=delete $what->{class}
        $self->{stores}{$when}{tie}=delete $what->{tie};
        $self->{stores}{$when}{args}=$what;
    }
    return $self;
}

#########################################################
sub key
{
    my($self, $id)=@_;
    return $self->{id} if @_==1;
    $self->{id}=$id;
}

#########################################################
sub when
{
    my($self, $when)=@_;
    return $self->{when} if @_==1;
    $self->{when}=$when;
}


#########################################################
sub store
{
    my($self, $when)=@_;
    my $config=$self->{stores}{$when};
    my %store;
    tie %store, $config->{tie}, $config->{args};
    return \%store;
}

#########################################################
sub save
{
    my($self, $objects, $when, $id)=@_;
    $when||=$self->{when};
    croak ref($self), " I don't know how to save '$when'." 
                unless $self->{stores}{$when};
    
    $id||=$self->{id};
    croak ref($self), "I need an identifying key" unless $id;

    DEBUG and warn "Saving $id for $when";

    my %data;
    while(my($name, $os)=each %$objects) {
        $os=[$os] unless ref $os;
        foreach my $o (@$os) {
            my $d;
            if(UNIVERSAL::can($o, "can")) {   # it's an object
                DEBUG and warn "lifetime_save() for a $name ($o)";
                $d=$o->lifetime_save($when);
            } 
            else {
                DEBUG and warn "Blindly saving a $name";
                $d=$o;                              # anon-reference
            }
            push @{$data{$name}}, $d if $d;
        }
    }
    my $store=$self->store($when);
    $store->{$id}=\%data;                           # save to the store
    return;
}

#########################################################
sub restore
{
    my($self, $objects, $when, $id)=@_;
    $when||='save';
    croak ref($self), "I don't know how to restore '$when'." 
                unless $self->{stores}{$when};
    
    $id||=$self->{id};
    croak ref($self), "I need an identifying key" unless $id;

    unless(keys %$objects or $self->{factory}) {  # didn't ask for any objects
        return;
    }
    DEBUG and warn "Restoring $id for $when";

    my $data;
    {
        my $store=$self->store($when);
        return 0 unless exists $store->{$id};
        $data = $store->{$id};                        # load from the store
    }
    
    unless(keys %$objects) {                        # no objects? create them
        DEBUG and warn "Creating blank objects with $self->{factory}";
        foreach my $name (keys %$data) {      
            my @objects;
            foreach my $o (@{$data->{$name}}) {
                DEBUG and warn "Creating a $name";
                my $new=$self->{factory}->make($name);
                if($new) {
                    push @objects, $new;
                }
                else {
                    push @objects, $o;
                    warn "Creating blank object $name: $@\n"
                        unless defined $new;
                    DEBUG and 
                        warn "$name ($o) propbably isn't an object";
                }
            }
            $objects->{$name}=\@objects;
        }
    }
    return unless keys %$objects;                   # still nothing, forget it

#    use Data::Denter;
#    warn Denter $objects, $data;

    # objects is name => [ destination objects ]
    # data is name => [saved objects ]
    while(my($name, $os)=each %$objects) {
        $os=[$os] unless ref $os;
        foreach my $o (@$os) {
            unless(@{$data->{$name}}) {             # mis-match
                DEBUG and warn "Mismatch: there is no $name";
                next;
            }
    
            my $d=shift @{$data->{$name}};
            if(UNIVERSAL::can($o, "can")) {   # it's an object
                DEBUG and warn "lifetime_restore() for a $name";
                $o->lifetime_restore($when, $d);
            } 
            else {
                DEBUG and warn "Blind copy of a $name";
                $o=$d;                              # anon-reference
            }
        }        
    }
#    warn Denter $objects;
    return 1;
}

#########################################################
sub expire
{
    my($self)=@_;
    foreach my $when (keys %{$self->{stores}}) {
        next unless $self->{stores}{$when}{expire};
        my $store=$self->store($when);
        tied(%$store)->expire($self->{stores}{$when}{expire});
    }
}

#########################################################
sub delete
{
    my($self, $objects, $when, $id)=@_;
    $when||='save';
    croak ref($self), "I don't know how to restore '$when'." 
                unless $self->{stores}{$when};
    
    $id||=$self->{id};
    croak ref($self), "I need an identifying key" unless $id;
    my $store=$self->store($when);
    return unless exists $store->{$id};
    delete $store->{$id};
    return 1;
}

#########################################################
sub lock
{
    my($self, $id, $when, $write)=@_;
    my $fh = Symbol::gensym();

    my $config=$self->{stores}{$when};
    unless($config) {
        warn "Can't find a store for $when: ", join ', ', keys %{$self->{stores}};
        return;
    }
    my $LockDirectory = $config->{args}{LockDirectory};
    unless($LockDirectory) {
        use Data::Denter;
        warn "Can't find directory to lock: ", Denter $config;
        return;
    }
    open($fh, "+>$LockDirectory/lifetime-$id.lock")
            or return 0;

    flock($fh, (LOCK_NB|($write ? LOCK_EX : LOCK_SH)))
            or return 1;
    warn "Locked lifetime-$id";
    return sub {
        # flush $fh;
        warn "Unlocking lifetime-$id";
        flock($fh, LOCK_UN);
        close $fh;
    };
}

1;
__END__
# Below is the stub of documentation for your module. You better edit it!

=head1 NAME

JAAS::Object::Lifetime - Perl extension for blah blah blah

=head1 SYNOPSIS

    use JAAS::Object::Lifetime;

    my $life=new JAAS::Object::Lifetime (
            profile=>{tie=>"Data::Squirrel::MySQL",
                       DataSource=>"dbi:mysql:profiles", 
                       UserName=>"something", Password=>"else",
                       LockDataSource=>"dbi:mysql:profiles", 
                       LockUserName=>"something", LockPassword=>"else",
                       expire=>60*60*24*30*6,   # 6 months
            },
            session=>{tie=>"Data::Squirrel::DB_File",
                        FileName=>"$data/something.db",
                        LockDirectory=>$data,
                        expire=>60*60*3,        # 3 hours
                    },
        );
    # YOW! This is an IDEAL place to use JAAS::Config

    $life->key($session_id);                # set the session key
    $life->save($objects, 'session');       # save all the objects
    $life->restore($objects, 'profile', $user_id);  # load a profile
    $life->expire();                        # expire stuff
    $life->delete('paranoid');              # get rid of paranoid log

=head1 DESCRIPTION

JAAS::Object::Lifetime is an interface to various data stores.  Lifetime is
another way of looking at persistance.  At various points in an application,
you ask each object if they are interested in being saved.  By asking each
object, you permit them to make "lazy" descisions.  Also, this allows the
objects to decide what gets saved and what doesn't.

Here's an example, using an e-commerce web application. You will want to
save objects at the following moments:

 when     |save              |restore          |expire  |delete
 ---------+------------------+-----------------+--------+------
 session  |After each HTTP   |When a new HTTP  |3 hours |
          |request           |request comes in |        |   
 ---------+------------------+-----------------+--------+------
 profil   |When user creates |When user loads  |1 year? |User asks to
          |or modifies profil|profil           |        |   
 ---------+------------------+-----------------+--------+------
 purchase |When a pruchase   |Debuging?  Stats?|        |   
          |is completed      |                 |        |   
 ---------+------------------+-----------------+--------+------
 paranoid |Just before doing |post-mortem of   |NEVER!  |When CC 
          |CC validation     |errors           |        |validation
          |Log Early!  Log Often!              |        |succeeds
 

Hope it gives the following more meaning to you.


=head1 METHODS

=head2 new

    my $life=new JAAS::Object::Lifetime (
        'purchase' => {tie=>'My::Crypto::Store', 
                       Directory=>'/some/obscure/directory',
                       LockDirectory=>'/some/obscure/directory',
                      }
    );

    # Can't fail, really.

Creates the new object.  Parameters are a hash of hashrefs.  Keys are the
various "when"s you're going to be using.  'save' is a special "when". 
See below.

Values are a hashref that defines the data store.  Two items are important,
the rest are passed as-is to the object that deals with the actual store:

=over 4

=item tie

Class that we tie a hash to.

=item expire

Time, in seconds, after which an object is removed from the store.  If not
defined (or 0), then expiration doesn't happen.  Note that you have to call
$life->expire() on your object at regular intervals for this to work.

=back

This means that the example given for new would tie an object like follows:

    tie %store, 'My::Crypto::Store', {
                       Directory=>'/some/obscure/directory',
                       LockDirectory=>'/some/obscure/directory',
                      };

The tie is done many times: for a save, for a restore, for an expire and for
deletes.

I should allow objects to be used also.

=head2 key

Sets the key under which all save/restore/delete operations will *mumble* in
the data store.

    $life->key($SESSIONID);         # sets the key
    my $id=$life->key;              # fetch the key

=head2 save

    $life->save($objects[, $when[, $key]]);

Commits objects in C<$objects> to the C<$when> store, with key=C<$key>.
Parameters:

=over 4

=item $objects

Hash of arrayrefs.  Keys (names), a way of distinguishing various types of
objects at restore time.  Values are arrayrefs of objects or references that
correspond to that "name".  Read description of C<restore> to see why we do
this.  

All the objects in $objects will be queried as to what data they want to
save at that moment.  This is done by calling lifetime_save() on the object,
with one parameter: $when.  This method is expected to return either undef()
if it does not want to save anything or any data that should be saved.  You
could return the entire object, or only the bits you want to have saved.

If C<$objects> is an empty hash ref, all objects are loaded into it.

=item $when

One of the "when"s you defined in new().  Defaults to... uh... 'save'.

=item $key

Sets the key under which the data will be saved.

=back


    $objects={Cart=>[$cart], User=>[$user], Mutter=>[$mutter1, $mibble]};
    $life->store($objects, 'purchase', $purchase_id);

This causes a hash like the following to be saved:

    $store{$purchase_id}={
        Cart=>[$cart->lifetime_save('purchase')],
        User=>[$user->lifetime_save('purchase')],
        Mutter=>[$mutter1, $mibble],  # couldn't tell they weren't
                                      #  objects, eh :)
    };


=head2 restore

    $life->save($objects[, $when[, $key]]);

Restores objects in C<$objects> from the C<$when> store, with key=C<$key>.
Parameters are the same as C<save()>.

C<restore> is where $objects start to make sense.

Each object in $objects is passed the data that was saved, via
C<lifetime_restore()> with 2 params: the "when" and whatever it returned
from C<lifetime_restore()> so that it can merge this data with itself.  This
is why we need "names" for the objects, to associate the restored data with
an "object" with the same "name".  If class names where used, we couldn't
have several objects of the same class.  Also, we couldn't save plain
references.  An arrayref is used so that many objects can have the same
name... if this is what you wanted.

If the layout of C<$objects> changes between save() and restore(), strange
things might happen.  

Continuing the example we've been developing, a restore would do the
following:

    my $data=$store{$purchase_id};
    $cart->lifetime_restore('purchase', $data->{Cart}[0]);
    $user->lifetime_restore('purchase', $data->{User}[0]);
    $objects->{Mutter}=@{$data->{Mutter}}[0..1];

=head2 expire

    $life->expire();

Because most of the parameters are set at creation time, this does its magic
alone.



=head2 delete

    $life->delete([$when[, $key]]);

Removes ALL objects from data store $when (default is 'save'), key $key
(default to what you set $life->key to).


=head1 INTERFACE

=head2 lifetime_save

Used to query each object that is to be saved.  Parameters are the "when" of
the save.

=head2 lifetime_restore

Used to give each object the data that was restored from the data store. 
Parameters are the "when" of the restore, and the data that lifetime_save
returned previously

If you don't have these two methods in your object, THE CODE WILL CRASH AND
BURN.  If you use a data store that reblesses objects, here is are the
shortest useful methods:

    sub lifetime_save { $_[0] }                 # save everything
    sub lifetime_restore { $_[0]=$_[2]}        # accept old self for new
    # However, this won't work :
    # sub lifetime_restore { my($self, $when, $data)=@_;
    #                        $self=$data};

=head1 AUTHOR

Philip Gwyn <Leolo at pied dot nu>

=head1 SEE ALSO

Data::Squirrel, perl(1).

=cut
