package Win32::ODBC::Ext;

# ----------------------------------------------
# Extentions to the Win32::ODBC.pm package
# by dread fil (fil@interlinx.qc.ca)
# the rainy july of 1996
#
# The semantics of these extentions where inspired by
# MS-Access DAO
#
# This package inherits from Win32::ODBC
# but unlike it, we die on errors.  Hmm... maybe
# I'll change this behaviour to stuffing error
# messages into $self->{'error'}.
# ----------------------------------------------

require Exporter;
require DynaLoader;
require Win32::ODBC;


# ----------------------------------------------
# $Win32::ODBC::Ext::Version = 960718;

@ISA= qw( Win32::ODBC Exporter);

# ----------------------------------------------
# Value to be put between array elements and hash key:value pairs when
# used in a SQL statement
$newline='%0D%0A';

# ----------------------------------------------
# create ourelves and blessed be
# $odbc = new Win32::ODBC::Ext('DSN');
sub new
{
    local($type)=@_;
    local $self;

    $self=Win32::ODBC::new(@_);
    if(not ($self =~/Win32::ODBC/))     # the Win32::ODBC constructor returns
    {                                   # some wierd error codes.
        return "Error: ".$self;
    }

    bless $self, $type;
    return $self;
}

# ----------------------------------------------
# Decode values that where probably previously encoded by perpVal
sub DeprepVal
{
    local($self, $value)=@_;
    $value=~s/%([0-9A-Fa-f]{2})/sprintf("%c", 0+hex($1))/eg;
    return $value;
}

# ----------------------------------------------
# Prepare a value for use in an SQL statement
sub PrepVal
{
    local($self, *valeur)=@_;
    local($temp);

    if(defined $valeur)
    {
        $temp=$valeur;                          # simple scalar
    } elsif(defined @valeur)
    {
        $temp=join $newline, @valeur;           # join arrays with the newline
    } elsif(defined %valeur)
    {
        $temp="";                               # make hashes into strings
        while(($key, $val)=each %valeur)
        {
            $temp .="$key: $val$newline";
        }
    } else
    {
        $temp='';       # hmmm, it could very well be numeric, but fuck-it
    }

    $temp=~s/([\x00-\x1f'])/sprintf("%%%2.2X", 0+ord($1))/eg; # ODBC can't have embeded
    return $temp;                            # quotes
}

# ----------------------------------------------
# Helper routine to get the name of the current table
sub getTable
{
    local($self, $table)=@_;

    $table=$self->{'table'} if(not $table);

    $self->{'table'}=$table;

    return undef if($table eq '');
    return $table;
}

# ----------------------------------------------
# Set the value of a field
# $odbc->Set('field', $value);
sub Set
{
    local($self, $field, $val)=@_;
    local($package, $filename, $line)=caller;
    local($temp);

    if(!$self->{'edit'})
    {
        die "You must call Append or Edit before Set at $filename line $line\n";
    }

    if(defined $self->{'field'}->{$field})
    {
        $temp=$self->PrepVal(\$val);
        $self->{'data'}->{$field}=$temp;
    } else
    {
        warn "Unknown field $field at $filename line $line\n";
    }
}

# ----------------------------------------------
# Get the value of a field
sub Get
{
    local($self, $field)=@_;
    local($package, $filename, $line, $temp)=caller;

    if(!$self->{'data'})        # Sql("Select") will popluate this hash
    {   # hmmm, should I die or warn
#        die "No fields are curently selected $filename line $line\n";
        warn "No fields are curently selected at $filename line $line\n";
    } elsif(defined $self->{'data'}->{$field})
    {
        $temp=$self->{'data'}->{$field};
        $temp=$self->DeprepVal($temp); # unless ($self->{'dont_deprep'});
        return $temp;
    } else
    {
        warn "Unknown field $field at $filename line $line\n";
    }
    return undef
}

# ----------------------------------------------
# Create a blank record to receive values from Set();
# $odbc->Append(["table"]);
sub Append
{
    local($self, $table)=@_;
    local($package, $filename, $line)=caller;

    $table=getTable($self, $table)
                or die "You must spcify a table at $filename line $line\n";
    $self->Sql("SELECT * FROM $table")
                and die $self->Error()." at $filename line $line\n";
#    $self->FetchRow()     or die $self->Error();

    undef $self->{'data'};
    $self->{'append'}=1;
    $self->{'edit'}=1;
}

# ----------------------------------------------
# Prepare the current record for modification
# $odbc->Edit(["table"]);
sub Edit
{
    local($self, $table)=@_;
    local($package, $filename, $line)=caller;

    if(!$self->{'field'})
    {
        die "You must call Sql or Find to select fields before Edit at $filename line $line.\n";
    }
    $table=getTable($self, $table)
                or die "You must specify a table at $filename line $line\n";

#    $self->FetchRow()     or die $self->Error;
    $self->{'current'}=$self->DataHash();
    $self->{'edit'}=1;
}

# ----------------------------------------------
# Set the current table
sub Table
{
    local($self, $table)=@_;
    $table=getTable($self, $table);
    return $table;
}

# ----------------------------------------------
# Quit the current Edit or Append
sub Abort
{
    local($self)=@_;
    undef $self->{'append'};
    undef $self->{'edit'};
    undef $self->{'current'};
}

# ----------------------------------------------
# Update the record after modification by Edit or Append
sub Update
{
    local($self)=@_;
    local($sqlStmt, $fields, $values, $f, $v);
    local($package, $filename, $line)=caller;

    die "Must call Edit or Append before Update at $filename line $line.\n"
        if(!$self->{'edit'});

    if($self->{'append'})
    {
        while(($f, $v)=each %{$self->{'data'}})
        {
            $fields.="[$f],";
            $values.="'$v',";
        }
        $fields=~s/,$//;
        $values=~s/,$//;
        # simple SQL statement, eh
        $sqlStmt="INSERT INTO $self->{'table'} ($fields) VALUES ($values)";
    } else
    {
        while(($f, $v)=each %{$self->{'data'}})
        {
            $values.="[$f]='$v',";
        }
        # attempt to uniquely identify the current record, if we don't
        # more than one record will be modified.  Of course, we could always
        # perform a check (at a performance hit), but I don't really have
        # the time to code one.
        # $self->{'current'} was populated by Edit()
        while(($f, $v)=each %{$self->{'current'}})
        {
            $fields.="[$f]='$v' AND ";
        }
        undef $self->{'current'};
        $values=~s/,$//;
        $fields=~s/ AND $//;
        # This is hoarier.
        $sqlStmt='UPDATE '.$self->{'table'}." SET $values WHERE ($fields)";
#        print $sqlStmt;
    }

    $self->Sql($sqlStmt)    and die $self->Error()." at $filename line $line.\n";
}

# ----------------------------------------------
# ----------------------------------------------
# Do we currently have a matched record.
sub Found
{
    local($self)=@_;
    return $self->{'matched'} && $self->{'finding'};
}

# ----------------------------------------------
# Find the first record for a given search clause
# $self->FindFirst(['$clause'], ['$table'])
# if clause is blank, the entire table will be used
sub FindFirst
{
    local($self, $clause, $table)=@_;
    local($sqlStmt);
    local($package, $filename, $line)=caller;

    if($clause and $clause =~/^SELECT/i)
    {
        $sqlStmt=$clause;
    } else
    {
        $table=getTable($self, $table)
                or die "You must specify a table at $filename line $line\n";

        $sqlStmt="SELECT * FROM $table";
        $sqlStmt.=" WHERE $clause"  if($clause);
    }

    $self->Sql($sqlStmt)    and die $self->Error." at $filename line $line\n";

    $self->{'finding'}=1;
    undef $self->{'f_function'};
    simpleNext($self)       or die $self->Error." at $filename line $line\n";
}

# ----------------------------------------------
# Initiate a search, but use a Perl subroutine to validate each record.
# $subRef=sub{ local(%data)=@_; blah blah blah };
# $odbc->FindFirstX($subRef, ['table'])
# The subroutine is passed a hash table with the current record in it.
# Returning TRUE (1) means it's part of this search, FLASE (0 or undef)
# means we go on to the next record
sub FindFirstX
{
    local($self, $clause_func, $table)=@_;
    local($sqlStmt);
    local($package, $filename, $line)=caller;

    $table=getTable($self, $table)
            or die "You must specify a table at $filename line $line\n";

#    $self->{'cursor'}='hello_world';
#    $sqlStmt="DECLARE hello_world CURSOR for SELECT * FROM $table";

    $sqlStmt="SELECT * FROM $table";
    $self->Sql($sqlStmt)    and die $self->Error;

    $self->{'finding'}=1;
    $self->{'f_function'}=$clause_func;
    simpleNext($self)       or die $self->Error;
    runFFunc($self)         or die $self->Error;
}

# ----------------------------------------------
# Helper routine for running the validation function
sub runFFunc
{
    local($self)=@_;
    while($self->{'finding'} and !&{$self->{'f_function'}}($self->DataHash()))
    {
        simpleNext($self)         or return undef;
    }
    return 1;
}

# ----------------------------------------------
# Goes to the next record in the current search
sub FindNext
{
    local($self)=@_;
    local($sqlStmt);
    local($package, $filename, $line)=caller;

    if($self->{'finding'})
    {
        simpleNext($self)     or die $self->Error;

        if($self->{'f_function'})
        {
            runFFunc($self)   or die $self->Error;
        }
    } else
    {
        die "Must call FindFirst or FindFirstX before FindNext at $filename line $line\n";
    }
}

# ----------------------------------------------
# Helper function to go to the next record and deal with
# possible errors
sub simpleNext
{
    local($self)=@_;

    undef $self->{'matched'};
    if($self->FetchRow())               # do we have any data
    {
        $self->{'matched'}=1;           # yes, cool
    } else
    {
        undef $self->{'finding'};       # no, damnit
        undef $self->{'f_function'};
        if($self->Error()=~/no data/i)  # Ugly, but it's the way to find out if
        {                               # we've ended the search
        } else
        {
            return undef;               # SQL error or something
        }
    }
    return 1;
}


1;

__END__

=head1 NAME

ODBC::Ext.pm - Extentions to the ODBC module

=head1 ABSTRACT

This module permits the modification and addition of data in ODBC tables.
It also includes more conginial searching functions.

=head2 SECTIONS

L<ABSTRACT> --
L<SECTIONS> --
L<DESCRIPTION> --
L<BIGER EXAMPLE> --
L<INSTALLATION> --
L<AVAILABILITY> --
L<BUGS!@#!> --
L<AUTHORS>

=head1 DESCRIPTION

=head2 (more like a history, but what the hey -:)

After looking over the excelent Win32::ODBC module, I realised that it's
oriented towards data retrieval.  This, if you ask me, is only half (or
even a third) of what one usualy does with databases.  The basic ODBC
functions (in ODBC.pll) didn't include any functions to modify the data
either.  I can well understand their lack, I can't make head or tail of
the SqlBind functions either.  So I decided to write some wrapers around
the SQL INSERT and UPDATE statements.  I also thought it would be useful
to have simpler searching functions and maybe even be able to use Perl
regexs for searching.

Of course, this is only an extention to the old Win32::ODBC package.
All the old functions that we all know and love are still available.

=head1 VARIABLES

=item $newline B<Static>

This variable is appended to each element of an array and each hash key: value
pair when it is used by the L<PrepVal> method.  It is currently defined as
'%0D%0A'.  You could change this to '<p>' if the data is only going to be
used on the web, or anything else that suits your fancy.

=head1 METHODS

=head2 List

L<Abort> --
L<Append> --
L<DeprepVal> --
L<Edit> --
L<FindFirst> --
L<FindFirstX> --
L<FindNext> --
L<Found> --
L<Get> --
L<PrepVal> --
L<Set> --
L<Table> --
L<Update> --
L<new>

=item new ODBC::Ext (['DSNname'])

Creates a ODBC::Ext object and opens a connection to the DSN name given.  If
an error occurs (say the DSN doesn't exist), new returns a string that
starts with 'Error: ' followed by an error message.

    $odbc = new ODBC::Ext ('Enviro');
    if($odbc =~ /^Error:/)
    {
        die "Connecting to 'Enviro', $odbc";
    }

=item Get ('field')

Returns the value of the given field in the current record.  Sql,
L<FindFirst>, L<FindFirstX> or L<Edit> should be called first to
select the record.

Note: because I use an SQL statement to modify values, some of them must
be encoded.  The Get method therefore decodes everything with
L<DeprepVal>.  See L<Edit> for an explanation.

    $odbc->Sql("SELECT ID_nom, ID_tele, ID_fax FROM Chercheur"
               ." WHERE (ID_nom='Jean Drolet'");
    print $odbc->Get('ID_nom')."\n";
    print $odbc->Get('ID_tele')."\n";
    print $odbc->Get('ID_fax')."\n";

=item Set ('field', $value)

Set the value of the current field to a new value.  The changes aren't
writen to disk until the L<Update> method is called.

B<Note>: SQL does not permit single quotes (') in a
statement (or if it does, I don't know how).  I<THIS IS VERY ANNOYING!>  Nor
does it permit exotic characters such as newlines and what not.  Ticks
(') are encoded as %27 (ie % followed by the ascii value in hex).
You will need to translate back anything at the other end (L<Get> does
this automaticaly).

Another thing to watch out for is date values.  Make sure they are in
cannonical form, because 1/3/1995 can be ambiguous.

    $personne{'addr'}=['that number','this street','some city'];
    $data->Append();
    $data->Set('ID_nom', $personne{'nom'});
    $data->Set('ID_addr', $data->PrepVal($personne{'addr'}));
    $data->Set('ID_tele', $personne{'tele'});
    $data->Set('ID_fax', $personne{'fax'});
    $data->Set('ID_e_mail', $personne{'e_mail'});
    $data->Update();


=item Edit (['table'])

Prepares the current record for modification.  The L<Get> and L<Set>
methods may then be used to modify the record.  Changes are only writen
to the table when the L<Update> method is called.  If the table isn't
specified, the current table is used.  Be aware of that L<Edit> has some
wonky behaviour.  See L<BUGS!@#!>.

    $odbc->Sql("SELECT * FROM Invoice WHERE ID='00101'")
    $odbc->Edit('Invoice');
    $odbc->Set('Status', 'done');
    $odbc->Update();

=item Append (['table'])

Creates a new record.  The L<Set> method should then be used to specify
the data in this record.  However, it is only writen to the table when
the L<Update> method is called.  If the table isn't specified, the
current table is used.

    $odbc->Append('Records');
    $odbc->Set('Artist', "Engelberg Humperdinkt");
    $odbc->Set('Record', "The yodel trance");
    $odbc->Update();

=item Abort ()

Kills the current L<Edit> or L<Append>.  While calling this is STRICTLY
nesecairy, one should to preserve sanity (because you might call L<Update>
later) and to clean up.

=item Update ()

Flushes any temporary data to the active table.  This adds a record if
L<Append> was called before, or modifies the record if L<Edit> was called.
Be aware of the wonky L<Edit> behaviour.  See L<BUGS!@#!>.

=item DeprepVal ($scalar)

Converts any escaped values to there original form.  ie %27 becomes '.
This function is automaticaly called by the L<Get> method, so you
probably don't need it.

=item PrepVal ($ref)

Prepares a value for use with the L<Set> method.  Arrays are
joined together, seperated by the L<$newline> class variable to form a
string.  Hashes under go a similar treatement, except each line is made
up of 'key: value'.  Prepval also excapes any caracters that can't be
handled in an SQL statement.  Read the perlref man page if you're not an
expert on references.

    $hello=$odbc->PrepVal(\@array);
    $odbc->Set('Memo_field', $hello);


=item Table (['table'])

Sets the current table to be used by future calls to L<Append>, L<Edit>,
L<FindFirst> and L<FindFirstX>.  It should B<NOT> be called during an
L<Append> or L<Edit> operation.  It can also be used to read the current
table.

    print "We are accessing ".$odbc->Table();

=item FindFirst (['clause'], ['table'])

Find the first record for a given search clause.  The current active
table is used if none is given.  The search clause is any valid SQL
WHERE clause.  I am far from being a SQL expert, so I can't really
enumerate them all here.  To find out if a record was found, call
L<Found>.  To proced to the next record in the search, call
L<FindNext>. If the clause is not given all records will be selected.
This is useful for steping through the entire table.

    $odbc->Sql("SELECT * FROM Comp_Books")  # see wonky Edit() bug
    $odbc->FindFirst('YEAR(start_date) IN (1991, 1992)', 'Comp_Books');
    while($odbc->Found())
    {
        $odbc->Edit();
        print $odbc->Set('Obsolete', 'Very!');
        $odbc->Update();

        $odbc->FindNext();
    }

=item FindFirstX ($sub_clause, ['table'])

This is similar to the L<FindFirst> method, except a Perl subroutine is
used to validate each record to see if it should be included in the
search.  The subroutine is passed a single parameter: a hash that has all
the field names as keys and their data as values.  To find out if a
record was found, call L<Found>.  To proced to the next record in the
search, call L<FindNext>.

I included this mainly so one could do some regex comparisons.

    $harold=sub    # Note no sub name!  Read perlref for an explanation
        {   local(%data)=@_;
            return ($data{'name'}=~/harold/i);
        }

    $odbc->FindFirst($harold, 'Kings_of_England');
    while($odbc->Found())
    {
        print $odbc->Get('name')."\n";
        print $odbc->Get('death')."\n";
        print $odbc->Get('major_wars')."\n";

        $odbc->FindNext();
    }


=item FindNext ()

Proceed to the next record in the current search.  Searchs are instigated
by L<FindFirst> and L<FindFirstX>.  See those methods for examples.

=item Found ()

This method revels wether there is currently a record available in the
current search.  See L<FindFirst> or L<FindFirstX> for examples.


=head1 BIGER EXAMPLE

    require ODBC::Ext;

    $odbc= new ODBC::Ext("Monarchs");

    $odbc->Append("Kings");
    $odbc->Set('name', 'dread fil');
    #    $odbc->Set('death', 'I'm not dead yet!');
    @feats=('Wrote some good CGIs in under 2 hours',
            'Learned more than 7 programing languages',
            'Played decent guitar',
            'Made a profit on self produced educational game',
            'Hitch-hiked across Canada (twice)',
            'Learned 3 human languages',
            'Became frightfuly witty and modest',
           );
    $odbc->Set('noted_actions', $odbc->PrepVal(\@feats));
    $odbc->Update();

    $odbc->Append("Queens");
    $odbc->Set('name', 'George the Beautious');
    $odbc->Set('death', '14/10/1969');
    @feets=('Charter member of the Entobicok LASM',
            'Never broke a heel',
            'Only beat up twice by those THRICE DAMNED gay bashers',
            'Learned not to tell construction workers not to F*** off',
            'Convinced mother to change nail polish'
            'NEVER appeared on Oprah'
           );
    $odbc->Set('noted_actions', $odbc->PrepVal(\@feets));
    $odbc->Abort();                     # this example is getting TO SILLY!!!

    $odbc->FindFirst('YEAR('death') < 1990');
    while($odbc->Found())
    {
        print 'Here's another moldy one: ', $odbc->Get('name'), "\n";
        $odbc->FindNext();
    }

    # this next example is brought to you by the letters M, L and silent-Q

    $sponsor=sub
        {   local(%data)=@;
            return ($data{'name'}=~/[ML]|(silent-Q)/);
        }

    $odbc->FindFirstX($sponsor, 'Kings, Queens');
    while($odbc->Found())
    {
        print 'Three cheers for ', $odbc->Get('name'), "\n";
        $odbc->FindNext();
    }



=head1 INSTALLATION

Copy the file Ext.pm to perl-lib-dir\Win32\ODBC\Ext.pm and away you go,
simple as that.

=head1 AVAILABILITY

I'll put it at http://www.interlinx.qc.ca/~fil/Perl, for now.  Eventualy
I hope to have it in CPAN.

=head1 BUGS!@#!

=item Wonky Edit behaviour.

Ok, there's no hiding it: I'm an SQL neophyte.  And I just could get enough
information to understand how to use cursors in SQL.  Sigh.  This means
the L<Update> method uses a WHERE clause to define which record should be
modified.  The WHERE clause is defined by all the data in all the fields
currently selected.  This could very well be insufficiant to uniquely
select the desired record.  B<So> you could very well end up modifing
more than one record.

To B<work around> this bit of sloppy coding, make sure you have enough
fields selected to uniquely define the record.  A Sql('SELECT * FROM
George'); usualy does the trick.  It should be noted that if your table
has totaly identical records, they will both be modified, no matter
what.  But then, databases as a rule shouldn't included such cases.

Of course, you may wish to make use of this I<feature>, but I can't
garanty that it will stick around. -:)

=item DataHash
If you bi-pass L<Get>, values will not be decoded.  See L<Edit>.

=item Other

Other bugs?  YES!  Probably many.  This code might interact in strange
ways with the Win32::ODBC package.  Your kilometerage may vary. When in
doubt test test test! and use the source!  I knocked this up in an
afternoon (in fact, this documentation is taking longer to write -:) and
don't have the time (nor the inclanation) to extensively debug it.  But
then, Perl is a very forgiving language, which is what I ADORE about it.
Long live Larry!

If you find anything out of order, or have comments or sugestions,
please by all means send them to me.  But don't expect me to drop
everything to fix a bug.

=head1 AUTHORS

Just lil' old me, so far. -:)  Philip Gwyn fil@interlinx.qc.ca

=head1 DISCLAIMER

This software is provided AS IS, without any warranty, neither explicit
nor implied as to it's suitablilty to any given task.  If you crash and
burn because of this software, that's your own look out.

