package POE::Component::IKC::Daemon;

############################################################
# $Id$
# Based on tests/refserver.perl
# Contributed by Artur Bergman <artur@vogon-solutions.com>
# Revised for 0.06 by Rocco Caputo <troc@netrus.net>
# Turned into a module by Philp Gwyn <fil@pied.nu>
#
# Copyright 1999 Philip Gwyn.  All rights reserved.
# This program is free software; you can redistribute it and/or modify
# it under the same terms as Perl itself.
#
# Contributed portions of IKC may be copyright by their respective
# contributors.  

use strict;
use Socket;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);
use POE qw(Wheel::ListenAccept Wheel::ReadWrite Wheel::SocketFactory
           Driver::SysRW Filter::Reference Filter::Line
          );
use POE::Component::IKC::Responder;

require Exporter;

@ISA = qw(Exporter);
@EXPORT = qw(create_ikc_daemon);
$VERSION = '0.01';

sub DEBUG { 1 }

###############################################################################
# Daemon instances are created by the listening session to handle
# connections.  They receive one or more thawed references, and pass
# them to the running Responder session for processing.

#----------------------------------------------------
# This is just a convenient way to create daemons.

sub create_ikc_daemon
{
  my($handle, $name, $on_connect) = @_;

  new POE::Session( _start => \&daemon_start,
                    _stop  => \&daemon_shutdown,
                    error  => \&daemon_error,

                    receive => \&daemon_receive,
                    'send' => \&daemon_send,
                    server_000 => \&server_000,
                    server_001 => \&negociate_001,
                    server_002 => \&server_002,
                    client_000 => \&client_000,
                    client_001 => \&negociate_001,
                    client_002 => \&client_002,
                    [ $handle, $name, $on_connect]
                  );
}

#----------------------------------------------------
# Accept POE's standard _start event, and begin processing data.

sub daemon_start 
{
    my ($kernel, $heap, $handle, $name, $on_connect) = 
                @_[KERNEL, HEAP, ARG0, ARG1, ARG2];

    my @name=unpack_sockaddr_in(getsockname($handle));
    $name[1]=inet_ntoa($name[1]);
    if($name)
    {
        $heap->{kernel_aliases}=[join '-', @name[1,0]];
        $heap->{kernel_name}=$name;
    } else
    {
        $heap->{kernel_aliases}=[];
        $heap->{kernel_name}=join '-', @name[1,0];
    }
    

    @name=unpack_sockaddr_in(getpeername($handle));
    $name[1]=inet_ntoa($name[1]);
    $heap->{remote_kernel}=join '-', @name[1,0];

    DEBUG && print "Daemon session $heap->{kernel_name}<->$heap->{remote_kernel} created.\n";

                                        # start reading and writing
    $heap->{wheel_client} = new POE::Wheel::ReadWrite
    ( Handle     => $handle,                    # on this handle
      Driver     => new POE::Driver::SysRW,     # using sysread and syswrite
      InputState => 'none',
      Filter     => POE::Filter::Line->new(),   # use a line filter for negociations
      ErrorState => 'error',            # generate this event on error
    );

    $heap->{on_connect}=$on_connect if ref($on_connect);
    _set_phase($kernel, $heap, '000');
}

#----------------------------------------------------
#### DEAL WITH NEGOCIATION PHASE
sub _set_phase
{
    my($kernel, $heap, $phase)=@_;
    if($phase eq 'ZZZ')
    {
        DEBUG && 
            print "Negociation done ($heap->{kernel_name}<->$heap->{remote_kernel}).\n";

        # generate this event on input
        $heap->{'wheel_client'}->event(InputState => 'receive');

        # For now, we use POE::Filter::Reference default
        #  parsing I/O as references
        $heap->{wheel_client}->new_filter(POE::Filter::Reference->new()); 

        # Register the foreign kernel with the responder

        create_ikc_responder();
        push @{$heap->{remote_aliases}}, $heap->{remote_kernel};
        $kernel->call('IKC', 'register', $heap->{remote_aliases});
    
        # Now that we're set up properly, we acknowledge the fact
        if($heap->{on_connect})
        {
            $heap->{on_connect}->();
            delete $heap->{on_connect};    
        }

        return
    } 

    my $neg='server_';
    $neg='client_' if($heap->{on_connect});

        # generate this event on input
    $heap->{'wheel_client'}->event(InputState => $neg.$phase);
    DEBUG && print "Negociation phase $neg$phase.\n";
    $kernel->yield($neg.$phase);               # Start the negociation phase
    return;
}

# First server state is
sub server_000
{
    my ($heap, $kernel, $line)=@_[HEAP, KERNEL, ARG0];

    unless(defined $line)
    {
        # wait for client to send HELLO

    } elsif($line eq 'HELLO')
    {
        $heap->{'wheel_client'}->put('IAM '.$heap->{kernel_name});   
                                          # put other server aliases here
        $heap->{aliases001}=[@{$heap->{kernel_aliases}}];   
        _set_phase($kernel, $heap, '001');

    } else    
    {
        warn "Client sent '$line' during phase 000 :(\n";
    }
    return;
}

# We tell who we are
sub negociate_001
{
    my ($heap, $kernel, $line)=@_[HEAP, KERNEL, ARG0];
    
    unless(defined $line)
    {

    } elsif($line eq 'OK')
    {
        my $a=pop @{$heap->{aliases001}};
        if($a)
        {
            $heap->{'wheel_client'}->put("IAM $a");
        } else
        {
            delete $heap->{aliases001};
            _set_phase($kernel, $heap, '002');
        }
    } else
    {
        warn "Recieved '$line' during phase 001 :(\n";
    }
    return;
}

# We find out who the client is
sub server_002
{
    my ($heap, $kernel, $line)=@_[HEAP, KERNEL, ARG0];

    unless(defined $line)
    {
        $heap->{'wheel_client'}->put('DONE');   

    } elsif($line eq 'DONE')
    {
        _set_phase($kernel, $heap, 'ZZZ');

    } elsif($line =~ /^IAM\s+([-.\w]+)$/)
    {   
        # Register this kernel alias with the responder
        push @{$heap->{remote_aliases}}, $1;
        $heap->{'wheel_client'}->put('OK');   

    } else    
    {
        warn "Client sent '$line' during phase 002 :(\n";
    }
    return;
}

#----------------------------------------------------
# These states is invoked for each line during the negociation phase on 
# the client's side

## Start negociation and listen to who the server is
sub client_000
{
    my ($heap, $kernel, $line)=@_[HEAP, KERNEL, ARG0];

    unless(defined $line)
    {
        $heap->{wheel_client}->put('HELLO');

    } elsif($line =~ /^IAM\s+([-.\w]+)$/)
    {   
        # Register this kernel alias with the responder
        push @{$heap->{remote_aliases}}, $1;
        $heap->{wheel_client}->put('OK');

    } elsif($line eq 'DONE')
    {
        $heap->{'wheel_client'}->put('IAM '.$heap->{kernel_name});   
        $heap->{aliases001}=[@{$heap->{kernel_aliases}}];
        _set_phase($kernel, $heap, '001');

    } else
    {
        warn "Server sent '$line' during negociation phase :(\n";
    }
    return;
}

# Game over
sub client_002
{
    my ($heap, $kernel, $line)=@_[HEAP, KERNEL, ARG0];

    unless(defined $line)
    {
        $heap->{'wheel_client'}->put('DONE');   
        _set_phase($kernel, $heap, 'ZZZ');

    } else    
    {
        warn "Server sent '$line' during phase 002 :(\n";
    }
    return;
}


#----------------------------------------------------
# This state is invoked for each error encountered by the session's
# ReadWrite wheel.

sub daemon_error 
{
    my ($heap, $kernel, $operation, $errnum, $errstr) =
        @_[HEAP, KERNEL, ARG0, ARG1, ARG2];

    if ($errnum) 
    {
        DEBUG && print "Daemon encountered $operation error $errnum: $errstr\n";
    }
    else 
    {
        DEBUG && print "The daemon's client closed its connection ($heap->{kernel_name}<->$heap->{remote_kernel})\n";
        $kernel->call('IKC', 'unregister', $heap->{remote_aliases});
        delete $heap->{remote_aliases};
    }
                                        # either way, shut down
    delete $heap->{wheel_client};
}

#----------------------------------------------------
# Process POE's standard _stop event by shutting down.

sub daemon_shutdown 
{
  my $heap = $_[ARG0];
  DEBUG && print "Daemon has shut down.\n";
  delete $heap->{wheel_client};
}

###########################################################################
## Next two events forward messages between Wheel::ReadWrite and the
## Responder
## Because the Responder know which foreign kernel sent a request,
## these events fill in some of the details.

#----------------------------------------------------
# Foreign kernel sent us a request
sub daemon_receive
{
    my ($kernel, $heap, $request) = @_[KERNEL, HEAP, ARG0];

    # we won't trust the other end to set this properly
    $request->{errors_to}={ kernel=>$heap->{remote_kernel},
                            session=>'IKC',
                            event=>'remote_error',
                          };
    # just in case
    $request->{call}->{kernel}||=$heap->{kernel_name};

    # call the Responder daemon to process
    $kernel->call('IKC', 'request', $request);
}

#----------------------------------------------------
# Local kernel is sending a request to a foreign kernel
sub daemon_send
{
    my ($heap, $request)=@_[HEAP, ARG0];

        # add our name so the foreign daemon can find us
        # TODO should we do this?  or should the other end do this?
    $request->{rsvp}->{kernel}||=$heap->{kernel_name}
            if $request->{rsvp};

    $heap->{'wheel_client'}->put($request);
}

###########################################################################


1;
__END__


=head1 NAME

POE::Component::IKC::Daemon - POE Inter-Kernel Communication I/O session

=head1 SYNOPSIS

    use POE;
    use POE::Component::IKC::Daemon;
    create_ikc_daemon($handle, $name, $on_connect);

=head1 DESCRIPTION

This module implements an POE IKC I/O.  
When a new connection
is established, C<IKC::Server> and C<IKC::Client> create an 
C<IKC::Daemon> to handle the I/O.  

=head1 EXPORTED FUNCTIONS

=item C<create_ikc_daemon($handle, $kernel_name, $on_connect)>

This function initiates all the work of connecting to a IKC connection
daemon.

IKC communication happens in 2 phases : negociation phase and normal phase.

The negociation phase uses C<Filter::Line> and is used to exchange various
parameters between kernels (example : kernel names, what type of freeze/thaw
to use, etc).  After negociation, C<IKC::Daemon> switches to a
C<Filter::Reference> and creates a C<IKC::Responder>, if needed.  After
this, the daemon forwards reads and writes between C<Wheel::ReadWrite> and
the Responder.  

C<IKC::Daemon> is also in charge of cleaning up kernel names when
the foreign kernel disconnects.

=over 3

=item C<$handle>

The perl handle we should hand to C<Wheel::ReadWrite::new>.

=item C<$kernel_name>

The name of the local kernel.  B<This is a stop-gap until event naming
has been resolved>.

=item C<$on_connect>

Code ref that is called when the negociation phase has terminated.  Normaly,
you would use this to start the sessions that post events to foreign
kernels.

=back

=head1 BUGS

=head1 AUTHOR

Philip Gwyn, <fil@pied.nu>

=head1 SEE ALSO

L<POE>, L<POE::Component::IKC::Server>, L<POE::Component::IKC::Client>,
L<POE::Component::IKC::Responder>


=cut
