package POE::Component::IKC::Responder;

############################################################
# $Id$
# Based on tests/refserver.perl
# Contributed by Artur Bergman <artur@vogon-solutions.com>
# Revised for 0.06 by Rocco Caputo <troc@netrus.net>
# Turned into a module by Philp Gwyn <fil@pied.nu>
#
# Copyright 1999 Philip Gwyn.  All rights reserved.
# This program is free software; you can redistribute it and/or modify
# it under the same terms as Perl itself.
#
# Contributed portions of IKC may be copyright by their respective
# contributors.  

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);
use POE qw(Session);

require Exporter;
@ISA = qw(Exporter);
@EXPORT = qw( create_ikc_responder);
$VERSION = '0.01';

sub DEBUG { 1 }

###############################################################################

#----------------------------------------------------
# This is just a convenient way to create only one responder.
{
    my $once=1;
    sub create_ikc_responder 
    {
        return unless $once--;
        new POE::Session( __PACKAGE__, [qw(
                      _start 
                      request post call    
                      remote_error
                      register unregister default  
                    )]);
    }
}

#----------------------------------------------------
# Accept POE's standard _start message, and start the responder.
sub _start
{
    my $kernel = $_[KERNEL];
    DEBUG && print "Responder started.\n";
    $kernel->alias_set('IKC');              # allow it to be called by name
}

#----------------------------------------------------
# Foreign kernel called something here
sub request
{
    my($kernel, $heap, $request) = @_[KERNEL, HEAP, ARG0];
    use Data::Dumper;
#    DEBUG && print Dumper $request;
    
    # We ignore the kernel for now, but we should really use it to decide
    # weither we should run the request or not
    my $to=$request->{call};
    eval
    {
        my $session=$kernel->alias_resolve($to->{session});
        die "Unknown session '$to->{session}'\n" unless $session;
        _thunked_post($request->{rsvp},
                      [$session, $to->{event}, $request->{params}]);
    };


    # Error handling consists of posting a "remote_error" event to
    # the foreign kernel.
    # $request->{errors_to} is set by the local IKC::Daemon
    if($@)                  
    {
        chomp($@);
        DEBUG && warn "Error in request: $@\n";
        $@.=' for '. join '::', @{$to}{qw(kernel session event)};
        unless($request->{is_error})    # don't send an error message back
        {                               # if this was an error
            _post($kernel, $heap, { call=>$request->{errors_to},
                                    params=>$@, is_error=>1,
                                  });
        }
    }
}

#----------------------------------------------------
# Register foreign kernels so that we can send events to them
sub register
{
    my($heap, $daemon, $names) = @_[HEAP, SENDER, ARG0];
    $names=[$names] if not ref $names;
    foreach my $name (@$names)
    {
        DEBUG && print "Registered kernel '$name'.\n";
        $heap->{kernel}->{$name}=$daemon;
        $heap->{default}||=$name;
    }
    return 1;
}

#----------------------------------------------------
# Register foreign kernels when this disconnect (say)
sub unregister
{
    my($heap, $daemon, $names) = @_[HEAP, SENDER, ARG0];
    $names=[$names] if not ref $names;
    foreach my $name (@$names)
    {
        DEBUG && print "Unregistered kernel '$name'.\n";
        delete $heap->{kernel}->{$name};
        $heap->{default}='' if $heap->{default} eq $name;
    }
    return 1;
}

#----------------------------------------------------
# Set a default foreign kernel to send events to
sub default
{
    my($heap, $name) = @_[HEAP, ARG0];
    unless($heap->{kernel}->{$name})
    {
        warn "We do not know the kernel $name.\n";
        return;
    }
    DEBUG && print "Default kernel $name.\n";
    $heap->{default}=$name;
    return 1;
}

###############################################################################
## These are the 3 events that interact with the foreign kernel

#----------------------------------------------------
# Internal function that does all the work of preparing a request to be
# sent
sub _post
{
    my($kernel, $heap, $msg)=@_;
    my $to=$msg->{call};

    warn "Need a session name.\n", return unless $to->{session};
    warn "Need an event name.\n", return unless $to->{event};

    my $name=$to->{kernel}||$heap->{'default'};
    die "Unable to decide which kernel to send event '$to->{event}' to.\n"
        unless $name;

    my $daemon=$heap->{kernel}->{$name};
    warn "Unknown kernel '$name'.\n", return unless $daemon;
    
    # This is where we should parse $msg->{params} to turn anything 
    # extravagant like a subref or $poe_kernel into a call back to us.

    DEBUG && print 'Remote call ', join('::', @{$to}{qw(kernel session event)}), "\n";
    # now send the message over the wire
    return $kernel->call($daemon, 'send', $msg);
}

#----------------------------------------------------
# Send a request to the foreign kernel
sub post
{
    my($kernel, $heap, $to, $params) = @_[KERNEL, HEAP, ARG0, ARG1];

    _post($kernel, $heap, {params=>$params, 'call'=>$to,} );
}

#----------------------------------------------------
# Send a request to the foreign kernel and ask it to provide 
# the event's return value back
sub call
{
    my($kernel, $heap, $to, $params, $rsvp) = 
                    @_[KERNEL, HEAP, ARG0, ARG1, ARG2];

    $rsvp={ session=>$_[SENDER], event=>$rsvp } unless ref $rsvp;

    _post($kernel, $heap, {params=>$params, 'call'=>$to, rsvp=>$rsvp, });
}

#----------------------------------------------------
# Remote kernel had an error 
sub remote_error
{
    my($heap, $params) = @_[HEAP, ARG0];

    warn "Remote error: $params\n";
}



##############################################################################
# These are Thunks used to post the actual event on behalf of the foreign
# kernel.  Currently, the thunks are used as a "proof of concept" and
# to accur extra over head. :)
# 
# The original idea was to make sure that $_[SENDER] would be something
# valid to the posted event.  However, garbage collection becomes a problem
# If we are to allow the poste session to "hold on to" $_[SENDER]. 
#
# Having the Responder save all Thunks until the related foreign connection
# disapears would be wasteful.  Or, POE could use soft-refs to track
# active sessions and the thunk would "kill" itself in _stop.  This
# would force POE to require 5.005, however.
#
# I think a much better idea would be IKC::Proxy sessions.  

*_thunked_post=\&POE::Component::IKC::Responder::Thunk::thunk;
package POE::Component::IKC::Responder::Thunk;

use strict;

use POE qw(Session);

sub DEBUG { 0 }

{
    my $name=__PACKAGE__.'0000';
    $name=~s/\W//g;
    sub thunk
    {
        POE::Session->new(__PACKAGE__, [qw(_start _stop _default)], 
                                   [$name++, @_]);
    }
}

sub _start
{
    my($kernel, $heap, $name, $caller, $args)=@_[KERNEL, HEAP, ARG0, ARG1, ARG2];
    $heap->{'caller'}=$caller;
    $heap->{name}=$name;
    DEBUG && print "$name created\n";
    DEBUG && print "$args->[1] posted\n";
    $kernel->post(@$args);
}

sub _stop
{
    DEBUG && print "$_[HEAP]->{name} delete\n";
}

sub _default
{
    my($kernel, $heap, $event, $args)=@_[KERNEL, HEAP, ARG0, ARG1];
    warn "Attempt to respond to a remote_post with $event\n"
        if not $heap->{rsvp};
    die "$heap->{name} $event called\n";
    $kernel->post('IKC', $event, $heap->{rsvp}, @$args);
}

1;
__END__

=head1 NAME

POE::Component::IKC::Responder - POE IKC event handler

=head1 SYNOPSIS

    use POE;
    use POE::Component::IKC::Responder;
    create_ikc_responder();
    ...
    $kernel->post('IKC', 'post', $to_event, $event);

=head1 DESCRIPTION

This module implements an POE IKC event handling.  The responder handles
posting events to foreign kernels and calling events in the local kernel at
the request of foreign kernels.

=head1 EVENTS

=head2 C<post>

Sends an event request to a foreign kernel.  Returns logical true if the
event was sent and logical false if it was unable to send the request to the 
foreign kernel.  This does not mean that the foreign kernel was able to 
post the event, however.  Parameters are as follows :

=over 2

=item C<foreign_event>

Identification of the foreign event.  Currently, this means a hash ref
of 3 key/value pairs.  C<kernel> is which foreign kernel (defaults to the
first registered kernel).  C<session> is the session alias on the foreign
kernel.  C<event> is the event name.
    
    {kernel=>'Timeserver', session=>'timer', event=>'register'}

This is the 'register' event in session 'timer' on kernel 'Timerserver'. 
Yes, this is overly verbose, but will have to do for now.  B<POE needs a
global event naming scheme!>

=item C<parameters>

A reference to anything you want the foreign event to get as ARG0.  If you
want to specify several parameters, use an array ref and have the foreign
event dereference it.

    $kernel->post('IKC', 'post', 
        {kernel=>'Syslog', session=>'logger', event=>'log'},
        [$faculty, $priority, $message];

This logs an event with a hypothetical logger.  

=back

=head2 C<call>

This is identical to C<post>, except it has a 3rd parameter that describes
what event should receive the return value from the foreign kernel.

B<call doesn't work yet!>

=over 3

=item C<foreign_event>

Identical to the C<post> C<foreign_event> parameter.

=item C<parameters>

Identical to the C<post> C<parameters> parameter.

=item C<rsvp>

Event identification for the callback.  That is, this event is called with
the return value of the foreign event.  Can be a C<foreign_event> specifier
or simply the name of an event in the current session.

=back

    $kernel->call('IKC', 'post', 
        {kernel=>'e-comm', session=>'CC', event=>'check'},
        {CC=>$cc, expiry=>$expiry}, folder=>$holder},
        'is_valid');

This asks the e-comm server to check if a credit card number is "well
formed".  Yes, this would probably be massive overkill. 

=head2 C<default>

Sets the default foreign kernel.  You must be connected to the foreign
kernel first.

Unique parameter is the name of the foreign kernel kernel.

Returns logical true on success.

=head2 C<register>

Registers foreign kernel names with the responder.  This is done during the
negociation phase of IKC and is normaly handled by C<IKC::Daemon>.  Will
define the default kernel if no previous default kernel exists.

Unique parameter is either a single kernel name or an array ref of kernel
names to be registered.

=head2 C<unregister>

Unregisters one or more foreign kernel names with the responder.  This is
done when the foreign kernel disconnects by C<IKC::Daemon>. If this is the
default kernel, there is no more default kernel.

Unique parameter is either a single kernel name or an array ref of kernel
names to be unregistered.

=head1 EXPORTED FUNCTIONS

=item C<create_ikc_responder>

This function creates the Responder session.  However, you don't need to
call this directly, because C<IKC::Daemon> does this for
you.

=head1 BUGS

C<call> event doesn't work yet.  

The event specifier format sucks.

Sending session references to a foreign kernel is a bad idea :)  At some
point it would be desirable to recurse through the paramerters and and turn
any session references into event specifiers.

=head1 AUTHOR

Philip Gwyn, <fil@pied.nu>

=head1 SEE ALSO

L<POE>, L<POE::Component::IKC::Server>

=cut
