package POE::Component::IKC::Responder;

############################################################
# $Id$
# Based on tests/refserver.perl
# Contributed by Artur Bergman <artur@vogon-solutions.com>
# Revised for 0.06 by Rocco Caputo <troc@netrus.net>
# Turned into a module by Philp Gwyn <fil@pied.nu>
#
# Copyright 1999 Philip Gwyn.  All rights reserved.
# This program is free software; you can redistribute it and/or modify
# it under the same terms as Perl itself.
#
# Contributed portions of IKC may be copyright by their respective
# contributors.  

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK $ikc_kernel);
use Carp;
use Data::Dumper;

use POE qw(Session);
use POE::Component::IKC::Proxy;
use POE::Component::IKC::Specifier;

require Exporter;
@ISA = qw(Exporter);
@EXPORT = qw(create_ikc_responder);
$VERSION = '0.02';

sub DEBUG { 1 }
sub DEBUG2 { 0 }

##############################################################################

#----------------------------------------------------
# This is just a convenient way to create only one responder.
sub create_ikc_responder 
{
    return if $ikc_kernel;
    new POE::Session( __PACKAGE__ , [qw(
                      _start 
                      request post call    
                      remote_error
                      register unregister default  
                      publish retract subscribe unsubscribe
                      do_you_have
                    )]);
}

#----------------------------------------------------
# Accept POE's standard _start message, and start the responder.
sub _start
{
    my($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
    DEBUG && print "Responder started.\n";
    $kernel->alias_set('IKC');              # allow it to be called by name
    $heap->{'local'}={IKC=>{remote_error=>1, do_you_have=>1}};

    $ikc_kernel=1;
}

#----------------------------------------------------
# Foreign kernel called something here
sub request
{
    my($kernel, $heap, $request) = @_[KERNEL, HEAP, ARG0];
    
    # We ignore the kernel for now, but we should really use it to decide
    # weither we should run the request or not
    my $to=specifier_parse($request->{event});
    eval
    {
        my $args=$request->{params};
        # allow proxied states to have multiple ARGs
        if($to->{state} eq 'IKC:proxy')   
        {
            $to->{state}=$args->[0];
            $args=$args->[1];
            DEBUG2 && print "IKC proxied request for ", specifier_name($to), "\n";
                
        } else
        {
            DEBUG2 && print "IKC request for ", specifier_name($to), "\n";
            $args=[$args];
        }
          
        # find out if the state we want to get at has been published
        if(exists $heap->{rsvp}{$to->{session}} and
           $heap->{rsvp}{$to->{session}}{$to->{state}}
          ) {
            $heap->{rsvp}{$to->{session}}{$to->{state}}--;
        }
        elsif(not exists $heap->{'local'}{$to->{session}}) {
            die "Session $to->{session} is not available for remote kernels\n";
        } 
        elsif(not exists $heap->{'local'}{$to->{session}}{$to->{state}}) {
            die "Session $to->{session} has not published state ",
                $to->{state}, "\n";
        }


        my $session=$kernel->alias_resolve($to->{session});
        die "Unknown session '$to->{session}'\n" unless $session;

        _thunked_post($request->{rsvp},
                      [$session, $to->{state}, @$args]);
    };


    # Error handling consists of posting a "remote_error" state to
    # the foreign kernel.
    # $request->{errors_to} is set by the local IKC::Channel
    if($@)                  
    {
        chomp($@);
        my $err=$@.' ['.specifier_name($to).']';
        DEBUG && warn "Error in request: $err\n";
        unless($request->{is_error})    # don't send an error message back
        {                               # if this was an error
            _post($kernel, $heap, { event=>$request->{errors_to},
                                    params=>$err, is_error=>1,
                                  });
        } else
        {
            warn Dumper $request;
        }
    }
}

#----------------------------------------------------
# Register foreign kernels so that we can send states to them
sub register
{
    my($heap, $channel, $names) = @_[HEAP, SENDER, ARG0];
    $names=[$names] if not ref $names;
    my($rid, @todo)=@$names;
    
    $heap->{channel}{$rid}=$channel;
    $heap->{remote}{$rid}=[];
    $heap->{alias}{$rid}=[@$names];

    DEBUG && print "Registered kernel '$rid' ";
    $heap->{default}||=$rid;

    foreach my $name (@todo)
    {
        DEBUG && print $name, ' ';
        $heap->{kernel}{$name}=$rid;
        $heap->{remote}{$name}||=[];
    }

    DEBUG && print "\n";
    return 1;
}

#----------------------------------------------------
# Register foreign kernels when this disconnect (say)
sub unregister
{
    my($kernel, $heap, $daemon, $names) = @_[KERNEL, HEAP, SENDER, ARG0];
    $names=[$names] if not ref $names;

    my @todo=@$names;
    my $name;
    while(@todo)
    {
        $name=shift @todo;
        if($heap->{channel}{$name})       # this is in fact the real name
        {
                                    # so we delete aliases too
            push @todo, @{$heap->{alias}{$name}};    
            $heap->{default}='' if $heap->{default} eq $name;

            # TODO : close the channel
            delete $heap->{channel}{$name};
            delete $heap->{alias}{$name};

            DEBUG && print "Unregistered kernel '$name'.\n";

        } elsif($heap->{kernel}{$name})
        {
            DEBUG && print "Unregistered kernel alias '$name'.\n";
            delete $heap->{kernel}{$name};
            $heap->{default}='' if $heap->{default} eq $name;
        } else
        {
            # already gone...
        }
                # tell the proxies they are no longer needed
        foreach my $alias (@{$heap->{remote}{$name}})
        {
            $kernel->post($alias, '_delete');
        }
        delete $heap->{remote}{$name};
    }
    return 1;
}

#----------------------------------------------------
# Set a default foreign channel to send messages to
sub default
{
    my($heap, $name) = @_[HEAP, ARG0];
    if(exists $heap->{kernel}{$name})
    {
        $heap->{default}=$heap->{kernel}{$name};

    } elsif(exists $heap->{channel}{$name})
    {
        $heap->{default}=$name;

    } else
    {
        warn "We do not know the kernel $name.\n";
        return;
    }

    DEBUG && print "Default kernel is on channel $name.\n";
    return 1;
}

##############################################################################
## These are the 3 states that interact with the foreign kernel

#----------------------------------------------------
# Internal function that does all the work of preparing a request to be
# sent
sub _post
{
    my($kernel, $heap, $msg)=@_;
    my $e=$msg->{rsvp} ? 'call' : 'post';

    my $to=specifier_parse($msg->{event});
    unless($to) {
        croak "Bad state ", Dumper $msg                     
    }
    unless($to) {
        warn "Bad or missing 'to' parameter '$msg->{event}' to poe:/IKC/$e\n";
        return;
    }
    unless($to->{session}) {
        warn "Need a session name in poe:/IKC/$e";
        return;
    }
    unless($to->{state})   {
        warn "Need an state name in poe:IKC/$e";
        return;
    }

    my $name=$to->{kernel}||$heap->{'default'};
    unless($name) {
        warn "Unable to decide which kernel to send state '$to->{state}' to.";
        return;
    }

    DEBUG2 && print "poe:/IKC/$e to ", specifier_name($to), "\n";

    my @channels;
        ## Ugh... turn a kernel into channel....
    if($name eq '*') {                              # any kernel
        @channels=values(%{$heap->{channel}}); 
    }  
    elsif(exists $heap->{kernel}{$name}) {          # kernel alias
        my $t=$heap->{kernel}{$name};
        unless(exists $heap->{channel}{$t}) {
            die "What happened to channel $t!";
        }
        @channels=$heap->{channel}{$t}; 
    }
    elsif(exists $heap->{channel}{$name}) {         # kernel ID
        @channels=$heap->{channel}{$name}; 
    }
    warn "Unknown kernel '$name'.\n", return unless @channels;
    

    # We need to be able to access this state w/out forcing folks
    # to use publish
    if($msg->{rsvp})
    {
        my $rsvp=$msg->{rsvp};
        DEBUG && print "Allow $rsvp->{session}/$rsvp->{state} once\n";
        $heap->{rsvp}{$rsvp->{session}}{$rsvp->{state}}++;
    }


    # This is where we should recurse $msg->{params} to turn anything 
    # extravagant like a subref, $poe_kernel, session etc into a call back to 
    # us.

    
    # now send the message over the wire
    # hmmm.... i wonder if this could be stream-lined into a direct call
    my $count=0;
    foreach my $channel (@channels)
    {
        $count+=$kernel->call($channel, 'send', $msg) ? 1 : 0;
    }
    DEBUG2 && print specifier_name($to), " sent to $count kernel(s).\n";
    return $count;
}

#----------------------------------------------------
# Send a request to the foreign kernel
sub post
{
    my($to, $params) = @_[ARG0, ARG1];

    $to="poe:$to" unless ref $to or $to=~/^poe:/;

    _post($_[KERNEL], $_[HEAP], {params=>$params, 'event'=>$to,}, 
          $_[SENDER]);
}

#----------------------------------------------------
# Send a request to the foreign kernel and ask it to provide 
# the state's return value back
sub call
{
    my($kernel, $heap, $sender, $to, $params, $rsvp) = 
                    @_[KERNEL, HEAP, SENDER, ARG0, ARG1, ARG2];

    $to="poe:$to"     if $to   and not ref $to   and $to!~/^poe:/;
    $rsvp="poe:$rsvp" if $rsvp and not ref $rsvp and $rsvp!~/^poe:/;

    my $t=specifier_parse($rsvp);
    unless($t)
    {
        if($rsvp)
        {
            warn "Bad 'rsvp' parameter '$rsvp' in poe:/IKC/call\n";
        } else
        {
            warn "Missing 'rsvp' parameter in poe:/IKC/call\n";
        }
        return;
    }
    $rsvp=$t;
    unless($rsvp->{state})
    {
        warn "rsvp state not set in poe:/IKC/call\n";
        return;
    }

    # Question : should $svp->{session} be forced to be the sender?
    # or will we allow people to point callbacks to other poe:kernel/sessions
    $rsvp->{session}||=$sender->{ID};                   # maybe a session ID?
    if(not $rsvp->{session})                            # no session alias
    {
        die "IKC call requires session IDs, please patch your version of POE\n";
    }
    DEBUG2 && print "RSVP is ", specifier_name($rsvp), "\n";

    _post($kernel, $heap, {params=>$params, 'event'=>$to, rsvp=>$rsvp, }, 
          $sender);
}

#----------------------------------------------------
# Remote kernel had an error 
sub remote_error
{
    my($heap, $params) = @_[HEAP, ARG0];

    warn "Remote error: $params\n";
}

##############################################################################
# publish/retract/subscribe mechanism of setting up foreign sessions

#----------------------------------------------------
sub publish
{
    my($kernel, $heap, $sender, $states, $session)=
        @_[KERNEL, HEAP, SENDER, ARG0, ARG1];

    $session||=$kernel->ID_lookup($sender);
    $heap->{'local'}->{$session}||={};
    my $p=$heap->{'local'}->{$session};
    foreach my $q (@$states)
    {
        DEBUG && print "Published poe:/$session/$q\n";
        $p->{$q}=1;
    }
    return 1;
}

#----------------------------------------------------
sub retract
{
    my($kernel, $heap, $sender, $states, $session)=
        @_[KERNEL, HEAP, SENDER, ARG0, ARG1];

    $session||=$kernel->ID_lookup($sender);
    return 0 unless $heap->{'local'}->{$session};

    my $p=$heap->{'local'}->{$session};
    foreach my $q (@$states)
    {
        delete $p->{$q};
    }
    return 1;
}

#----------------------------------------------------
# Subscribing is in two phases
# 1- we call a IKC/do_you_have to the foreign kernels
# 2- the foreign responds with the session-specifier (if it has published it)
#
# We create a unique state for the callback for each subscription request
# from the user session.  It keeps count of how many subscription receipts
# it receives and when they are all subscribed, it localy posts the callback
# event.  
#
# If more then one kernel sends a subscription receipt
sub subscribe
{
    my($kernel, $heap, $sender, $sessions, $callback)=
                @_[KERNEL, HEAP, SENDER, ARG0, ARG1];
    $sessions=[$sessions] unless ref $sessions;
    return unless @$sessions;

    my $s_id=$sender->{'ID'};
    $callback||='';

    my($ses, $s, $fiddle);
                                # unique identifier for this request
    my $unique="IKC:receipt $s_id $callback";  
    my $id=$kernel->ID;

    my $count;
    foreach my $spec (@$sessions)
    {
        $ses=specifier_parse($spec);   # Session specifier
                                    # Create the subscription receipt state
        $kernel->state($unique.$spec, 
                       sub {_subscribe_receipt($unique, $spec, @_)} );
        $kernel->delay(60, $unique.$spec);

        if($ses->{kernel})
        {
            $count=_post($kernel, $heap, 
                    {event=>{kernel=>$ses->{kernel}, session=>'IKC', 
                                state=>'do_you_have'},
                     params=>[$ses, $id],
                     rsvp=>{kernel=>$id, session=>'IKC', state=>$unique.$spec }
                    }
                 );
            # TODO What if this post failed?  Session that posted this would
            # surely want to know
        } else
        {                       # Bleh.  User shouldn't be that dumb       
            die "You must specify a kernel for $spec in poe:/IKC/subscribe\n";
        }
        

        if($callback)           # We need to keep some information around
        {                       # for when the subscription receipt comes in
            $heap->{subscription_callback}{$unique}||=
                        {   sender=>$sender, callback=>$callback, 
                            sessions=>{}, yes=>[], 'no'=>[], count=>0, 
                            states=>{},
                        };
            $fiddle=$heap->{subscription_callback}{$unique};
            $fiddle->{states}{$unique.$spec}=$count;
            $fiddle->{count}+=($count||0);
            $fiddle->{sessions}->{$spec}=1;
            if(not $count)
            {
                $kernel->yield($unique.$spec);
            }
        }
        
    }
    return 1;
}

# Called by a foreign IKC session 
# We respond with the session, or with "NOT $specifier";
sub do_you_have
{
    my($kernel, $heap, $param)=@_[KERNEL, HEAP, ARG0];
    my $ses=specifier_parse($param->[0]);
    die "Bad state $param->[0]\n" unless $ses;
    
    DEBUG && print "Wants to subscribe to ", specifier_name($ses), "\n";
    if(exists $heap->{'local'}->{$ses->{session}} and
       (not $ses->{state} or 
        exists $heap->{'local'}->{$ses->{session}}->{$ses->{state}}
       ))
    {
        $ses->{kernel}||=$kernel->ID;      # uniquely identify this session
        return $ses;
    } else
    {
        DEBUG && print specifier_name($ses), " is not published in this kernel\n";
        return "NOT ".specifier_name($ses);
    }
}

# Subscription receipt
# All foreign kernel's that have published the desired session
# will send back a receipt.  
# Others will send a "NOT".
# This will cause problems when the Proxy session creates an alias :(
#
# Callback is called we are "done".  But what is "done"?  When at least
# one remote kernel has allowed us to subscribe to each session we are
# waiting for.  However, at some point we should give up.
# 
# Scenarios :
# one foreign kernel says 'yes', one 'no'.
#   - 'yes' creates a proxy
#   - 'no' decrements wait count 
#       ... callback is called with session specifier
# 2 foreign kernels says 'yes'
#   - first 'yes' creates a proxy
#   - 2nd 'yes' should also create a proxy!  alias conflict (for now)
#       ... callback is called with session specifier
# one foreign kernel says 'no', and after, another says yes
#   - first 'no' decrements wait count
#   - second 'no' decrements wait count
#       ... Subscription failed!  callback is called with specifier
# no answers ever came...
#   - we wait forever :(

sub _subscribe_receipt
{
    my $unique=shift;
    my $spec=shift;
    my($kernel, $heap, $ses)=@_[KERNEL, HEAP, ARG0];
    my $status='yes';

    if(not $ses or not ref $ses)
    {
        warn "Refused to subscribe to $spec\n";
        $status='no';
    } else
    {
        $ses=specifier_parse($ses);
        die "Bad state" unless $ses;

        DEBUG && print "Create proxy for ", specifier_name($ses), "\n";
        my $proxy=create_ikc_proxy($ses->{kernel}, $ses->{session});

        push @{$heap->{remote}->{$ses->{kernel}}}, $proxy;
    }

    # fiddle with subscription callback    
    if(exists $heap->{subscription_callback}{$unique})
    {
        DEBUG && print "Subscription callback... ";
        my $fiddle=$heap->{subscription_callback}{$unique};
        if($fiddle->{sessions}->{$spec} and $status eq 'yes')
        {
            delete $fiddle->{sessions}->{$spec};
            push @{$fiddle->{sessions}->{$status}}, $spec;
        }

        $fiddle->{count}-- if $fiddle->{count};
        if(0==$fiddle->{count})
        {
            DEBUG && print "yes.\n";
            delete $heap->{subscription_callback}{$unique};
            $kernel->post(  $fiddle->{sender}, 
                            $fiddle->{callback}, 
                            $fiddle->{yes}
                         );
        } else
        {
            DEBUG && print "no, $fiddle->{count} left.\n";
        }
        
        $fiddle->{states}{$unique.$spec}--;
        if($fiddle->{states}{$unique.$spec}<=0)
        {
            # this state is no longer needed
            $kernel->state($unique.$spec);
            delete $fiddle->{states}{$unique.$spec};
        }
    } else
    {
        # this state is no longer needed
        $kernel->state($unique.$spec);
    }
}

#----------------------------------------------------
sub unsubscribe
{
    my($kernel, $heap, $states)=@_[KERNEL, HEAP, ARG0];
    $states=[$states] unless ref $states;
    return unless @$states;
}


##############################################################################
# These are Thunks used to post the actual state on behalf of the foreign
# kernel.  Currently, the thunks are used as a "proof of concept" and
# to accur extra over head. :)
# 
# The original idea was to make sure that $_[SENDER] would be something
# valid to the posted state.  However, garbage collection becomes a problem
# If we are to allow the poste session to "hold on to" $_[SENDER]. 
#
# Having the Responder save all Thunks until the related foreign connection
# disapears would be wasteful.  Or, POE could use soft-refs to track
# active sessions and the thunk would "kill" itself in _stop.  This
# would force POE to require 5.005, however.
#

*_thunked_post=\&POE::Component::IKC::Responder::Thunk::thunk;
package POE::Component::IKC::Responder::Thunk;

*_post=\&POE::Component::IKC::Responder::_post;

use strict;

use POE qw(Session);

sub DEBUG { 0 }

{
    my $name=__PACKAGE__.'00000000';
    $name=~s/\W//g;
    sub thunk
    {
        POE::Session->new(__PACKAGE__, [qw(_start _stop _default)], 
                                   [$name++, @_]);
    }
}

sub _start
{
    my($kernel, $heap, $name, $caller, $args)=@_[KERNEL, HEAP, ARG0, ARG1, ARG2];
    $heap->{'caller'}=$caller;
    $heap->{name}=$name;
    DEBUG && print "$name created\n";
    DEBUG && print "$args->[1] posted\n";
    if($caller)                             # foreign session wants return
    {
        my $ret=$kernel->call(@$args);
        $kernel->call('IKC', 'post', $caller, $ret) if defined $ret;
    } else
    {
        $kernel->call(@$args);
    }
}

sub _stop
{
    DEBUG && print "$_[HEAP]->{name} delete\n";
}

sub _default
{
    my($kernel, $heap, $state, $args)=@_[KERNEL, HEAP, ARG0, ARG1];
    warn "Attempt to respond to a remote_post with $state\n"
        if not $heap->{rsvp};
    die "$heap->{name} $state called\n";
    $kernel->post('IKC', $state, $heap->{rsvp}, @$args);
}

1;
__END__

=head1 NAME

POE::Component::IKC::Responder - POE IKC state handler

=head1 SYNOPSIS

    use POE;
    use POE::Component::IKC::Responder;
    create_ikc_responder();
    ...
    $kernel->post('IKC', 'post', $to_state, $state);

=head1 DESCRIPTION

This module implements an POE IKC state handling.  The responder handles
posting states to foreign kernels and calling states in the local kernel at
the request of foreign kernels.

=head1 EVENTS

=head2 C<post>

Sends an state request to a foreign kernel.  Returns logical true if the
state was sent and logical false if it was unable to send the request to the 
foreign kernel.  This does not mean that the foreign kernel was able to 
post the state, however.  Parameters are as follows :

=over 2

=item C<foreign_state>

Identification of the foreign state.  Currently, this means a hash ref
of 3 key/value pairs.  C<kernel> is which foreign kernel (defaults to the
first registered kernel).  C<session> is the session alias on the foreign
kernel.  C<state> is the state name.
    
    {kernel=>'Timeserver', session=>'timer', state=>'register'}

This is the 'register' state in session 'timer' on kernel 'Timerserver'. 
Yes, this is overly verbose, but will have to do for now.  B<POE needs a
global state naming scheme!>

=item C<parameters>

A reference to anything you want the foreign state to get as ARG0.  If you
want to specify several parameters, use an array ref and have the foreign
state dereference it.

    $kernel->post('IKC', 'post', 
        {kernel=>'Syslog', session=>'logger', state=>'log'},
        [$faculty, $priority, $message];

This logs an state with a hypothetical logger.  

=back

=head2 C<call>

This is identical to C<post>, except it has a 3rd parameter that describes
what state should receive the return value from the foreign kernel.

B<call doesn't work yet!>

=over 3

=item C<foreign_state>

Identical to the C<post> C<foreign_state> parameter.

=item C<parameters>

Identical to the C<post> C<parameters> parameter.

=item C<rsvp>

Event identification for the callback.  That is, this state is called with
the return value of the foreign state.  Can be a C<foreign_state> specifier
or simply the name of an state in the current session.

=back

    $kernel->call('IKC', 'post', 
        {kernel=>'e-comm', session=>'CC', state=>'check'},
        {CC=>$cc, expiry=>$expiry}, folder=>$holder},
        'is_valid');

This asks the e-comm server to check if a credit card number is "well
formed".  Yes, this would probably be massive overkill. 

=head2 C<default>

Sets the default foreign kernel.  You must be connected to the foreign
kernel first.

Unique parameter is the name of the foreign kernel kernel.

Returns logical true on success.

=head2 C<register>

Registers foreign kernel names with the responder.  This is done during the
negociation phase of IKC and is normaly handled by C<IKC::Channel>.  Will
define the default kernel if no previous default kernel exists.

Unique parameter is either a single kernel name or an array ref of kernel
names to be registered.

=head2 C<unregister>

Unregisters one or more foreign kernel names with the responder.  This is
done when the foreign kernel disconnects by C<IKC::Channel>. If this is the
default kernel, there is no more default kernel.

Unique parameter is either a single kernel name or an array ref of kernel
names to be unregistered.

=head1 EXPORTED FUNCTIONS

=item C<create_ikc_responder>

This function creates the Responder session.  However, you don't need to
call this directly, because C<IKC::Channel> does this for
you.

=head1 BUGS

C<call> state doesn't work yet.  

The state specifier format sucks.

Sending session references to a foreign kernel is a bad idea :)  At some
point it would be desirable to recurse through the paramerters and and turn
any session references into state specifiers.

=head1 AUTHOR

Philip Gwyn, <fil@pied.nu>

=head1 SEE ALSO

L<POE>, L<POE::Component::IKC::Server>

=cut
