package POE::Component::IKC::Server;

############################################################
# $Id$
# Based on refserver.perl
# Contributed by Artur Bergman <artur@vogon-solutions.com>
# Revised for 0.06 by Rocco Caputo <troc@netrus.net>
# Turned into a module by Philp Gwyn <fil@pied.nu>
#
# Copyright 1999 Philip Gwyn.  All rights reserved.
# This program is free software; you can redistribute it and/or modify
# it under the same terms as Perl itself.
#
# Contributed portions of IKC may be copyright by their respective
# contributors.  

use strict;
use Socket;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);
use POE qw(Wheel::ListenAccept Wheel::SocketFactory);
use POE::Component::IKC::Channel;
use POE::Component::IKC::Responder;

require Exporter;

@ISA = qw(Exporter);
@EXPORT = qw(create_ikc_server);
$VERSION = '0.05';

sub DEBUG { 0 }


###############################################################################
#----------------------------------------------------
# This is just a convenient way to create servers.  To be useful in
# multi-server situations, it probably should accept a bind address
# and port.
sub create_ikc_server
{
    my(%parms)=@_;
    $parms{ip}||='0.0.0.0';            # INET_ANY
    $parms{port}||=603;                # POE! (almost :)
    create_ikc_responder();
    new POE::Session( _start   => \&server_start,
#                   _stop   => sub {warn "server _stop\n"},
                    error    => \&server_error,
                    'accept' => \&server_accept,
                    [\%parms],
                  );
}

#----------------------------------------------------
# Accept POE's standard _start event, and set up the listening socket
# factory.

sub server_start 
{
    my($heap, $parms) = @_[HEAP, ARG0];

    DEBUG && print "Server starting $parms->{ip}:$parms->{port}.\n";
                                        # create a socket factory
    $heap->{wheel} = new POE::Wheel::SocketFactory
    ( BindPort       => $parms->{port},
      BindAddress    => $parms->{ip},
      Reuse          => 'yes',          # and allow immediate reuse of the port
      SuccessState   => 'accept',       # generating this event on connection
      FailureState   => 'error'         # generating this event on error
    );
    $heap->{name}=$parms->{name};
}

#----------------------------------------------------
# Log server errors, but don't stop listening for connections.  If the
# error occurs while initializing the factory's listening socket, it
# will exit anyway.

sub server_error 
{
    my ($operation, $errnum, $errstr) = @_[ARG0, ARG1, ARG2];
    DEBUG && print "Server encountered $operation error $errnum: $errstr\n";
}

#----------------------------------------------------
# The socket factory invokes this state to take care of accepted
# connections.

sub server_accept 
{
    my ($heap, $handle, $peer_host, $peer_port) = @_[HEAP, ARG0, ARG1, ARG2];

    DEBUG &&
        print "Server connection from ", inet_ntoa($peer_host), " $peer_port\n";
                                        # give the connection to a channel
    create_ikc_channel($handle, $heap->{name});
}



1;
__END__
# Below is the stub of documentation for your module. You better edit it!

=head1 NAME

POE::Component::IKC::Server - POE Inter-kernel Communication server

=head1 SYNOPSIS

    use POE;
    use POE::Component::IKC::Server;
    create_ikc_server(
        ip=>$ip, 
        port=>$port,
        name=>'Server',);
    ...
    $poe_kernel->run();

=head1 DESCRIPTION

This module implements a POE IKC server.  A IKC server listens for incoming
connections from IKC clients.  When a client connects, it negociates certain
connection parameters.  After this, the POE server and client are pretty much
identical.

=head1 EXPORTED FUNCTIONS

=item C<create_ikc_server>

This function initiates all the work of building the IKC server.  
Parameters are :

=over 3

=item C<ip>

Address to listen on.  Can be a doted-quad ('127.0.0.1') or a host name
('foo.pied.nu').  Defaults to '0.0.0.0', aka INADDR_ANY.

=item C<port>

Port to listen on.  Can be numeric (80) or a service ('http').

=item C<name>

Local kernel name.  This is how we shall "advertise" ourself to foreign
kernels. It acts as a "kernel alias".  This parameter is temporary, pending
the addition of true kernel names in the POE core.

=back

=head1 BUGS

=head1 AUTHOR

Philip Gwyn, <fil@pied.nu>

=head1 SEE ALSO

L<POE>, L<POE::Component::IKC::Client>

=cut
