package POEx::URI;

use strict;
use warnings;

use URI::Escape qw(uri_unescape);
use URI::_server;
use Carp;

use vars qw( @ISA $VERSION );
@ISA = qw(URI::_server);
$VERSION = '0.0100';

sub default_port { 603 }

##############################################
sub kernel
{
    my $self = shift;
    my $old = $self->authority;
    if( @_ ) {
        my $tmp = $old;
        $tmp = "" unless defined $old;
        my $ui = ($tmp =~ /(.*@)/) ? $1 : "";
        my $new = shift;
        $new = "" unless defined $new;
        if (length $new) {
            $new =~ s/[@]/%40/g;   # protect @
        }
        if( $ui or length $new ) {
            $self->authority( "$ui$new" );
        }
        else {
            $self->authority( undef );
        }
    }
    return undef unless defined $old;
    $old =~ s/.*@//;
    return uri_unescape($old);
}


##############################################
sub path
{
    my $self = shift;
    my $old = $self->SUPER::path;
    if( @_ ) {
        my $new = shift;
        if( $new =~ m,(.+)/(.+), ) {
            
            my $session = $1;
            my $event = $2;
            $session =~ s,^/+,,;
            $session =~ s,/,%2F,g;
            $new = join '/', $session, $event;
        }
        $self->SUPER::path( $new );
    }
    return $old;
}

##############################################
sub path_segments
{
    my $self = shift;

    my @seg = $self->SUPER::path_segments;
    if( @_ ) {
        my @new = @_;
        shift @new if $new[0] eq '';
        if( 2 <= @new ) {
            my $event = pop @new;
            @new = ( join( '/', @new ), $event );
        }
        $self->SUPER::path_segments( @new );
    }
    return @seg;
}

##############################################
sub session
{
    my $self = shift;
    my @seg = $self->path_segments;
    shift @seg if defined $seg[0] and $seg[0] eq '';
    my $event;    
    if( 1==@seg ) {     # only an event?
        $event = $seg[0];
        @seg = ();        
    }   
    if( @seg >= 2 ) {   # session + event
        $event = pop @seg;
    }
    my $old = join '/', @seg[0..$#seg];

    if( @_ ) { 
        my $new = shift;
        $new = '' unless defined $new;
        $self->path_segments( $new, (defined $event ? $event : '' ) );
    }   

    return $old;
}

##############################################
sub event
{
    my $self = shift;
    my $old = ( $self->path_segments )[-1];
    $old = '' unless defined $old;
    if( @_ ) {
        my @seg = $self->path_segments;
        my $new = shift;
        if( @seg >= 2 ) {           # session/event
            $seg[-1] = $new||'';
        }
        elsif( @seg ) {             # session
            push @seg, $new||'';
        }
        else {                      # nothing
            if( $self->kernel and defined $new ) {
                carp "It makes no sense to set an event without a session";
            }
            @seg = ('', $new||'');
        }
        
        $self->path_segments( @seg );
    }
    return $old;
}

##############################################
sub _user
{
    my $self = shift;
    my $old = $self->userinfo;
    $old =~ s/:.*$//;
    return $old;
}
sub user
{
    my $self = shift;
    my $old = $self->userinfo;
    $old =~ s/:.*$//;

    if( @_ ) {
        my $pw = $self->_password;
        my $new = shift;
        my $ui = $new;
        if( defined $new ) {
            $new =~ s/:/%3A/g;
            $ui = $new;
            $ui .= ":$pw" if( defined $pw );
        }
        elsif( defined $pw ) {
            $ui = ":$pw";
        }
        $self->userinfo( $ui );
    }

    $old =~ s/%3A/:/g if $old;
    return $old;
}

##############################################
sub _password
{
    my $self = shift;
    my $old = $self->userinfo;
    undef( $old ) unless $old =~ s/^.*?://;
    return $old;
}
sub password
{
    my $self = shift;
    my $old = $self->userinfo;
    undef( $old ) unless $old =~ s/^.*?://;

    if( @_ ) {
        my $user = $self->_user;
        $user = '' unless defined $user;
        my $new = shift;
        if( defined $new ) {
            $new =~ s/:/%3A/g;
            $self->userinfo( "$user:$new" );
        }
        else {
            $self->userinfo( $user );
        }
    }
    $old =~ s/%3A/:/g if $old;
    return $old;
}

##############################################
sub _is_inet
{
    my $kernel = shift;
    return unless $kernel;
    return 1 if $kernel =~ /:\d*$/;
    return 1 if $kernel =~ /^\d+\.\d+\.\d+\.\d+/;
    return 1 if $kernel =~ /^[-\w.]+$/;
}
sub canonical
{
    my( $self ) = @_;
    my $other = $self->URI::_generic::canonical();

    my $kernel = $self->kernel;
    if( _is_inet( $kernel ) ) {
        $other = $other->clone if $other == $self;
        $other->kernel( lc $kernel );
    }
    my $port = $other->_port;
    if( defined($port) && ($port eq "" || $port == $self->default_port) ) {
        $other = $other->clone if $other == $self;
        $other->port(undef);
    }

    if( $other =~ m(poe:/[^/]) ) {
        $other = $other->clone if $other == $self;
        $$other =~ s(poe:/)(poe:);
        
    }

    return $other;
}

##############################################
sub query
{
    croak "->query() currently not supported";
}

sub query_form
{
    croak "->query_form() currently not supported";
}

sub fragment
{
    croak "->fragment() currently not supported";

}

1;
__END__

=head1 NAME

POEx::URI - URI extension for POE event specfiers

=head1 SYNOPSIS

    use URI;

    my $uri = URI->new( "$session/$event" );

    $poe_kernel->post( $uri->session, $uri->event, @args );
    $poe_kernel->post( @$uri, @args );

    $uri->host( $host );
    $uri->port( 33100 );
    $poe_kernel->post( IKC => $uri, @args );


=head1 DESCRIPTION

This module implements the URIs that reference POE session/event tuples.
Objects of this class represent non-standard "Uniform Resource Identifier
references" of the I<poe:> scheme.

The canonical forms of POE URIs are:

    poe:event
    poe:session/
    poe:session/event
    poe://kernel/session/
    poe://kernel/session/event

Event parameters are currently not supported.

URI fragements (the bits after C<#>) make no sense.


=head1 METHODS

=head2 event

    my $name = $uri->event
    $old = $uri->event( $name );

Sets and returns the event part of the $uri.  If the C<$name> contains a
forward-slash (/), it is escaped (%2F).

To clear the event name, use C<''> or C<undef>, which are equivalent.

=head2 session

    my $name = $uri->session
    $old = $uri->session( $name );

Sets and returns the session part of the $uri.  If the C<$name> contains a
forward-slash (/), it is escaped (%2F).

To clear the event name, use C<''> or C<undef>, which are equivalent

=head2 kernel

    my $kernel = $uri->kernel;
    $old = $uri->kernel( $name );

Sets and returns the kernel part of the $uri.  A kernel may be host:port,
host (IKC), path to unix socket (IKC) or a local or remote kernel ID or
alias.

To clear the kernel name, use C<''> or C<undef>, which are equivalent.

=head2 host

    $host = $uri->host;
    $old = $uri->host( $host );

Sets and returns the host part of the $uri's kernel.  If the kernel wasn't 
host:port, then it is converted to that.

=head2 port

    $port = $uri->port;
    $old = $uri->port( $port );

Sets and retuns the port part of the $uri's kernel.  If the kernel wasn't a
host name, then it becomes one.

=head2 default_port

The default POE port is 603 which is POE upside-down and backwards.  Almost.

=head2 user

    $user = $uri->user;
    $old = $uri->user( $user );

Sets and returns the username part of the $uri's L<URI/userinfo>.  If the
user name contains C<:>, it is escaped.

=head2 password

    $pw = $uri->password;
    $old = $uri->password( $passwd );

Sets and returns the password part of the $uri's L<URI/userinfo>.  If the
password contains C<:>, it is escaped.

The user name and password are seperated by C<:>.  This is might be a security
issue.  Beware.

While this method is called I<password>, it works just as well with pass
phrases.

=head2 canonical

    my $full = $uri->canonical;

Returns a normalized version of the URI.  For POE URIs, the hostname is 
folded to lower case.


=head2 path

    $path = $uri->path;
    $old = $uri->path( $new_path );

Sets and returns the session/event tupple of a $uri.  If the new path
contains more then one slash, the last segment of the path is the event, and
the others are the session and those slash are escaped.

=head2 path_segments

    ( $session, $event ) = $uri->path_segments;
    @old = $uri->path_segments( @new );

Sets and returns the path.  In a scalar context, it returns the same value
as $uri->path.  In a list context, it returns the unescaped path segments
that make up the path.  See L<URI/path_segments> for more details.

=head1 SEE ALSO

L<POE>, L<URI>.

=head1 AUTHOR

Philip Gwyn, E<lt>gwyn -at- cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2009 by Philip Gwyn

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.8 or,
at your option, any later version of Perl 5 you may have available.


=cut
