/***********************************************************************
 Copyright 2002 Ben Rudiak-Gould.

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA,
 or visit <http://www.gnu.org/copyleft/gpl.html>.
***********************************************************************/


#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#pragma hdrstop
#include <windowsx.h>
#include <commdlg.h>
#include <process.h>
#include "ui.h"
#include "logscsi.h"
#include "resource.h"


#include <stdio.h>


HINSTANCE g_hinstance;

DvsDockingBayGlobal* g_callbacks;


const char vendor_specific[] = "vendor-specific";
const char undefined[] = "unknown";

const char* scsi_command_names_00_5D[] = {
/*00*/   "TEST UNIT READY",
/*01*/   "REZERO UNIT",
/*02*/   vendor_specific,
/*03*/   "REQUEST SENSE",
/*04*/   "FORMAT UNIT",
/*05*/   "READ BLOCK LIMITS",
/*06*/   vendor_specific,
/*07*/   "REASSIGN BLOCKS",
/*08*/   "READ(6)",
/*09*/   vendor_specific,
/*0A*/   "WRITE(6)",
/*0B*/   "SEEK(6)",
/*0C*/   vendor_specific,
/*0D*/   vendor_specific,
/*0E*/   vendor_specific,
/*0F*/   vendor_specific,
/*10*/   "SYNCHRONIZE BUFFER / WRITE FILEMARKS",
/*11*/   "SPACE",
/*12*/   "INQUIRY",
/*13*/   "VERIFY(6)",
/*14*/   "RECOVER BUFFERED DATA",
/*15*/   "MODE SELECT(6)",
/*16*/   "RESERVE",
/*17*/   "RELEASE",
/*18*/   "COPY",
/*19*/   "ERASE",
/*1A*/   "MODE SENSE(06)",
/*1B*/   "STOP/START UNIT",
/*1C*/   "RECEIVE DIAGNOSTIC RESULTS",
/*1D*/   "SEND DIAGNOSTIC",
/*1E*/   "PREVENT/ALLOW MEDIUM REMOVAL",
/*1F*/   undefined,
/*20*/   vendor_specific,
/*21*/   vendor_specific,
/*22*/   vendor_specific,
/*23*/   "READ FORMAT CAPACITIES",
/*24*/   "SET WINDOW",
/*25*/   "READ CAPACITY",
/*26*/   vendor_specific,
/*27*/   vendor_specific,
/*28*/   "READ(10)",
/*29*/   "READ GENERATION",
/*2A*/   "WRITE(10)",
/*2B*/   "SEEK(10)",
/*2C*/   "ERASE(10)",
/*2D*/   "READ UPDATED BLOCK",
/*2E*/   "WRITE AND VERIFY(10)",
/*2F*/   "VERIFY(10)",
/*30*/   "SEARCH DATA HIGH(10)",
/*31*/   "SEARCH DATA EQUAL(10)",
/*32*/   "SEARCH DATA LOW(10)",
/*33*/   "SET LIMITS(10)",
/*34*/   "PRE-FETCH",
/*35*/   "SYNCHRONIZE CACHE",
/*36*/   "LOCK/UNLOCK CACHE",
/*37*/   "READ DEFECT DATA(10)",
/*38*/   "MEDIUM SCAN",
/*39*/   "COMPARE",
/*3A*/   "COPY AND VERIFY",
/*3B*/   "WRITE BUFFER",
/*3C*/   "READ BUFFER",
/*3D*/   "UPDATE BLOCK",
/*3E*/   "READ LONG",
/*3F*/   "WRITE LONG",
/*40*/   "CHANGE DEFINITION",
/*41*/   "WRITE SAME",
/*42*/   "READ SUB-CHANNEL",
/*43*/   "READ TOC/PMA/ATIP",
/*44*/   "READ HEADER",
/*45*/   "PLAY AUDIO(10)",
/*46*/   "GET CONFIGURATION",
/*47*/   "PLAY AUDIO MSF",
/*48*/   "PLAY AUDIO TRACK INDEX",
/*49*/   "PLAY TRACK RELATIVE(10)",
/*4A*/   "GET EVENT/STATUS NOTIFICATION",
/*4B*/   "PAUSE/RESUME",
/*4C*/   "LOG SELECT",
/*4D*/   "LOG SENSE",
/*4E*/   "STOP PLAY/SCAN",
/*4F*/   undefined,
/*50*/   undefined,
/*51*/   "READ DISC INFORMATION",
/*52*/   "READ TRACK/RZONE INFORMATION",
/*53*/   "RESERVE TRACK",
/*54*/   "SEND OPC INFORMATION",
/*55*/   "MODE SELECT(10)",
/*56*/   "RESERVE(10)",
/*57*/   "RELEASE(10)",
/*58*/   "REPAIR TRACK",
/*59*/   "READ MASTER CUE",
/*5A*/   "MODE SENSE(10)",
/*5B*/   "CLOSE TRACK/SESSION",
/*5C*/   "READ BUFFER CAPACITY",
/*5D*/   "SEND CUE SHEET",
};

const char* scsi_command_names_A0_BF[] = {
/*A0*/   "REPORT LUNS",
/*A1*/   "BLANK",
/*A2*/   "VERIFY(12) / SEND EVENT",
/*A3*/   "SEND KEY",
/*A4*/   "REPORT KEY",
/*A5*/   "PLAY AUDIO(12)",
/*A6*/   "LOAD/UNLOAD MEDIUM",
/*A7*/   "SET READ-AHEAD",
/*A8*/   "READ(12)",
/*A9*/   "PLAY TRACK RELATIVE(12)",
/*AA*/   "WRITE(12)",
/*AB*/   undefined,
/*AC*/   "ERASE(12) / GET PERFORMANCE",
/*AD*/   "READ DVD STRUCTURE",
/*AE*/   "WRITE AND VERIFY(12)",
/*AF*/   "VERIFY(12)",
/*B0*/   "SEARCH DATA HIGH(12)",
/*B1*/   "SEARCH DATA EQUAL(12)",
/*B2*/   "SEARCH DATA LOW(12)",
/*B3*/   "SET LIMITS(12)",
/*B4*/   undefined,
/*B5*/   "REQUEST VOLUME ELEMENT ADDRESS",
/*B6*/   "SEND VOLUME TAG / SET STREAMING",
/*B7*/   "READ DEFECT DATA(12)",
/*B8*/   "READ ELEMENT STATUS",
/*B9*/   "READ CD MSF",
/*BA*/   "SCAN",
/*BB*/   "SET CD SPEED",
/*BC*/   "PLAY CD",
/*BD*/   "MECHANISM STATUS",
/*BE*/   "READ CD",
/*BF*/   "SEND DVD STRUCTURE",
};

const char* srb_status_names[] = {
/*00*/   "SRB_STATUS_PENDING",
/*01*/   "SUCCESS",
/*02*/   "SRB_STATUS_ABORTED",
/*03*/   "SRB_STATUS_ABORT_FAILED",
/*04*/   "ERROR",
/*05*/   "SRB_STATUS_BUSY",
/*06*/   "SRB_STATUS_INVALID_REQUEST",
/*07*/   "SRB_STATUS_INVALID_PATH_ID",
/*08*/   "SRB_STATUS_NO_DEVICE",
/*09*/   "SRB_STATUS_TIMEOUT",
/*0A*/   "SRB_STATUS_SELECTION_TIMEOUT",
/*0B*/   "SRB_STATUS_COMMAND_TIMEOUT",
/*0C*/   0,
/*0D*/   "SRB_STATUS_MESSAGE_REJECTED",
/*0E*/   "SRB_STATUS_BUS_RESET",
/*0F*/   "SRB_STATUS_PARITY_ERROR",
/*10*/   "SRB_STATUS_REQUEST_SENSE_FAILED",
/*11*/   "SRB_STATUS_NO_HBA",
/*12*/   "SRB_STATUS_DATA_OVERRUN",
/*13*/   "SRB_STATUS_UNEXPECTED_BUS_FREE",
/*14*/   "SRB_STATUS_PHASE_SEQUENCE_FAILURE",
/*15*/   "SRB_STATUS_BAD_SRB_BLOCK_LENGTH",
/*16*/   "SRB_STATUS_REQUEST_FLUSHED",
/*17*/   0,
/*18*/   0,
/*19*/   0,
/*1A*/   0,
/*1B*/   0,
/*1C*/   0,
/*1D*/   0,
/*1E*/   0,
/*1F*/   0,
/*20*/   "SRB_STATUS_INVALID_LUN",
/*21*/   "SRB_STATUS_INVALID_TARGET_ID",
/*22*/   "SRB_STATUS_BAD_FUNCTION",
/*23*/   "SRB_STATUS_ERROR_RECOVERY",
/*24*/   "SRB_STATUS_NOT_POWERED"
};

const char* target_status_names[] = {
/*00*/   "GOOD",
/*02*/   "CHECK CONDITION",
/*04*/   "CONDITION MET",
/*06*/   0,
/*08*/   "BUSY",
/*0A*/   0,
/*0C*/   0,
/*0E*/   0,
/*10*/   "INTERMEDIATE",
/*12*/   0,
/*14*/   "INTERMEDIATE COND MET",
/*16*/   0,
/*18*/   "RESERVATION CONFLICT",
/*1A*/   0,
/*1C*/   0,
/*1E*/   0,
/*20*/   0,
/*22*/   "COMMAND TERMINATED",
/*24*/   0,
/*26*/   0,
/*28*/   "QUEUE FULL"
};

const char* sense_key_names[] = {
   "NO SENSE",
   "RECOVERED ERROR",
   "NOT READY",
   "MEDIUM ERROR",
   "HARDWARE ERROR",
   "ILLEGAL REQUEST",
   "UNIT ATTENTION",
   "DATA PROTECT",
   "BLANK CHECK",
   "VENDOR-SPECIFIC",
   "COPY ABORTED",
   "ABORTED COMMAND",
   "EQUAL",
   "VOLUME OVERFLOW",
   "MISCOMPARE",
   "RESERVED"
};

struct { scsi_result_t result; const char* text; }
result_code_names[] = {
  { 0x00000, "NO ADDITIONAL SENSE INFORMATION" },
  { 0x00001, "FILEMARK DETECTED" },
  { 0x00002, "END-OF-PARTITION/MEDIUM DETECTED" },
  { 0x00003, "SETMARK DETECTED" },
  { 0x00004, "BEGINNING-OF-PARTITION/MEDIUM DETECTED" },
  { 0x00005, "END-OF-DATA DETECTED" },
  { 0x00011, "AUDIO PLAY OPERATION IN PROGRESS" },
  { 0x00012, "AUDIO PLAY OPERATION PAUSED" },
  { 0x00013, "AUDIO PLAY OPERATION SUCCESSFULLY COMPLETED" },
  { 0x00014, "AUDIO PLAY OPERATION STOPPED DUE TO ERROR" },
  { 0x00015, "NO CURRENT AUDIO STATUS TO RETURN" },
  { 0x00016, "OPERATION IN PROGRESS" },
  { 0x10B00, "WARNING" },
  { 0x10B01, "WARNING - SPECIFIED TEMPERATURE EXCEEDED" },
  { 0x10B02, "WARNING - ENCLOSURE DEGRADED" },
  { 0x10C0A, "WRITE ERROR - PADDING BLOCKS ADDED" },
  { 0x11700, "RECOVERED DATA WITH NO ERROR CORRECTION APPLIED" },
  { 0x11701, "RECOVERED DATA WITH RETRIES" },
  { 0x11702, "RECOVERED DATA WITH POSITIVE HEAD OFFSET" },
  { 0x11703, "RECOVERED DATA WITH NEGATIVE HEAD OFFSET" },
  { 0x11704, "RECOVERED DATA WITH RETRIES AND/OR CIRC APPLIED" },
  { 0x11705, "RECOVERED DATA USING PREVIOUS SECTOR ID" },
  { 0x11706, "RECOVERED DATA WITHOUT ECC - DATA AUTO-REALLOCATED" },
  { 0x11707, "RECOVERED DATA WITHOUT ECC - RECOMMEND REASSIGNMENT" },
  { 0x11708, "RECOVERED DATA WITHOUT ECC - RECOMMEND REWRITE" },
  { 0x11709, "RECOVERED DATA WITHOUT ECC - DATA REWRITTEN" },
  { 0x11800, "RECOVERED DATA WITH ERROR CORRECTION APPLIED" },
  { 0x11801, "RECOVERED DATA WITH ERROR CORR. & RETRIES APPLIED" },
  { 0x11802, "RECOVERED DATA - DATA AUTO-REALLOCATED" },
  { 0x11803, "RECOVERED DATA WITH CIRC" },
  { 0x11804, "RECOVERED DATA WITH L-EC" },
  { 0x11805, "RECOVERED DATA - RECOMMEND REASSIGNMENT" },
  { 0x11806, "RECOVERED DATA - RECOMMEND REWRITE" },
  { 0x11807, "RECOVERED DATA WITH ECC - DATA REWRITTEN" },
  { 0x11808, "RECOVERED DATA WITH LINKING" },
  { 0x11E00, "RECOVERED ID WITH ECC CORRECTION" },
  { 0x13700, "ROUNDED PARAMETER" },
  { 0x15D00, "FAILURE PREDICTION THRESHOLD EXCEEDED - PREDICTED LOGICAL UNIT FAILURE" },
  { 0x15D01, "FAILURE PREDICTION THRESHOLD EXCEEDED - PREDICTED MEDIA FAILURE" },
  { 0x15D02, "LOGICAL UNIT FAILURE PREDICTION THRESHOLD EXCEEDED" },
  { 0x15DFF, "FAILURE PREDICTION THRESHOLD EXCEEDED (FALSE)" },
  { 0x16A00, "INFORMATIONAL, REFER TO LOG" },
  { 0x17301, "POWER CALIBRATION AREA ALMOST FULL" },
  { 0x17306, "PROGRAM MEMORY AREA/RMA IS (ALMOST) FULL" },
  { 0x20400, "LOGICAL UNIT NOT READY, CAUSE NOT REPORTABLE" },
  { 0x20401, "LOGICAL UNIT IS IN PROCESS OF BECOMING READY" },
  { 0x20402, "LOGICAL UNIT NOT READY, INITIALIZING CMD. REQUIRED" },
  { 0x20403, "LOGICAL UNIT NOT READY, MANUAL INTERVENTION REQUIRED" },
  { 0x20404, "LOGICAL UNIT NOT READY, FORMAT IN PROGRESS" },
  { 0x20405, "LOGICAL UNIT NOT READY, REBUILD IN PROGRESS" },
  { 0x20406, "LOGICAL UNIT NOT READY, RECALCULATION IN PROGRESS" },
  { 0x20407, "LOGICAL UNIT NOT READY, OPERATION IN PROGRESS" },
  { 0x20408, "LOGICAL UNIT NOT READY, LONG WRITE IN PROGRESS" },
  { 0x20500, "LOGICAL UNIT DOES NOT RESPOND TO SELECTION" },
  { 0x20600, "NO REFERENCE POSITION FOUND (MEDIUM MAY BE UPSIDE DOWN)" },
  { 0x23000, "INCOMPATIBLE MEDIUM INSTALLED" },
  { 0x23001, "CANNOT READ MEDIUM - UNKNOWN FORMAT" },
  { 0x23002, "CANNOT READ MEDIUM - INCOMPATIBLE FORMAT" },
  { 0x23003, "CLEANING CARTRIDGE INSTALLED" },
  { 0x23004, "CANNOT WRITE MEDIUM - UNKNOWN FORMAT" },
  { 0x23005, "CANNOT WRITE MEDIUM - INCOMPATIBLE FORMAT" },
  { 0x23006, "CANNOT FORMAT MEDIUM - INCOMPATIBLE MEDIUM" },
  { 0x23007, "CLEANING FAILURE" },
  { 0x23502, "ENCLOSURE SERVICES UNAVAILABLE" },
  { 0x23A00, "MEDIUM NOT PRESENT" },
  { 0x23A01, "MEDIUM NOT PRESENT - TRAY CLOSED" },
  { 0x23A02, "MEDIUM NOT PRESENT - TRAY OPEN" },
  { 0x23B11, "MEDIUM MAGAZINE NOT ACCESSIBLE" },
  { 0x23E00, "LOGICAL UNIT HAS NOT SELF-CONFIGURED YET" },
  { 0x25302, "MEDIUM REMOVAL PREVENTED" },
  { 0x26800, "LOGICAL UNIT NOT CONFIGURED" },
  { 0x30200, "NO SEEK COMPLETE" },
  { 0x30300, "PERIPHERAL DEVICE WRITE FAULT" },
  { 0x30301, "NO WRITE CURRENT" },
  { 0x30302, "EXCESSIVE WRITE ERRORS" },
  { 0x30600, "NO REFERENCE POSITION FOUND" },
  { 0x30C00, "WRITE ERROR" },
  { 0x30C01, "WRITE ERROR - RECOVERED WITH AUTO REALLOCATION" },
  { 0x30C02, "WRITE ERROR - AUTO REALLOCATION FAILED" },
  { 0x30C03, "WRITE ERROR - RECOMMEND REASSIGNMENT" },
  { 0x30C04, "COMPRESSION CHECK MISCOMPARE ERROR" },
  { 0x30C05, "DATA EXPANSION OCCURRED DURING COMPRESSION" },
  { 0x30C06, "BLOCK NOT COMPRESSIBLE" },
  { 0x30C07, "WRITE ERROR - RECOVERY NEEDED" },
  { 0x30C08, "WRITE ERROR - RECOVERY FAILED" },
  { 0x30C09, "WRITE ERROR - LOSS OF STREAMING" },
  { 0x30C0A, "WRITE ERROR - PADDING BLOCKS ADDED" },
  { 0x31000, "ID CRC OR ECC ERROR" },
  { 0x31100, "UNRECOVERED READ ERROR" },
  { 0x31101, "READ RETRIES EXHAUSTED" },
  { 0x31102, "ERROR TOO LONG TO CORRECT" },
  { 0x31103, "MULTIPLE READ ERRORS" },
  { 0x31104, "UNRECOVERED READ ERROR - AUTO REALLOCATE FAILED" },
  { 0x31105, "L-EC UNCORRECTABLE ERROR" },
  { 0x31106, "CIRC UNRECOVERED ERROR" },
  { 0x31107, "RE-SYNCHRONIZATION ERROR" },
  { 0x31108, "INCOMPLETE BLOCK READ" },
  { 0x31109, "NO GAP FOUND" },
  { 0x3110A, "MISCORRECTED ERROR" },
  { 0x3110B, "UNRECOVERED READ ERROR - RECOMMEND REASSIGNMENT" },
  { 0x3110C, "UNRECOVERED READ ERROR - RECOMMEND REWRITE THE DATA" },
  { 0x3110D, "DE-COMPRESSION CRC ERROR" },
  { 0x3110E, "CANNOT DECOMPRESS USING DECLARED ALGORITHM" },
  { 0x3110F, "ERROR READING UPC/EAN NUMBER" },
  { 0x31110, "ERROR READING ISRC NUMBER" },
  { 0x31200, "ADDRESS MARK NOT FOUND FOR ID FIELD" },
  { 0x31300, "ADDRESS MARK NOT FOUND FOR DATA FIELD" },
  { 0x31400, "RECORDED ENTITY NOT FOUND" },
  { 0x31401, "RECORD NOT FOUND" },
  { 0x31402, "FILEMARK OR SETMARK NOT FOUND" },
  { 0x31403, "END-OF-DATA NOT FOUND" },
  { 0x31404, "BLOCK SEQUENCE ERROR" },
  { 0x31405, "RECORD NOT FOUND - RECOMMEND REASSIGNMENT" },
  { 0x31406, "RECORD NOT FOUND - DATA AUTO-REALLOCATED" },
  { 0x31500, "RANDOM POSITIONING ERROR" },
  { 0x31501, "MECHANICAL POSITIONING ERROR" },
  { 0x31502, "POSITIONING ERROR DETECTED BY READ OF MEDIUM" },
  { 0x31600, "DATA SYNCHRONIZATION MARK ERROR" },
  { 0x31601, "DATA SYNC ERROR - DATA REWRITTEN" },
  { 0x31602, "DATA SYNC ERROR - RECOMMEND REWRITE" },
  { 0x31603, "DATA SYNC ERROR - DATA AUTO-REALLOCATED" },
  { 0x31604, "DATA SYNC ERROR - RECOMMEND REASSIGNMENT" },
  { 0x31900, "DEFECT LIST ERROR" },
  { 0x31901, "DEFECT LIST NOT AVAILABLE" },
  { 0x31902, "DEFECT LIST ERROR IN PRIMARY LIST" },
  { 0x31903, "DEFECT LIST ERROR IN GROWN LIST" },
  { 0x31F00, "PARTIAL DEFECT LIST TRANSFER" },
  { 0x32D00, "OVERWRITE ERROR ON UPDATE IN PLACE" },
  { 0x33100, "MEDIUM FORMAT CORRUPTED" },
  { 0x33101, "FORMAT COMMAND FAILED" },
  { 0x33102, "ZONED FORMATTING FAILED DUE TO SPARE LINKING" },
  { 0x33200, "NO DEFECT SPARE LOCATION AVAILABLE" },
  { 0x33201, "DEFECT LIST UPDATE FAILURE" },
  { 0x33300, "TAPE LENGTH ERROR" },
  { 0x33600, "RIBBON, INK, OR TONER FAILURE" },
  { 0x33B00, "SEQUENTIAL POSITIONING ERROR" },
  { 0x33B01, "TAPE POSITION ERROR AT BEGINNING-OF-MEDIUM" },
  { 0x33B02, "TAPE POSITION ERROR AT END-OF-MEDIUM" },
  { 0x33B03, "TAPE OR ELECTRONIC VERTICAL FORMS UNIT NOT READY" },
  { 0x33B06, "FAILED TO SENSE TOP-OF-FORM" },
  { 0x33B07, "FAILED TO SENSE BOTTOM-OF-FORM" },
  { 0x33B08, "REPOSITION ERROR" },
  { 0x33B09, "READ PAST END OF MEDIUM" },
  { 0x33B0A, "READ PAST BEGINNING OF MEDIUM" },
  { 0x33B0B, "POSITION PAST END OF MEDIUM" },
  { 0x33B0C, "POSITION PAST BEGINNING OF MEDIUM" },
  { 0x35100, "ERASE FAILURE" },
  { 0x35200, "CARTRIDGE FAULT" },
  { 0x35700, "UNABLE TO RECOVER TABLE-OF-CONTENTS" },
  { 0x35C02, "SPINDLES NOT SYNCHRONIZED" },
  { 0x36100, "VIDEO ACQUISITION ERROR" },
  { 0x36101, "UNABLE TO ACQUIRE VIDEO" },
  { 0x36102, "OUT OF FOCUS" },
  { 0x36C00, "REBUILD FAILURE OCCURRED" },
  { 0x36D00, "RECALCULATE FAILURE OCCURRED" },
  { 0x37100, "DECOMPRESSION EXCEPTION LONG ALGORITHM ID" },
  { 0x37200, "SESSION FIXATION ERROR" },
  { 0x37201, "SESSION FIXATION ERROR WRITING LEAD-IN" },
  { 0x37202, "SESSION FIXATION ERROR WRITING LEAD-OUT" },
  { 0x37300, "CD CONTROL ERROR" },
  { 0x37302, "POWER CALIBRATION AREA IS FULL" },
  { 0x37303, "POWER CALIBRATION AREA ERROR" },
  { 0x37304, "PROGRAM MEMORY AREA/RMA UPDATE FAILURE" },
  { 0x37305, "PROGRAM MEMORY AREA/RMA IS FULL" },
  { 0x37306, "RMA/PMA IS FULL" },
  { 0x40017, "CLEANING REQUESTED" },
  { 0x40100, "NO INDEX/SECTOR SIGNAL" },
  { 0x40500, "LOGICAL UNIT DOES NOT RESPOND TO SELECTION" },
  { 0x40800, "LOGICAL UNIT COMMUNICATION FAILURE" },
  { 0x40801, "LOGICAL UNIT COMMUNICATION TIME-OUT" },
  { 0x40802, "LOGICAL UNIT COMMUNICATION PARITY ERROR" },
  { 0x40803, "LOGICAL UNIT COMMUNICATION CRC ERROR (ULTRA-DMA/32)" },
  { 0x40900, "TRACK FOLLOWING ERROR" },
  { 0x40901, "TRACKING SERVO FAILURE" },
  { 0x40902, "FOCUS SERVO FAILURE" },
  { 0x40903, "SPINDLE SERVO FAILURE" },
  { 0x40904, "HEAD SELECT FAULT" },
  { 0x41500, "RANDOM POSITIONING ERROR" },
  { 0x41501, "MECHANICAL POSITIONING ERROR" },
  { 0x41B00, "SYNCHRONOUS DATA TRANSFER ERROR" },
  { 0x41C00, "DEFECT LIST NOT FOUND" },
  { 0x41C01, "PRIMARY DEFECT LIST NOT FOUND" },
  { 0x41C02, "GROWN DEFECT LIST NOT FOUND" },
  { 0x43400, "ENCLOSURE FAILURE" },
  { 0x43500, "ENCLOSURE SERVICES FAILURE" },
  { 0x43503, "ENCLOSURE SERVICES TRANSFER FAILURE" },
  { 0x43B04, "SLEW FAILURE" },
  { 0x43B05, "PAPER JAM" },
  { 0x43B16, "MECHANICAL POSITIONING OR CHANGER ERROR" },
  { 0x43E01, "LOGICAL UNIT FAILURE" },
  { 0x43E02, "TIMEOUT ON LOGICAL UNIT" },
  { 0x44000, "RAM FAILURE" },
  { 0x44100, "DATA PATH FAILURE" },
  { 0x44200, "POWER-ON OR SELF-TEST FAILURE" },
  { 0x44400, "INTERNAL TARGET FAILURE" },
  { 0x44600, "UNSUCCESSFUL SOFT RESET" },
  { 0x44700, "SCSI PARITY ERROR" },
  { 0x44A00, "COMMAND PHASE ERROR" },
  { 0x44B00, "DATA PHASE ERROR" },
  { 0x44C00, "LOGICAL UNIT FAILED SELF-CONFIGURATION" },
  { 0x45300, "MEDIA LOAD OR EJECT FAILED" },
  { 0x45400, "SCSI TO HOST SYSTEM INTERFACE FAILURE" },
  { 0x46000, "LAMP FAILURE" },
  { 0x46200, "SCAN HEAD POSITIONING ERROR" },
  { 0x46500, "VOLTAGE FAULT" },
  { 0x46600, "AUTOMATIC DOCUMENT FEEDER COVER UP" },
  { 0x46601, "AUTOMATIC DOCUMENT FEEDER LIFT UP" },
  { 0x46602, "DOCUMENT JAM IN AUTOMATIC DOCUMENT FEEDER" },
  { 0x46603, "DOCUMENT MISS FEED AUTOMATIC IN DOCUMENT FEEDER" },
  { 0x46700, "CONFIGURATION FAILURE" },
  { 0x46701, "CONFIGURATION OF INCAPABLE LOGICAL UNITS FAILED" },
  { 0x46702, "ADD LOGICAL UNIT FAILED" },
  { 0x46703, "MODIFICATION OF LOGICAL UNIT FAILED" },
  { 0x46704, "EXCHANGE OF LOGICAL UNIT FAILED" },
  { 0x46705, "REMOVE OF LOGICAL UNIT FAILED" },
  { 0x46706, "ATTACHMENT OF LOGICAL UNIT FAILED" },
  { 0x46707, "CREATION OF LOGICAL UNIT FAILED" },
  { 0x46900, "DATA LOSS ON LOGICAL UNIT" },
  { 0x46901, "MULTIPLE LOGICAL UNIT FAILURES" },
  { 0x46902, "A PARITY/DATA MISMATCH" },
  { 0x46E00, "COMMAND TO LOGICAL UNIT FAILED" },
  { 0x50011, "AUDIO PLAY OPERATION IN PROGRESS" },
  { 0x50012, "AUDIO PLAY OPERATION PAUSED" },
  { 0x50013, "AUDIO PLAY OPERATION SUCCESSFULLY COMPLETED" },
  { 0x50014, "AUDIO PLAY OPERATION STOPPED DUE TO ERROR" },
  { 0x50015, "NO CURRENT AUDIO STATUS TO RETURN" },
  { 0x50700, "MULTIPLE PERIPHERAL DEVICES SELECTED" },
  { 0x51A00, "PARAMETER LIST LENGTH ERROR" },
  { 0x52000, "INVALID COMMAND OPERATION CODE" },
  { 0x52100, "LOGICAL BLOCK ADDRESS OUT OF RANGE" },
  { 0x52101, "INVALID ELEMENT ADDRESS" },
  { 0x52102, "INVALID ADDRESS FOR WRITE" },
  { 0x52200, "ILLEGAL FUNCTION" },
  { 0x52400, "INVALID FIELD IN CDB" },
  { 0x52500, "LOGICAL UNIT NOT SUPPORTED" },
  { 0x52600, "INVALID FIELD IN PARAMETER LIST" },
  { 0x52601, "PARAMETER NOT SUPPORTED" },
  { 0x52602, "PARAMETER VALUE INVALID" },
  { 0x52603, "THRESHOLD PARAMETERS NOT SUPPORTED" },
  { 0x52604, "INVALID RELEASE OF ACTIVE PERSISTENT RESERVATION" },
  { 0x52700, "WRITE PROTECTED" },
  { 0x52701, "HARDWARE WRITE PROTECTED" },
  { 0x52702, "LOGICAL UNIT SOFTWARE WRITE PROTECTED" },
  { 0x52703, "ASSOCIATED WRITE PROTECT" },
  { 0x52704, "PERSISTENT WRITE PROTECT" },
  { 0x52705, "PERMANENT WRITE PROTECT" },
  { 0x52B00, "COPY CANNOT EXECUTE SINCE HOST CANNOT DISCONNECT" },
  { 0x52C00, "COMMAND SEQUENCE ERROR" },
  { 0x52C01, "TOO MANY WINDOWS SPECIFIED" },
  { 0x52C02, "INVALID COMBINATION OF WINDOWS SPECIFIED" },
  { 0x52C03, "CURRENT PROGRAM AREA IS NOT EMPTY" },
  { 0x52C04, "CURRENT PROGRAM AREA IS EMPTY" },
  { 0x52C05, "PERSISTENT PREVENT CONFLICT" },
  { 0x53002, "CANNOT READ MEDIUM - INCOMPATIBLE FORMAT" },
  { 0x53004, "CANNOT WRITE MEDIUM - UNKNOWN FORMAT" },
  { 0x53005, "CANNOT WRITE MEDIUM - INCOMPATIBLE FORMAT" },
  { 0x53006, "CANNOT FORMAT MEDIUM - INCOMPATIBLE MEDIUM" },
  { 0x53008, "CANNOT WRITE - APPLICATION CODE MISMATCH" },
  { 0x53009, "CURRENT SESSION NOT FIXATED FOR APPEND" },
  { 0x53501, "UNSUPPORTED ENCLOSURE FUNCTION" },
  { 0x53504, "ENCLOSURE SERVICES TRANSFER REFUSED" },
  { 0x53800, "RESERVED" },
  { 0x53900, "SAVING PARAMETERS NOT SUPPORTED" },
  { 0x53B0D, "MEDIUM DESTINATION ELEMENT FULL" },
  { 0x53B0E, "MEDIUM SOURCE ELEMENT EMPTY" },
  { 0x53D00, "INVALID BITS IN IDENTIFY MESSAGE" },
  { 0x54300, "MESSAGE ERROR" },
  { 0x55302, "MEDIUM REMOVAL PREVENTED" },
  { 0x55500, "SYSTEM RESOURCE FAILURE" },
  { 0x56300, "END OF USER AREA ENCOUNTERED ON THIS TRACK" },
  { 0x56301, "PACKET DOES NOT FIT IN AVAILABLE SPACE" },
  { 0x56400, "ILLEGAL MODE FOR THIS TRACK" },
  { 0x56401, "INVALID PACKET SIZE" },
  { 0x56F00, "COPY PROTECTION KEY EXCHANGE FAILURE - AUTHENTICATION FAILURE" },
  { 0x56F01, "COPY PROTECTION KEY EXCHANGE FAILURE - KEY NOT PRESENT" },
  { 0x56F02, "COPY PROTECTION KEY EXCHANGE FAILURE - KEY NOT ESTABLISHED" },
  { 0x56F03, "READ OF SCRAMBLED SECTOR WITHOUT AUTHENTICATION" },
  { 0x56F04, "MEDIA REGION CODE IS MISMATCHED TO LOGICAL UNIT REGION" },
  { 0x56F05, "DRIVE REGION MUST BE PERMANENT/REGION RESET COUNT ERROR" },
  { 0x57203, "SESSION FIXATION ERROR - INCOMPLETE TRACK IN SESSION" },
  { 0x57204, "EMPTY OR PARTIALLY WRITTEN RESERVED TRACK" },
  { 0x57205, "NO MORE RZONE RESERVATIONS ARE ALLOWED" },
  { 0x60A00, "ERROR LOG OVERFLOW" },
  { 0x62800, "NOT READY TO READY CHANGE, MEDIUM MAY HAVE CHANGED" },
  { 0x62801, "IMPORT OR EXPORT ELEMENT ACCESSED" },
  { 0x62900, "POWER ON, RESET, OR BUS DEVICE RESET OCCURRED" },
  { 0x62901, "POWER ON OCCURRED" },
  { 0x62902, "SCSI BUS RESET OCCURRED" },
  { 0x62903, "BUS DEVICE RESET FUNCTION OCCURRED" },
  { 0x62904, "DEVICE INTERNAL RESET" },
  { 0x62A00, "PARAMETERS CHANGED" },
  { 0x62A01, "MODE PARAMETERS CHANGED" },
  { 0x62A02, "LOG PARAMETERS CHANGED" },
  { 0x62A03, "RESERVATIONS PREEMPTED" },
  { 0x62E00, "INSUFFICIENT TIME FOR OPERATION" },
  { 0x62F00, "COMMANDS CLEARED BY ANOTHER INITIATOR" },
  { 0x63B0D, "MEDIUM DESTINATION ELEMENT FULL" },
  { 0x63B0E, "MEDIUM SOURCE ELEMENT EMPTY" },
  { 0x63B0F, "END OF MEDIUM REACHED" },
  { 0x63B11, "MEDIUM MAGAZINE NOT ACCESSIBLE" },
  { 0x63B12, "MEDIUM MAGAZINE REMOVED" },
  { 0x63B13, "MEDIUM MAGAZINE INSERTED" },
  { 0x63B14, "MEDIUM MAGAZINE LOCKED" },
  { 0x63B15, "MEDIUM MAGAZINE UNLOCKED" },
  { 0x63F00, "TARGET OPERATING CONDITIONS HAVE CHANGED" },
  { 0x63F01, "MICROCODE HAS BEEN CHANGED" },
  { 0x63F02, "CHANGED OPERATING DEFINITION" },
  { 0x63F03, "INQUIRY DATA HAS CHANGED" },
  { 0x65501, "SYSTEM BUFFER FULL" },
  { 0x65A00, "OPERATOR REQUEST OR STATE CHANGE INPUT" },
  { 0x65A01, "OPERATOR MEDIUM REMOVAL REQUEST" },
  { 0x65A02, "OPERATOR SELECTED WRITE PROTECT" },
  { 0x65A03, "OPERATOR SELECTED WRITE PERMIT" },
  { 0x65B00, "LOG EXCEPTION" },
  { 0x65B01, "THRESHOLD CONDITION MET" },
  { 0x65B02, "LOG COUNTER AT MAXIMUM" },
  { 0x65B03, "LOG LIST CODES EXHAUSTED" },
  { 0x65C00, "RPL STATUS CHANGE" },
  { 0x65C01, "SPINDLES SYNCHRONIZED" },
  { 0x65D00, "FAILURE PREDICTION THRESHOLD EXCEEDED" },
  { 0x65DFF, "FAILURE PREDICTION THRESHOLD EXCEEDED (FALSE)" },
  { 0x65E00, "LOW POWER CONDITION ON" },
  { 0x65E01, "IDLE CONDITION ACTIVATED BY TIMER" },
  { 0x65E02, "STANDBY CONDITION ACTIVATED BY TIMER" },
  { 0x65E03, "IDLE CONDITION ACTIVATED BY COMMAND" },
  { 0x65E04, "STANDBY CONDITION ACTIVATED BY COMMAND" },
  { 0x66A00, "INFORMATIONAL, REFER TO LOG" },
  { 0x66B00, "STATE CHANGE HAS OCCURRED" },
  { 0x66B01, "REDUNDANCY LEVEL GOT BETTER" },
  { 0x66B02, "REDUNDANCY LEVEL GOT WORSE" },
  { 0x72700, "WRITE PROTECTED" },
  { 0x72701, "HARDWARE WRITE PROTECTED" },
  { 0x72702, "LOGICAL UNIT SOFTWARE WRITE PROTECTED" },
  { 0x72703, "ASSOCIATED WRITE PROTECT" },
  { 0x72704, "PERSISTENT WRITE PROTECT" },
  { 0x72705, "PERMANENT WRITE PROTECT" },
  { 0xA1D00, "MISCOMPARE DURING VERIFY OPERATION" },
  { 0xB0006, "I/O PROCESS TERMINATED, PLAY OPERATION ABORTED" },
  { 0xB1111, "READ ERROR - LOSS OF STREAMING" },
  { 0xB4500, "SELECT OR RESELECT FAILURE" },
  { 0xB4800, "INITIATOR DETECTED ERROR MESSAGE RECEIVED" },
  { 0xB4900, "INVALID MESSAGE ERROR" },
  { 0xB4E00, "OVERLAPPED COMMANDS ATTEMPTED" },
  { 0xE1D00, "MISCOMPARE DURING VERIFY OPERATION" },
};

const char* HexByteString(const unsigned char* bytes, int available_length, int actual_length) {
   static char buf[128];
   buf[0] = 0;
   int len = min(available_length, actual_length);
   for (int i=0; i<len; ++i) {
      wsprintf(buf + i*3, "%02X ", bytes[i]);
   }
   if (actual_length <= available_length) {
      buf[len*3-1] = 0;
   } else {
      wsprintf(buf + len*3 - 1, "... (%d bytes)", actual_length);
   }
   return buf;
}


const char* GetItemText(LogScsiUser* that, int row, int col) {
   if (row == LogScsiUser::max_scis - 1) {
      switch (col) {
      case 1: return "[Log buffer full]";
      case 2: return "[Clear it to continue logging]";
      default: return 0;
      }
   }
   static char buf[128];
   SavedCommandInfo sci = that->scis[row];
   switch (col) {
   case 0:  // command
      if (sci.cdb[0] <= 0x5D) {
         return scsi_command_names_00_5D[sci.cdb[0]];
      } else if (sci.cdb[0] >= 0xA0 && sci.cdb[0] <= 0xBF) {
         return scsi_command_names_A0_BF[sci.cdb[0]-0xA0];
      } else {
         return undefined;
      }
   case 1:  // CDB
      return HexByteString(sci.cdb, 12, sci.cdb_size+1);
   case 2:  // Result
      {
         if (sci.result == SCSIRESULT_SUCCESS) {
            return "success";
         }
         unsigned char srb_stat = SCSIRESULT_SRBSTAT(sci.result);
         unsigned char targ_stat = SCSIRESULT_TARGSTAT(sci.result);
         unsigned char sense_key = SCSIRESULT_SENSEKEY(sci.result);
         unsigned char asc = SCSIRESULT_ASC(sci.result);
         unsigned char ascq = SCSIRESULT_ASCQ(sci.result);
         if (srb_stat != SRB_STATUS_SUCCESS && srb_stat != SRB_STATUS_ERROR) {
            if (srb_stat <= 0x24 && srb_status_names[srb_stat] != 0) {
               strcpy(buf, srb_status_names[srb_stat]);
            } else {
               wsprintf(buf, "%02X", srb_stat);
            }
            wsprintf(buf+strlen(buf), " (%02X %X %02X %02X)", targ_stat, sense_key, asc, ascq);
            return buf;
         }
         buf[0] = 0;
         if (srb_stat != SRB_STATUS_ERROR || targ_stat != SCSISTAT_CHECK_CONDITION) {
            if (targ_stat <= 0x28 && target_status_names[targ_stat>>1] != 0) {
               wsprintf(buf, "%s/%s/", srb_status_names[srb_stat], target_status_names[targ_stat>>1]);
            } else {
               wsprintf(buf, "%s/%02X/", srb_status_names[srb_stat], targ_stat);
            }
         }
         // note: depends on format of scsi_result_t
         for (int i=0; i<sizeof(result_code_names)/sizeof(result_code_names[0]); ++i) {
            if (result_code_names[i].result == SCSIRESULT_SENSE(sci.result)) {
               strcat(buf, result_code_names[i].text);
               return buf;
            }
         }
         wsprintf(buf+strlen(buf), "%s (%02X %02X)", sense_key_names[sense_key], asc, ascq);
         return buf;
      }
   case 3:  // Data out
      if ((sci.inout & 2) && sci.buffer_size != 0) {
         return HexByteString(sci.buffer, 12, sci.buffer_size);
      } else {
         return "";
      }
   case 4:  // Data in
      if ((sci.inout & 1) && sci.buffer_size != 0) {
         return HexByteString(sci.buffer, 12, sci.buffer_size);
      } else {
         return "";
      }
   default:
      return "???";
   }
}


void SaveLog(HWND hwndOwner, LogScsiUser* u) {
   static char filename[MAX_PATH];
   OPENFILENAME ofn;
   ofn.lStructSize = 76;
   ofn.hwndOwner = hwndOwner;
   ofn.hInstance = g_hinstance;
   ofn.lpstrFilter = "Comma-delimited text file (*.csv)\0*.csv\0";
   ofn.lpstrCustomFilter = NULL;
   ofn.nMaxCustFilter = 0;
   ofn.nFilterIndex = 2;
   ofn.lpstrFile = filename;
   ofn.nMaxFile = MAX_PATH;
   ofn.lpstrFileTitle = NULL;
   ofn.nMaxFileTitle = 0;
   ofn.lpstrInitialDir = NULL;
   ofn.lpstrTitle = "Save SCSI log";
   ofn.Flags = OFN_OVERWRITEPROMPT | OFN_HIDEREADONLY;
   ofn.lpstrDefExt = "csv";
   ofn.lpfnHook = NULL;
   ofn.lpTemplateName = NULL;
   if (GetSaveFileName(&ofn)) {
      FILE* out = fopen(filename, "wb");
      if (out) {
         for (int row = 0; row < u->num_scis; ++row) {
            for (int col = 0; col < 5; ++col) {
               if (col != 0)
                  putc(',', out);
               const char* text = GetItemText(u, row, col);
               if (text) {
                  const char* comma = strchr(text, ',');
                  if (comma)
                     putc('"', out);
                  fputs(text, out);
                  if (comma)
                     putc('"', out);
               }
            }
            putc('\n', out);
         }
         fclose(out);
      } else {
         MessageBox(hwndOwner, "Could not open the file for writing.", "SCSI logger", MB_OK);
      }
   }
}


class LogCommandsListView : public ListView {
   LogScsiUser* that;
public:
   LogCommandsListView(NestableWindow* parent, LogScsiUser* _that) : ListView(parent), that(_that) {}
   const char* GetItemText(int row, int col) { return ::GetItemText(that, row, col); }
};


class LogCommandsFrameWindow : public FrameWindow {
   HMENU hmenu;
   LogScsiUser* that;
public:
   bool hidden;
   LogCommandsFrameWindow(const char* name, LogScsiUser* _that)
     : FrameWindow(name, hmenu = LoadMenu(g_hinstance, MAKEINTRESOURCE(IDR_LOGCOMMANDS))),
       that(_that)
   {
      hidden = false;
   }
   bool OnClose() {
      Hide();
      hidden = true;
      return true;
   }
   bool OnMenuCommand(int id, bool is_accelerator) {
      static MENUITEMINFO mii = { 48, MIIM_TYPE, MFT_STRING };
      switch (id) {
      case ID_STARTSTOP:
         that->k->log = !that->k->log;
         mii.dwTypeData = (char*)(that->k->log ? "&Stop!" : "&Start!");
         SetMenuItemInfo(hmenu, id, FALSE, &mii);
         DrawMenuBar(hwnd);
         return true;
      case ID_CLEAR:
         that->num_scis = 0;
         that->listview->SetNumItems(0);
         return true;
      case ID_FILE_SAVEAS:
         SaveLog(hwnd, that);
         return true;
      case ID_OPTIONS_IGNORETESTUNITREADY:
         that->k->ignore_testunitready = !that->k->ignore_testunitready;
         CheckMenuItem(hmenu, id, that->k->ignore_testunitready ? MF_CHECKED : MF_UNCHECKED);
         return true;
      default:
         return false;
      }
   }
};


unsigned __stdcall ThreadProc(void* param) {
   LogScsiUser* u = (LogScsiUser*)param;
   LogScsiKernel* k = u->k;
   for (;;) {
      WaitForSingleObject(u->event, INFINITE);
      if (u->quit)
         break;
      unsigned h = k->head, t = k->tail;
      if (t < h) {
         do {
            if (u->num_scis < u->max_scis) {
               u->scis[u->num_scis++] = k->scis[t % k->queue_size];
               u->listview->AddItem(true);
            }
            ++t;
         } while (t < h);
         k->tail = t;
      }
   }
   return 0;
}


void LogScsiUser_ShowHide(void* self, int) {
   LogScsiUser* u = (LogScsiUser*) self;
   u->frame->hidden = !u->frame->hidden;
   if (u->frame->hidden)
      u->frame->Hide();
   else
      u->frame->Show();
}


void LogScsiUser_AddDeviceMenuItems(DvsDeviceUser* self, DvsMenu* menu) {
   LogScsiUser* u = (LogScsiUser*) self;
   menu->vtable->AddItem(menu, "Show SCSI log", !u->frame->hidden, LogScsiUser_ShowHide, u, 0);
}


void LogScsiUser_Delete(DvsDeviceUser* self) {
   LogScsiUser* u = (LogScsiUser*) self;
   g_callbacks->Driver_Unload(u->driver_handle);
   u->quit = true;
   SetEvent(u->event);
   WaitForSingleObject(u->hthread, INFINITE);
   CloseHandle(u->hthread);
   g_callbacks->CloseKernelEventHandle(u->k->event);
   CloseHandle(u->event);
   delete u->frame;
   delete u->listview;
   delete u;
}


DvsDeviceUser_vtable logscsiuser_vtable = {
   LogScsiUser_AddDeviceMenuItems,
   0, // queryunplug
   LogScsiUser_Delete
};


DvsDeviceUser* HookDevice(DvsDeviceKernel** pkernel, DvsDockingBay* bay) {
   LogScsiUser* u = new LogScsiUser;
   u->vtable = &logscsiuser_vtable;
   u->name = "SCSI log";
   u->frame = new LogCommandsFrameWindow(u->name, u);
   u->listview = new LogCommandsListView(u->frame, u);
   u->listview->InsertColumn(0, 100, "Command", 0);
   u->listview->InsertColumn(1, 200, "CDB", 0);
   u->listview->InsertColumn(2, 120, "Result", 0);
   u->listview->InsertColumn(3, 100, "Data out", 0);
   u->listview->InsertColumn(4, 100, "Data in", 0);
   u->frame->SetChild(u->listview);
   u->event = CreateEvent(NULL, FALSE, FALSE, NULL);
   u->quit = false;
   u->num_scis = 0;

   u->driver_handle = g_callbacks->Driver_Load("LogScsi.kll");

   u->k = (LogScsiKernel*)bay->vtable->SharedPool_Alloc(bay, sizeof(LogScsiKernel));
   u->k->ScsiCommand = (dvs_scsi_func*)g_callbacks->Driver_Call(u->driver_handle, "GetDispatchFunc", "");
   u->k->child = *pkernel;
   u->k->event = g_callbacks->OpenKernelEventHandle(u->event);
   u->k->log = true;
   u->k->ignore_testunitready = true;
   u->k->head = 0;
   u->k->tail = 0;

   unsigned thread_id;
   u->hthread = (HANDLE)_beginthreadex(NULL, 0, ThreadProc, u, 0, &thread_id);

   *pkernel = u->k;
   return u;
}


DvsFilterGlobal plugin_info = {
	0,
	"SCSI protocol spy",
	DVDSYNTH_FILTER_ALL_DEVICE_CLASSES,
	HookDevice,
	0
};


extern "C"
DvsFilterGlobal* DvdsynthFilterPluginEntry(DvsDockingBayGlobal* callbacks) {
	g_callbacks = callbacks;
	return &plugin_info;
}

extern "C"
BOOL WINAPI DllMain(HINSTANCE hinst, DWORD dwReason, LPVOID) {
	g_hinstance = hinst;
	return TRUE;
}
